package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceClient;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceConfig;
import com.amazon.mws.finances._2015_05_01.model.*;
import com.amazonservices.mws.client.MwsUtl;
import com.js.common.constant.Constant;
import com.js.common.enums.AmazonEndpointEnum;
import com.js.common.enums.SiteEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.dal.dao.mapper.JsPaySunrateBankMapper;
import com.js.dal.dao.mapper.JsSyncAmazonFinancialEventGroupMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.mapper.KycSunrateStoreMapper;
import com.js.dal.dao.model.JsPaySunrateBank;
import com.js.dal.dao.model.JsSyncAmazonFinancialEventGroup;
import com.js.dal.dao.model.KycStore;
import com.js.api.sync.service.AmazonFinanceEventGroupService;
import com.js.dal.dao.model.KycSunrateStore;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.DBUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import javax.xml.datatype.XMLGregorianCalendar;
import java.time.*;
import java.util.*;
import java.util.concurrent.CompletableFuture;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonFinanceEventGroupServiceImpl implements AmazonFinanceEventGroupService {

    @Autowired
    SqlSessionFactory sqlSessionFactory;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsPaySunrateBankMapper jsPaySunrateBankMapper;
    @Autowired
    KycSunrateStoreMapper kycSunrateStoreMapper;

    @Override
    public CompletableFuture<Void> startSyncFinanceEventGroupWhole(String storeId) {
        // 抓取条件设为当前时间的两年之前
        Date syncDate = Date.from(ZonedDateTime.of(LocalDateTime.now(), ZoneId.of("UTC")).minusYears(2).toInstant());
        log.info("准备全量抓取，抓取最近两年的事件组。使用时间为：{}", syncDate);
        try {
            startSyncFinanceEventGroupBySyncDate(storeId, syncDate);
        } catch (Exception e) {
            log.error("startSyncFinanceEventGroupBySyncDate 异常", e);
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> startSyncFinanceEventGroupIncrement(String storeId) {
        Date previousSyncDate = AmaMWSCommonUtil.getSyncDate(storeId, JsSyncAmazonFinancialEventGroup.class, jsSyncAmazonFinancialEventGroupMapper);
        if (previousSyncDate.getTime() < 0) {
            log.info("亚马逊店铺事件组尚未获取到上次更新时间，改为使用当前时间。店铺 id: {}", storeId);
            previousSyncDate = new Date();
        }
        log.info("准备增量抓取，抓取最近一个月的事件组。使用时间为：{}", previousSyncDate);
        Date syncDate = Date.from(ZonedDateTime.ofInstant(previousSyncDate.toInstant(), ZoneId.of("UTC")).minusMonths(1).toInstant());
        try {
            startSyncFinanceEventGroupBySyncDate(storeId, syncDate);
        } catch (Exception e) {
            log.error("startSyncFinanceEventGroupBySyncDate 异常", e);
        }
        log.info("亚马逊店铺事件组增量抓取完毕 storeId: {}", storeId);
        return CompletableFuture.completedFuture(null);
    }

    private void startSyncFinanceEventGroupBySyncDate(String storeId, Date syncDate) {
        if (ObjectUtil.isEmpty(storeId) || ObjectUtil.isEmpty(syncDate)) {
            log.error("亚马逊付款账单同步失败 storeId, syncDate 不可以为空");
            return;
        }
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (ObjectUtil.isEmpty(kycStore)) {
            log.error("付款账单同步失败，无此店铺 id：{}", storeId);
            return;
        }
        // 查看当前店铺应绑定的虚拟卡
        String bankNo = getSunrateBankNo(storeId);

        MWSFinancesServiceClient client = getClient(kycStore);

        ListFinancialEventGroupsRequest request = new ListFinancialEventGroupsRequest();
        String sellerId = kycStore.getSellId();
        request.setSellerId(sellerId);
        String mwsAuthToken = kycStore.getMwsAuthToken();
        request.setMWSAuthToken(mwsAuthToken);
        Example example = new Example(JsSyncAmazonFinancialEventGroup.class);
        Example.Criteria criteria = example.createCriteria();
        // TODO　目前亚马逊这个　api 有 bug ,时间范围限定的和查出来的不符最终会导致更新的数据和老数据对不上。所以直接把库里数据全部取出来。
        criteria.andEqualTo("storeId", kycStore.getId());
        List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventOldGroups =
                jsSyncAmazonFinancialEventGroupMapper.selectByExample(example);

        // 设置一页显示几条
//        Integer maxResultsPerPage = 30;
//        request.setMaxResultsPerPage(maxResultsPerPage);
        // 设置抓取的时间范围
        XMLGregorianCalendar financialEventGroupStartedAfter = MwsUtl.getDTF().newXMLGregorianCalendar();
        ZonedDateTime utcSearch = ZonedDateTime.ofInstant(syncDate.toInstant(), ZoneId.of("UTC"));
        financialEventGroupStartedAfter.setYear(utcSearch.getYear());
        financialEventGroupStartedAfter.setMonth(utcSearch.getMonthValue());
        financialEventGroupStartedAfter.setDay(utcSearch.getDayOfMonth());
        request.setFinancialEventGroupStartedAfter(financialEventGroupStartedAfter);
        log.info("准备同步付款账单数据，同步使用的时间为：{}", utcSearch);
        // 获取查询结果
        ListFinancialEventGroupsResponse response = client.listFinancialEventGroups(request);
        ListFinancialEventGroupsResult listFinancialEventGroupsResult = response.getListFinancialEventGroupsResult();
        List<FinancialEventGroup> financialEventGroupList = listFinancialEventGroupsResult.getFinancialEventGroupList();

        // 取出对应市场的事件组
        financialEventGroupList = checkFinancialEventGroup(client, kycStore, financialEventGroupList);
        // 保存付款账单
        updateFinanceEventGroup(kycStore, bankNo, financialEventGroupList, jsSyncAmazonFinancialEventOldGroups);
        // 如果没传完隔一分钟后继续
        if (listFinancialEventGroupsResult.isSetNextToken()) {
            getFinanceEventGroupsByNextToken(client, kycStore, bankNo, mwsAuthToken,
                    listFinancialEventGroupsResult.getNextToken(), jsSyncAmazonFinancialEventOldGroups);
        } else {
            log.info("付款账单拉取完毕");
        }

    }

    /**
     * 检查入账单的市场信息。只返回符合条件的事件组。
     */
    private List<FinancialEventGroup> checkFinancialEventGroup(MWSFinancesServiceClient client,
                                                               KycStore kycStore,
                                                               List<FinancialEventGroup> eventGroups) {
        List<FinancialEventGroup> newfilterGroupList = new ArrayList<>();
        if (eventGroups.size() == 0) {
            return eventGroups;
        }
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        for (FinancialEventGroup eventGroup : eventGroups) {
            try {
                log.info("检查事件组市场，开始拉取事件。暂停5秒钟");
                Thread.sleep(5000);
                ListFinancialEventsRequest request = new ListFinancialEventsRequest();
                request.setSellerId(kycStore.getSellId());
                request.setMWSAuthToken(kycStore.getMwsAuthToken());
                request.setFinancialEventGroupId(eventGroup.getFinancialEventGroupId());
                ListFinancialEventsResponse listFinancialEventsResponse = client.listFinancialEvents(request);
                ListFinancialEventsResult listFinancialEventsResult = listFinancialEventsResponse.getListFinancialEventsResult();
                FinancialEvents financialEvents = listFinancialEventsResult.getFinancialEvents();
                List<ShipmentEvent> shipmentEventList = financialEvents.getShipmentEventList();
                if (shipmentEventList.size() > 0) {
                    // 这里只是随意取了一具事件。如果有 nextToken 也不向下计算找了，没有意义
                    String marketplaceName = shipmentEventList.get(0).getMarketplaceName();
                    if (amazonEndpointEnum.getMarketplaceName().equals(marketplaceName)) {
                        newfilterGroupList.add(eventGroup);
                    }
                } else {
                    SiteEnum siteEnum = SiteEnum.valueOf(kycStore.getStoreTheSiteCode());
                    // 有些服务费事件会没有明细，导致无法计算出市场。这里只能使用源币种做近似计算了。
                    if (eventGroup.getOriginalTotal().getCurrencyCode().equals(siteEnum.getCurrency())) {
                        newfilterGroupList.add(eventGroup);
                    }
                }
            } catch (Exception e) {
                log.error("拉取付款事件时出错。事件组 id：{}", eventGroup.getFinancialEventGroupId());
                log.error("拉取付款事件时出错堆栈", e);
            }
        }
        return newfilterGroupList;
    }

    private String getSunrateBankNo(String storeId) {
        Example example = new Example(KycSunrateStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycStoreId", storeId);
        KycSunrateStore kycSunrateStore = kycSunrateStoreMapper.selectOneByExample(example);
        if (kycSunrateStore == null) {
            return "";
        }
        Example example1 = new Example(JsPaySunrateBank.class);
        Example.Criteria criteria1 = example1.createCriteria();
        criteria1.andEqualTo("kycSunrateStoreId", kycSunrateStore.getId());
        JsPaySunrateBank jsPaySunrateBank = jsPaySunrateBankMapper.selectOneByExample(example1);
        if (jsPaySunrateBank == null) {
            return "";
        } else {
            return jsPaySunrateBank.getBankCardNo();
        }
    }

    private MWSFinancesServiceClient getClient(KycStore kycStore) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        MWSFinancesServiceConfig config = new MWSFinancesServiceConfig();
        config.setServiceURL(amazonEndpointEnum.getEndpoint());
        String applicationName = "dmjishiyupay";
        String applicationVersion = "1.0";
        return new MWSFinancesServiceClient(amazonEndpointEnum.getAccessKey(),
                amazonEndpointEnum.getSecrectKey(), applicationName, applicationVersion, config);
    }

    private void updateFinanceEventGroup(KycStore kycStore, String bankNo, List<FinancialEventGroup> financialEventGroupList, List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventOldGroups) {
        if (ObjectUtil.isEmpty(jsSyncAmazonFinancialEventOldGroups)) {
            saveFinanceEventGroup(kycStore, bankNo, financialEventGroupList);
        } else {
            List<JsSyncAmazonFinancialEventGroup> updateList = new ArrayList<>();
            List<FinancialEventGroup> insertList = new ArrayList<>();
            Date syncDate = Date.from(Instant.now());
            for (FinancialEventGroup financialEventGroup : financialEventGroupList) {
                boolean isExist = false;
                for (JsSyncAmazonFinancialEventGroup jsSyncAmazonFinancialEventOldGroup : jsSyncAmazonFinancialEventOldGroups) {
                    if (jsSyncAmazonFinancialEventOldGroup.getFinancialEventGroupId().equals(financialEventGroup.getFinancialEventGroupId())) {
                        isExist = true;
                        jsSyncAmazonFinancialEventOldGroup.setSyncDate(syncDate);
                        setJsSyncFinancialEventGroupProperties(jsSyncAmazonFinancialEventOldGroup, financialEventGroup);
                        updateList.add(jsSyncAmazonFinancialEventOldGroup);
                        break;
                    }
                }
                if (!isExist) {
                    insertList.add(financialEventGroup);
                }
            }
            saveFinanceEventGroup(kycStore, bankNo, insertList);
            checkStoreAuthStatus(kycStore, bankNo, updateList);
            DBUtil.updateBatch(updateList, JsSyncAmazonFinancialEventGroupMapper.class, sqlSessionFactory);
        }
    }

    private void saveFinanceEventGroup(KycStore kycStore, String bankNo, List<FinancialEventGroup> financialEventGroupList) {
        if (ObjectUtil.isEmpty(financialEventGroupList) || ObjectUtil.isEmpty(kycStore)) {
            return;
        }
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        List<JsSyncAmazonFinancialEventGroup> insertList = new ArrayList<>();
        Date syncDate = Date.from(Instant.now());
        for (FinancialEventGroup financialEventGroup : financialEventGroupList) {
            JsSyncAmazonFinancialEventGroup jsSyncAmazonFinancialEventGroup = new JsSyncAmazonFinancialEventGroup();
            AmaMWSCommonUtil.initBaseData(jsSyncAmazonFinancialEventGroup);
            jsSyncAmazonFinancialEventGroup.setStoreId(kycStore.getId());
            jsSyncAmazonFinancialEventGroup.setKycNaturalId(kycStore.getKycNaturalId());
            jsSyncAmazonFinancialEventGroup.setMarketplaceName(amazonEndpointEnum.getMarketplaceName());
            jsSyncAmazonFinancialEventGroup.setSyncDate(syncDate);
            setJsSyncFinancialEventGroupProperties(jsSyncAmazonFinancialEventGroup, financialEventGroup);
            insertList.add(jsSyncAmazonFinancialEventGroup);
        }
        // 计算最近的一笔成功回款的银行卡号是不是我们自己平台的并更新店铺状态。
        checkStoreAuthStatus(kycStore, bankNo, insertList);
        jsSyncAmazonFinancialEventGroupMapper.insertListWithKey(insertList);
    }

    private void checkStoreAuthStatus(KycStore kycStore, String bankNo, List<JsSyncAmazonFinancialEventGroup> eventGroupList) {
        Optional<JsSyncAmazonFinancialEventGroup> group = eventGroupList.stream()
                .filter(e -> "Closed".equals(e.getProcessingStatus()))
                .filter(e -> "Succeeded".equals(e.getFundTransferStatus()))
                .filter(e -> e.getFundTransferDate() != null)
                .sorted(Comparator.comparing(JsSyncAmazonFinancialEventGroup::getFundTransferDate).reversed())
                .limit(1)
                .findFirst();
        if (group.isPresent()) {
            String accountTail = group.get().getAccountTail();
            if (ObjectUtil.isNotNull(bankNo)) {
                if (bankNo.endsWith(accountTail)) {
                    // 证明已经使用了我们自己平台的银行卡，则更新店铺授权状态。
                    if (kycStore.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal()) {
                        // 如果店铺已经是授权且已经回款的状态。则不进行任务操作。
                        // nothing to do
                    } else if (kycStore.getAuthStatus() == StoreAuthStatusEnum.AUTH.ordinal()) {
                        // 如果店铺是授权未回款的状态。则更新店铺授权相关信息。
                        kycStore.setAuthStatus(StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal());
                        kycStore.setAuthWithAccountDate(new Date());
                        kycStoreMapper.updateByPrimaryKeySelective(kycStore);
                    } else if (kycStore.getAuthStatus() == StoreAuthStatusEnum.CANCEL_AUTH.ordinal()) {
                        kycStore.setAuthStatus(StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal());
                        kycStore.setAuthWithAccountDate(new Date());
                        kycStore.setEnable(true);
                        kycStoreMapper.updateByPrimaryKeySelective(kycStore);
                    } else {
                        log.info("店铺回款状态检查1 stroeId: {}, authStatus: {}, bankNo: {}, accountTail: {}",
                                kycStore.getId(), kycStore.getAuthStatus(), bankNo, accountTail);
                    }
                } else {
                    if (kycStore.getAuthStatus() == StoreAuthStatusEnum.AUTH.ordinal()
                            || kycStore.getAuthStatus() == StoreAuthStatusEnum.UN_AUTH.ordinal()) {
                        // nothing to do
                    } else if (kycStore.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal()) {
                        // 证明用户更改了绑定银行卡
                        // TODO 怎么报警
                        kycStore.setAuthStatus(StoreAuthStatusEnum.CANCEL_AUTH.ordinal());
                        kycStore.setEnable(false);
                        kycStoreMapper.updateByPrimaryKeySelective(kycStore);
                    } else {
                        log.info("店铺回款状态检查2 stroeId: {}, authStatus: {}, bankNo: {}, accountTail: {}",
                                kycStore.getId(), kycStore.getAuthStatus(), bankNo, accountTail);
                    }
                }
            }
        }
    }

    private void getFinanceEventGroupsByNextToken(MWSFinancesServiceClient client, KycStore kycStore, String bankNo,
                                                  String mwsAuthToken, String nextToken,
                                                  List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventOldGroups) {
        ListFinancialEventGroupsByNextTokenResult listFinancialEventGroupsByNextTokenResult;
        try {
            log.info("付款列表有 nextToken 暂停十秒钟");
            Thread.sleep(10000);

            ListFinancialEventGroupsByNextTokenRequest request = new ListFinancialEventGroupsByNextTokenRequest();
            request.setSellerId(kycStore.getSellId());
            request.setMWSAuthToken(mwsAuthToken);
            request.setNextToken(nextToken);
            ListFinancialEventGroupsByNextTokenResponse response = client.listFinancialEventGroupsByNextToken(request);
            listFinancialEventGroupsByNextTokenResult = response.getListFinancialEventGroupsByNextTokenResult();
            List<FinancialEventGroup> financialEventGroupList = listFinancialEventGroupsByNextTokenResult.getFinancialEventGroupList();

            financialEventGroupList = checkFinancialEventGroup(client, kycStore, financialEventGroupList);
            // 保存付款账单
            updateFinanceEventGroup(kycStore, bankNo, financialEventGroupList, jsSyncAmazonFinancialEventOldGroups);
        } catch (Exception e) {
            log.error("拉取亚马逊付款数据时出错：{}, 当前正拉取的 nextToken：{}", e.getMessage(), nextToken);
            log.error("拉取亚马逊付款数据时出错堆栈", e);
            return;
        }

        // 如果没有传完，递归。
        if (listFinancialEventGroupsByNextTokenResult.isSetNextToken()) {
            getFinanceEventGroupsByNextToken(client, kycStore, bankNo, mwsAuthToken,
                    listFinancialEventGroupsByNextTokenResult.getNextToken(), jsSyncAmazonFinancialEventOldGroups);
        } else {
            log.info("付款账单拉取完毕");
        }
    }

    /**
     * 设置实体属性
     */
    private void setJsSyncFinancialEventGroupProperties(JsSyncAmazonFinancialEventGroup jsSyncAmazonFinancialEventGroup,
                                                        FinancialEventGroup financialEventGroup) {
        jsSyncAmazonFinancialEventGroup.setUpdateDts(null);
        jsSyncAmazonFinancialEventGroup.setFinancialEventGroupId(financialEventGroup.getFinancialEventGroupId());
        jsSyncAmazonFinancialEventGroup.setProcessingStatus(financialEventGroup.getProcessingStatus());
        jsSyncAmazonFinancialEventGroup.setFundTransferStatus(financialEventGroup.getFundTransferStatus());
        if (ObjectUtil.isNotEmpty(financialEventGroup.getOriginalTotal())) {
            jsSyncAmazonFinancialEventGroup.setOriginalTotalCode(financialEventGroup.getOriginalTotal().getCurrencyCode());
            jsSyncAmazonFinancialEventGroup.setOriginalTotal(financialEventGroup.getOriginalTotal().getCurrencyAmount());
        }
        if (ObjectUtil.isNotEmpty(financialEventGroup.getConvertedTotal())) {
            jsSyncAmazonFinancialEventGroup.setConvertedTotalCode(financialEventGroup.getConvertedTotal().getCurrencyCode());
            jsSyncAmazonFinancialEventGroup.setConvertedTotal(financialEventGroup.getConvertedTotal().getCurrencyAmount());
        }
        jsSyncAmazonFinancialEventGroup.setFundTransferDate(
                AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(financialEventGroup.getFundTransferDate()));
        jsSyncAmazonFinancialEventGroup.setTraceId(financialEventGroup.getTraceId());
        jsSyncAmazonFinancialEventGroup.setAccountTail(financialEventGroup.getAccountTail());
        if (ObjectUtil.isNotEmpty(financialEventGroup.getBeginningBalance())) {
            jsSyncAmazonFinancialEventGroup.setBeginningBalanceCode(financialEventGroup.getBeginningBalance().getCurrencyCode());
            jsSyncAmazonFinancialEventGroup.setBeginningBalance(financialEventGroup.getBeginningBalance().getCurrencyAmount());
        }
        jsSyncAmazonFinancialEventGroup.setFinancialEventGroupStart(
                AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(financialEventGroup.getFinancialEventGroupStart()));
        jsSyncAmazonFinancialEventGroup.setFinancialEventGroupEnd(
                AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(financialEventGroup.getFinancialEventGroupEnd()));
    }

}
