package com.js.sync.mq;


import com.js.common.model.req.common.MailReq;
import com.js.sync.service.common.MailService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 参考文档： https://docs.spring.io/spring-amqp/docs/2.1.10.RELEASE/reference/html/#async-annotation-driven
 */
@Component
@RabbitListener(
        bindings = {@QueueBinding(
                value = @Queue(name = "${spring.rabbitmq.commonQueue}"),
                exchange = @Exchange(name = "${spring.rabbitmq.commonExchange}"),
                key = "${spring.rabbitmq.commonKey}"
        )}
)
@Slf4j
public class RabbitMQJsCommonReceiver {

    @Autowired
    MailService mailService;

    /**
     * 使用同一个路由，但是走不同的方法，具体使用哪一个方法。通过传入的参数类型进行判断
     * 使用的参数对象必须可序列化，且必须为单参数。
     * 方法本身还可以支持一个以 @Payload 标注的参数和一个以 @Header 标注的 map map 里会有一些传递使用的相关信息
     */
    @RabbitHandler
    public boolean sendMail(MailReq mailReq) {
        log.info("使用通用邮件发送服务发送邮件");
        try {
            mailService.sendMail(mailReq);
            log.info("邮件发送成功: {}", mailReq.getSubject());
            return true;
        } catch (Exception e) {
            log.error("邮件发送出现异常", e);
            return false;
        }
    }

    /**
     * 上述方法还可以写成下面这样，其中 headers 是返回全部信息的 map 也可以改为 header 但与此同时必须指定某一个 key
     * 这两种写法不可以同时出现将来增加功能时，可以根据需要，看使用哪种声明的方式。
     */
    /*@RabbitHandler
    public void sendMail(@Payload FadadaSignContractResultReq fadadaSignContractResultReq, @Headers Map map) {
        log.info("sendMail");
    }*/
    @RabbitHandler(isDefault = true)
    public void defaultMethod(Message obj) {
        log.error("没有匹配到正确的调用方法: {}", obj);
    }

}
