package com.js.sync.job;

import com.js.api.sync.service.*;
import com.js.dal.dao.model.KycStore;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.*;
import java.util.stream.Collectors;

@Slf4j
@Component
public class AmazonDataSync {


    @Autowired
    AmazonOrderService amazonOrderService;
//    @Autowired
//    AmazonOrderItemService amazonOrderItemService;
    @Autowired
    AmazonFinanceEventGroupService amazonFinanceEventGroupService;
    @Autowired
    AmazonFinanceEventService amazonFinanceEventService;
    @Autowired
    AmazonInventoryReportService amazonInventoryReportService;
    @Autowired
    AmazonSettlementReportService amazonSettlementReportService;
    @Autowired
    @Qualifier("incrementTaskExecutor")
    Executor incrementTaskExecutor;

    /**
     * 同步该 kyc 名下全部的店铺数据。
     * @param kycStores 需要同步的店铺
     */
    public Boolean syncAmazonData(List<KycStore> kycStores) {
        // 因为不同的店铺可能拥有同一组 token（不同小区的，例如加拿大与美国站用的就是一样的 token），
        // 所以在拉取服务时为了防止亚马逊节流，事先将店铺进行分批处理，保证同一批次下的店铺的 token 是不重复的。
        // 且不同批次之间是串行的。
        List<List<KycStore>> storeBatches = new ArrayList<>();
        List<String> storeIds = kycStores.stream().map(KycStore::getId).collect(Collectors.toList());
        getStoreBatches(kycStores, storeIds, storeBatches);
        for (List<KycStore> storeBatch : storeBatches) {
            List<CompletableFuture> futures = new ArrayList<>();
            try {
                for (KycStore kycStore : storeBatch) {
                    String storeId = kycStore.getId();
                    CompletableFuture<Void> future1 = amazonOrderService.startSyncOrderIncrement(storeId)
                            .handleAsync((s, t) -> {
//                                if (t == null) {
//                                    CompletableFuture<Void> future = amazonOrderItemService.startSyncOrderItemIncrement(storeId);
//                                    try {
//                                        future.get(30, TimeUnit.MINUTES);
//                                    } catch (InterruptedException | ExecutionException | TimeoutException e) {
//                                        log.error("startSyncOrderItemIncrement 异常", e);
//                                    }
//                                }
                                return s;
                            }, incrementTaskExecutor);
                    CompletableFuture<Void> future2 = amazonFinanceEventGroupService.startSyncFinanceEventGroupIncrement(storeId)
                            .handleAsync((s, t) -> {
                                if (t == null) {
                                    // 没有异常抛出，证明已经成功拉取完了事件组，开始同步事件明细。
                                    CompletableFuture<Void> future = amazonFinanceEventService.startSyncFinanceEventIncrement(storeId);
                                    try {
                                        future.get(30, TimeUnit.MINUTES);
                                    } catch (InterruptedException | ExecutionException | TimeoutException e) {
                                        log.error("startSyncFinanceEventIncrement 异常", e);
                                    }
                                }
                                return s;
                            }, incrementTaskExecutor);
                    CompletableFuture<Void> future3 = amazonInventoryReportService.startSyncInventoryReportIncrement(storeId);
                    CompletableFuture<Void> future4 = amazonSettlementReportService.startSyncSettlementReportIncrement(storeId);
                    futures.add(future1);
                    futures.add(future2);
                    futures.add(future3);
                    futures.add(future4);
                }
                // 每一批次完成后再同步下一批次。 TODO 将来需要控制同步的店铺数量，现在线程线队列只有10000个。
                CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()])).get(30, TimeUnit.MINUTES);
            } catch (Exception e) {
                log.error("同步数据组出现错误：", e);
                return false;
            }
        }
        return true;
    }

    private void getStoreBatches(List<KycStore> kycStores, List<String> storeIds, List<List<KycStore>> storeBatches) {
        if (storeIds.size() > 0) {
            List<KycStore> tempList = new ArrayList<>();
            for (KycStore kycStore : kycStores) {
                if (storeIds.contains(kycStore.getId())) {
                    if (tempList.size() == 0) {
                        tempList.add(kycStore);
                        storeIds.remove(kycStore.getId());
                    } else {
                        boolean duplicate = false;
                        for (KycStore store : tempList) {
                            if (store.getSellId().equals(kycStore.getSellId())
                                    && store.getMwsAuthToken().equals(kycStore.getMwsAuthToken())) {
                                duplicate = true;
                            }
                        }
                        if (!duplicate) {
                            tempList.add(kycStore);
                            storeIds.remove(kycStore.getId());
                        }
                    }
                }
            }
            storeBatches.add(tempList);
            getStoreBatches(kycStores, storeIds, storeBatches);
        } else {
            return;
        }
    }
}
