package com.js.sync.config;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import com.js.common.constant.CommonConstant;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.common.util.RedisDistributedLock;
import com.js.dal.dao.mapper.JsLoanApplyInfoMapper;
import com.js.dal.dao.mapper.JsLoanRiskInfoMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.JsLoanApplyInfo;
import com.js.dal.dao.model.JsLoanRiskInfo;
import com.js.dal.dao.model.KycStore;
import com.js.sync.job.*;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.annotation.Scheduled;
import tk.mybatis.mapper.entity.Example;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * 亚马逊拉取服务定时任务
 */

@Slf4j
@Configuration
@EnableScheduling
public class AmazonTaskJobConfig {

    final String LOCK_KEY = "CONFIGURE_DAY_TASKS";
    final String LOCK_KEY2 = "UPDATE_AMAZON_DATA";

    @Autowired
    AmazonDataAnalysisByKyc amazonDataAnalysisByKyc;
    @Autowired
    AmazonDataAnalysisByStore amazonDataAnalysisByStore;
    @Autowired
    AmazonRiskQuota amazonRiskQuota;
    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    AmazonCheckAuthToken amazonCheckAuthToken;
    @Autowired
    JsLoanRiskInfoMapper jsLoanRiskInfoMapper;
    @Autowired
    JsLoanApplyInfoMapper jsLoanApplyInfoMapper;
    @Autowired
    AmazonDataSync amazonDataSync;
    @Autowired
    RedisDistributedLock redisDistributedLock;

    // 每天凌晨1点执行一次
    @Scheduled(cron = "0 0 1 * * ?")
    public void configureDayTasks() {
        String token = IdUtil.simpleUUID();
        if (!redisDistributedLock.tryGetDistributedLock(LOCK_KEY, token, 30 * 60 * 1000)) {
            log.info("configureDayTasks 加锁失败，跳过本次更新");
            return;
        }
        List<String> kycNaturalIds = getLoanKycNaturalIds();
        List<KycStore> kycStores = getKycStores(e -> {
            e.andEqualTo("enable", true);
            e.andIn("authStatus", Arrays.asList(
                StoreAuthStatusEnum.AUTH.ordinal(),
                StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal()));
            if (kycNaturalIds.size() > 0) {
                // 如果是贷款或准入申请的用户会按一小时一次的频率更新，这里就不在按日跑批里再次执行了。
                e.andNotIn("kycNaturalId", kycNaturalIds);
            }
            e.andEqualTo("delFlag", false);
        });
        List<String> storeIds = kycStores.stream()
                .map(KycStore::getId)
                .collect(Collectors.toList());
        Boolean syncResult = amazonDataSync.syncAmazonData(kycStores);
        redisDistributedLock.releaseDistributedLock(LOCK_KEY, token);
        if (!syncResult) {
            log.error("亚马逊数据日跑批同步失败");
            return;
        }
        // 全部更新完了之后计算单店铺年平均指标
        for (String storeId : storeIds) {
            amazonDataAnalysisByStore.calStoreQuotaByStoreId(storeId);
        }
        // 更新总店铺趋势指标
        List<String> kycNaturalIds2 = kycStores.stream()
                .map(KycStore::getKycNaturalId)
                .collect(Collectors.toList());
        Date now = new Date();
        for (String kycNaturalId : kycNaturalIds2) {
            amazonDataAnalysisByKyc.calKycQuotaById(kycNaturalId, now);
            amazonRiskQuota.calQuota(kycNaturalId);
        }
    }

    /**
     * 一小时执行一次
     */
    @Scheduled(cron = "0 0 0/1 * * ?")
    public void updateAmazonData() {
        String token = IdUtil.simpleUUID();
        if (redisDistributedLock.tryGetDistributedLock(LOCK_KEY2, token, 10 * 60 * 1000)) {
            log.info("updateAmazonData 加锁失败，跳过本次更新");
            return;
        }
        List<String> kycNaturalIds = getLoanKycNaturalIds();
        if (kycNaturalIds.size() > 0) {
            // 获取符合要求的同步店铺
            List<KycStore> kycStores = getKycStores(e -> {
                e.andIn("kycNaturalId", kycNaturalIds);
                e.andEqualTo("enable", true);
                e.andIn("authStatus", Arrays.asList(
                        StoreAuthStatusEnum.AUTH.ordinal(),
                        StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal()));
                e.andEqualTo("delFlag", false);
            });
            if (kycStores.size() > 0) {
                Boolean syncResult = amazonDataSync.syncAmazonData(kycStores);
                if (!syncResult) {
                    log.error("亚马逊数据日跑批同步失败");
                    redisDistributedLock.releaseDistributedLock(LOCK_KEY2, token);
                    return;
                }
                List<String> storeIds = kycStores.stream()
                        .map(KycStore::getId)
                        .collect(Collectors.toList());
                // 全部更新完了之后计算单店铺年平均指标
                for (String storeId : storeIds) {
                    amazonDataAnalysisByStore.calStoreQuotaByStoreId(storeId);
                }
                Date now = new Date();
                for (String kycNaturalId : kycNaturalIds) {
                    amazonDataAnalysisByKyc.calKycQuotaById(kycNaturalId, now);
                    amazonRiskQuota.calQuota(kycNaturalId);
                }
            } else {
                log.info("目前无需要快速更新的店铺数据1");
            }
        } else {
            log.info("目前无需要快速更新的店铺数据2");
        }
        redisDistributedLock.releaseDistributedLock(LOCK_KEY2, token);
    }

    /**
     * 获取正在等待准入审核或贷款申请的用户 kycId
     */
    private List<String> getLoanKycNaturalIds() {
        // 获取等待审核的准入信息。
        List<JsLoanRiskInfo> jsLoanRiskInfos = getVerifyJsLoanRiskInfo();
        // 获取等待审批的贷款申请
        List<JsLoanApplyInfo> jsLoanApplyInfos = getVerifyJsLoanApplyInfos();
        // 取出 kycNatualIds
        List<String> ids1 = jsLoanRiskInfos.stream()
                .map(JsLoanRiskInfo::getKycNaturalId)
                .collect(Collectors.toList());
        List<String> ids2 = jsLoanApplyInfos.stream()
                .map(JsLoanApplyInfo::getKycNaturalId)
                .collect(Collectors.toList());
        ids1.addAll(ids2);
        return ids1.stream()
                .filter(ObjectUtil::isNotEmpty)
                .distinct()
                .collect(Collectors.toList());
    }

    private List<JsLoanApplyInfo> getVerifyJsLoanApplyInfos() {
        Example example = new Example(JsLoanApplyInfo.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("delFlag", false);
        criteria.andIn("loanStatus", Arrays.asList(
                CommonConstant.LOAN_APPLY_STATUS_INIT,
                CommonConstant.LOAN_APPLY_STATUS_I_VERIFY_ADOPT));
        return jsLoanApplyInfoMapper.selectByExample(example);
    }

    private List<JsLoanRiskInfo> getVerifyJsLoanRiskInfo() {
        Example example = new Example(JsLoanRiskInfo.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("status", CommonConstant.LOAN_RISK_STATUS_VERIFY);
        criteria.andEqualTo("delFlag", false);
        return jsLoanRiskInfoMapper.selectByExample(example);
    }

    // 每5分钟执行一次
    @Scheduled(cron = "0 0/5 * * * ?")
    public void checkToken() {
        try {
            amazonCheckAuthToken.checkToken();
        } catch (Exception e) {
            log.error("店铺授权监控状态异常", e);
        }
    }

    private List<KycStore> getKycStores(Consumer<Example.Criteria> condition) {
        Example example = new Example(KycStore.class);
        Example.Criteria criteria = example.createCriteria();
        condition.accept(criteria);
        return kycStoreMapper.selectByExample(example);
    }
    // 每日跑批走的都是增量拉取的接口。全量接口的在添加店铺并授权成功后调用。目前店铺不存在修改token的情况。
    // 查询全部的用户
    // 查询到全部授权的店铺
    // 1 拉取订单服务 一天一次
    // 2 拉取付款事件组 一天一次
    // 3 拉取库存信息 一天一次
    // 4 拉取结算报告 一天一次
    // 订单服务拉取完毕后-------------------------
    // 1-1拉取订单明细 一天一次
    // 事件组拉取完毕后----------------
    // 2-1拉取事件明细 一天一次
    // 基础数据全部拉取完毕后------------
    // 5 计算单店铺年平均指标 一天一次 这里计算只是留存一份原始数据。实际风控查看时还会重新拉取计算当天最新的指标
    // 6 用户维度的数据统计 一个月一次

}
