package com.js.sync.job;

import cn.hutool.core.date.DateUtil;
import com.js.common.JsException.LogicException;
import com.js.common.enums.AmazonEventTypeEnum;
import com.js.common.enums.ResultEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.AveragingBigDecimal;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 风控后台店铺列表数据计算。
 *
 * 算法以风控给出的 excel 为准
 *
 * 设计上计算指标前需要先去亚马逊拉取一次数据，保证计算时数据是最新的。
 * 指标的计算均不依赖指标表里的数据。只要源数据存在即可。
 * 这样在异步调用时就不需控制计算顺序了。
 * 列表显示时，每个店铺只取目前最新的那一条。
 *
 * @author liutianyu
 */
@Slf4j
@Component
public class AmazonDataAnalysisByStore {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    PlatformStatisticsByStoreMapper platformStatisticsByStoreMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;
    @Autowired
    JsSyncAmazonShipmentEventChargeMapper jsSyncAmazonShipmentEventChargeMapper;

    public PlatformStatisticsByStore calStoreQuotaByStoreId(String storeId) {

        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        String kycNaturalId = kycStore.getKycNaturalId();
        List<KycStore> kycStores = getKycStoresByKycNaturalId(kycNaturalId);
        List<KycStore> authWithAccountStore = kycStores.stream()
                .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal())
                .collect(Collectors.toList());

        ZonedDateTime now = ZonedDateTime.now();
        ZonedDateTime start = now.minusYears(2);

        // 获取该用户下近两年的全部的店铺订单
        // 正常情况下可以拉到用户一年半的订单数据。
        // 为了防止将来订单数据量越来越大，只取最近两年的订单。
        List<JsSyncAmazonOrder> jsSyncAmazonOrders = getJsSyncAmazonOrders(storeId, start, now);
        if (jsSyncAmazonOrders.size() < 1) {
            log.info("该店铺下无订单数据。指标计算停止");
            return null;
        }
        // 获取该店铺近两年的事件组
        List<JsSyncAmazonFinancialEventGroup> eventGroups = getJsSyncAmazonFinancialEventGroups(storeId, start, now);

        // 获取对应的事件
        List<JsSyncAmazonShipmentEvent> events = getJsSyncAmazonShipmentEvents(storeId, eventGroups);

        // 获取事件明细消费
        List<JsSyncAmazonShipmentEventCharge> eventCharges = getJsSyncAmazonShipmentEventsCharge(storeId, events);

        // 获取商口库存信息
        List<JsSyncAmazonInventory> inventories = getJsSyncAmazonInventories(storeId, start, now);

        // 总月均销售额（万）
        BigDecimal monthlySale = calMonthlySale(jsSyncAmazonOrders, events, now);
        // 笔均入账金额（万)
        BigDecimal averageIncome;
        if (authWithAccountStore.size() <= 0) {
            averageIncome = BigDecimal.ZERO;
        } else {
            averageIncome = calAverageIncome(eventGroups);
        }
        // 平均客单价
        BigDecimal perCustomerTransaction = calPerCustomerTransaction(jsSyncAmazonOrders, events, now);
        // 月均退款额（万）
        BigDecimal refundAmount = calRefundAmount(jsSyncAmazonOrders, events, now);
        // 总税率
        BigDecimal taxRate = calTaxRate(jsSyncAmazonOrders, eventCharges, now);
        // 平均订单数
        BigDecimal averageOrderNum = calAverageOrderNum(jsSyncAmazonOrders, events, now);
        // 平均SKU数量
        BigDecimal averageSku = calAverageSku(inventories, now);

        PlatformStatisticsByStore platformStatisticsByStore = new PlatformStatisticsByStore();
        platformStatisticsByStore.setKycNaturalId(kycStore.getKycNaturalId());
        platformStatisticsByStore.setStoreId(storeId);
        platformStatisticsByStore.setMonthlySale(monthlySale);
        platformStatisticsByStore.setAverageIncome(averageIncome);
        platformStatisticsByStore.setPerCustomerTransaction(perCustomerTransaction);
        platformStatisticsByStore.setRefundAmount(refundAmount);
        platformStatisticsByStore.setTaxRate(taxRate);
        platformStatisticsByStore.setAverageOrderNum(averageOrderNum);
        platformStatisticsByStore.setAverageSku(averageSku);

        ZonedDateTime statisticsTime = now.withHour(1).withMinute(0).withSecond(0).withNano(0);

        Example example = new Example(PlatformStatisticsByStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andEqualTo("statisticsTime", Date.from(statisticsTime.toInstant()));
        criteria.andEqualTo("delFlag", false);
        List<PlatformStatisticsByStore> platformStatisticsByStores = platformStatisticsByStoreMapper.selectByExample(example);
        if (platformStatisticsByStores.size() == 0) {
            AmaMWSCommonUtil.initBaseData(platformStatisticsByStore);
            platformStatisticsByStore.setStatisticsTime(Date.from(statisticsTime.toInstant()));
            platformStatisticsByStoreMapper.insertSelective(platformStatisticsByStore);
            log.info("单店铺年平均指标插入成功 storeId: {}, stataisticsTime: {}", storeId, statisticsTime);
        } else if (platformStatisticsByStores.size() == 1) {
            PlatformStatisticsByStore statistics = platformStatisticsByStores.get(0);
            platformStatisticsByStore.setId(statistics.getId());
            platformStatisticsByStore.setCreateId(statistics.getCreateId());
            platformStatisticsByStore.setCreateName(statistics.getCreateName());
            platformStatisticsByStore.setCreateDts(statistics.getCreateDts());
            platformStatisticsByStore.setUpdateId("sys");
            platformStatisticsByStore.setUpdateName("sys");
            platformStatisticsByStore.setUpdateDts(new Date());
            platformStatisticsByStore.setStatisticsTime(Date.from(statisticsTime.toInstant()));
            log.info("单店铺年平均指标更新成功 storeId: {}, stataisticsTime: {}", storeId, statisticsTime);
        } else {
            log.error("单店铺年平均指标数据条目异常，storeId: {}, stattisticsTime: {}", storeId, statisticsTime);
            return null;
        }
        return platformStatisticsByStore;
    }

    private BigDecimal calAverageSku(List<JsSyncAmazonInventory> inventories, ZonedDateTime now) {
        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;
        if (dayOfMonth >= 20) {
            end = now;
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(11);
        } else {
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = end.minusMonths(12);
        }
        // 库存报告一天只存一份，同一天库存报告的条数就等于 sku 数
        Map<String, Long> map = inventories.stream()
                .filter(e -> e.getStatisticsTime().getTime() >= start.toInstant().toEpochMilli())
                .filter(e -> e.getStatisticsTime().getTime() < end.toInstant().toEpochMilli())
                .collect(Collectors.groupingBy(e -> DateUtil.format(e.getStatisticsTime(), "yyyy-MM-dd"),
                        Collectors.counting()));
        long count = map.entrySet().stream()
                .mapToLong(Map.Entry::getValue)
                .count();
        if (map.keySet().size() != 0) {
            return new BigDecimal(count).divide(new BigDecimal(map.keySet().size()), 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private List<JsSyncAmazonInventory> getJsSyncAmazonInventories(String storeId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonInventory.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andGreaterThanOrEqualTo("statisticsTime", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("statisticsTime", Date.from(now.toInstant()));
        criteria.andEqualTo("delFlag", false);
        return jsSyncAmazonInventoryMapper.selectByExample(example);
    }

    private BigDecimal calAverageOrderNum(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;
        if (dayOfMonth >= 20) {
            end = now;
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(11);
        } else {
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = end.minusMonths(12);
        }
        long month;
        if (checkOrderDataEnough(jsSyncAmazonOrders, start)) {
            month = 12;
        } else {
            month = ZonedDateTime.ofInstant(jsSyncAmazonOrders.get(0).getPurchaseDate().toInstant(), ZoneId.systemDefault())
                    .until(end, ChronoUnit.MONTHS);
        }
        List<String> orderIds = events.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal())
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        long count = jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getPurchaseDate().getTime() < end.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .count();
        if (month > 0) {
            return new BigDecimal(count).divide(new BigDecimal(month), 4, RoundingMode.HALF_UP);
        } else {
            return new BigDecimal(count);
        }
    }

    private BigDecimal calTaxRate(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEventCharge> eventCharges, ZonedDateTime now) {

        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;
        if (dayOfMonth >= 20) {
            end = now;
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(11);
        } else {
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = end.minusMonths(12);
        }

        List<String> orderIds = eventCharges.stream()
                .map(JsSyncAmazonShipmentEventCharge::getOrderId)
                .distinct()
                .collect(Collectors.toList());

        // 计算
        List<JsSyncAmazonOrder> orderList = jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getPurchaseDate().getTime() < end.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .collect(Collectors.toList());
        // 将订单按月份进行分组
        Map<String, List<JsSyncAmazonOrder>> group = orderList.stream()
                .collect(Collectors.groupingBy(e -> DateUtil.format(e.getPurchaseDate(), "yyyy-MM"),
                        Collectors.toList()));
        List<BigDecimal> taxRates = new ArrayList<>();
        for (Map.Entry<String, List<JsSyncAmazonOrder>> entry : group.entrySet()) {
            // 计算本月的销售额
            BigDecimal sum = entry.getValue().stream()
                    .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            List<String> orderIds2 = entry.getValue().stream()
                    .map(JsSyncAmazonOrder::getAmazonOrderId)
                    .collect(Collectors.toList());
            // 计算本月的税额
            BigDecimal tax = eventCharges.stream()
                    .filter(e -> orderIds2.contains(e.getOrderId()))
                    .filter(e -> "Tax".equals(e.getChargeType()))
                    .map(e -> convertAmount(e, JsSyncAmazonShipmentEventCharge::getChargeAmount, JsSyncAmazonShipmentEventCharge::getChargeAmountCode))
                    .reduce(BigDecimal.ZERO, BigDecimal::add);
            // 计算税率
            if (sum.compareTo(BigDecimal.ZERO) != 0) {
                taxRates.add(tax.divide(sum, 4, RoundingMode.HALF_UP));
            } else {
                taxRates.add(BigDecimal.ZERO);
            }

        }

        if (taxRates.size() > 0) {
            return taxRates.stream()
                    .reduce(BigDecimal.ZERO, BigDecimal::add)
                    .divide(new BigDecimal(taxRates.size()), 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private List<JsSyncAmazonShipmentEventCharge> getJsSyncAmazonShipmentEventsCharge(String storeId, List<JsSyncAmazonShipmentEvent> events) {
        // 获取事件对应的订单 id
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        if (orderIds.size() > 0) {
            Example example = new Example(JsSyncAmazonShipmentEventCharge.class);
            Example.Criteria criteria = example.createCriteria();
            criteria.andIn("orderId", orderIds);
            criteria.andEqualTo("storeId", storeId);
            criteria.andEqualTo("delFlag", false);
            return jsSyncAmazonShipmentEventChargeMapper.selectByExample(example);
        } else {
            return Collections.emptyList();
        }

    }

    private BigDecimal calRefundAmount(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {

        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;
        if (dayOfMonth >= 20) {
            end = now;
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(11);
        } else {
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = end.minusMonths(12);
        }

        List<String> orderIds = events.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.REFOUN.ordinal())
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        long month;
        if (checkOrderDataEnough(jsSyncAmazonOrders, start)) {
            month = 12;
        } else {
            month = ZonedDateTime.ofInstant(jsSyncAmazonOrders.get(0).getPurchaseDate().toInstant(), ZoneId.systemDefault())
                    .until(end, ChronoUnit.MONTHS);
        }

        BigDecimal sum = jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getPurchaseDate().getTime() < end.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        if (month > 0) {
            return sum.divide(new BigDecimal(month), 4, RoundingMode.HALF_UP);
        } else {
            return sum;
        }
    }

    private BigDecimal calPerCustomerTransaction(List<JsSyncAmazonOrder> jsSyncAmazonOrders,
                                                 List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;
        if (dayOfMonth >= 20) {
            end = now;
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(11);
        } else {
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = end.minusMonths(12);
        }
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        Map<String, BigDecimal> map = jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getPurchaseDate().getTime() < end.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .collect(Collectors.groupingBy(e -> DateUtil.format(e.getPurchaseDate(), "yyyy-MM"),
                        Collectors.mapping(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode),
                                new AveragingBigDecimal())));

        // 计算平均客单价
        return map.entrySet().stream()
                .map(Map.Entry::getValue)
                .collect(new AveragingBigDecimal());
    }

    private BigDecimal calAverageIncome(List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        // 获取最近一笔回账的日期，因为事件组排过序。所以第一个就是最近的。
        if (eventGroups.size() == 0) {
            return BigDecimal.ZERO;
        }
        JsSyncAmazonFinancialEventGroup eventGroup = eventGroups.get(0);
        ZonedDateTime fundTransferDate = ZonedDateTime.ofInstant(eventGroup.getFundTransferDate().toInstant(), ZoneId.of("UTC"));
        // 向前推 1 年
        ZonedDateTime start = fundTransferDate.minusYears(1);
        // 计算在该时间范围的入账事件组
        List<JsSyncAmazonFinancialEventGroup> groups = eventGroups.stream()
                .filter(e -> e.getFundTransferDate() != null)
                .filter(e -> e.getFundTransferDate().getTime() <= fundTransferDate.toInstant().toEpochMilli())
                .filter(e -> e.getFundTransferDate().getTime() >= start.toInstant().toEpochMilli())
                .collect(Collectors.toList());

        // 计算总入账金额
        BigDecimal sum = groups.stream()
                .map(e -> convertAmount(e, JsSyncAmazonFinancialEventGroup::getOriginalTotal,
                        JsSyncAmazonFinancialEventGroup::getOriginalTotalCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 计算笔均入账金额
        return sum.divide(new BigDecimal(groups.size()), 4, RoundingMode.HALF_UP);
    }

    private List<KycStore> getKycStoresByKycNaturalId(String kycNaturalId) {
        Example example = new Example(KycStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andEqualTo("delFlag", false);
        return kycStoreMapper.selectByExample(example);
    }

    private List<JsSyncAmazonOrder> getJsSyncAmazonOrders(String storeId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonOrder.class);
        example.setOrderByClause("purchase_date asc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andGreaterThanOrEqualTo("purchaseDate", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("purchaseDate", Date.from(now.toInstant()));
        criteria.andEqualTo("delFlag", false);
        return jsSyncAmazonOrderMapper.selectByExample(example);
    }

    private List<JsSyncAmazonFinancialEventGroup> getJsSyncAmazonFinancialEventGroups(String storeId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonFinancialEventGroup.class);
        example.setOrderByClause("fund_transfer_date desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andEqualTo("processingStatus", "Closed");
        criteria.andEqualTo("fundTransferStatus", "Succeeded");
        criteria.andGreaterThanOrEqualTo("fundTransferDate", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("fundTransferDate", Date.from(now.toInstant()));
        criteria.andEqualTo("delFlag", false);
        Example.Criteria criteria1 = example.or();
        criteria1.andEqualTo("storeId", storeId);
        criteria1.andEqualTo("processingStatus", "Open");
        criteria1.andEqualTo("delFlag", false);
        return jsSyncAmazonFinancialEventGroupMapper.selectByExample(example);
    }

    private List<JsSyncAmazonShipmentEvent> getJsSyncAmazonShipmentEvents(String storeId, List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        List<String> groupIds = eventGroups.stream()
                .map(JsSyncAmazonFinancialEventGroup::getFinancialEventGroupId)
                .collect(Collectors.toList());
        Example example = new Example(JsSyncAmazonShipmentEvent.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andIn("financialEventGroupId", groupIds);
        criteria.andEqualTo("delFlag", false);
        return jsSyncAmazonShipmentEventMapper.selectByExample(example);
    }

    private BigDecimal calMonthlySale(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        // 计算过滤时间。以发起时间为准，向前推一年
        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;

        if (dayOfMonth >= 20) {
            // 计算本月销售额 + 前11个月的销售额
            end = now;
            start = now.withDayOfMonth(1).minusMonths(11);
        } else {
            // 计算前十二个月的销售额
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = end.minusMonths(12);
        }

        Boolean enough = checkOrderDataEnough(jsSyncAmazonOrders, start);
        // 计算时间段内全部订单的价格
        BigDecimal sum = calOrderSale(jsSyncAmazonOrders, events, end, start);
        if (enough) {
            return sum.divide(new BigDecimal("12"), 4, RoundingMode.HALF_UP);
        } else {
            ZonedDateTime purchaseDate = ZonedDateTime.ofInstant(jsSyncAmazonOrders.get(0).getPurchaseDate().toInstant(), ZoneId.of("UTC"));
            long days = purchaseDate.until(end, ChronoUnit.DAYS);
            BigDecimal months = new BigDecimal(days).divide(new BigDecimal("30"), 4, RoundingMode.HALF_UP);
            return sum.divide(months, 4, RoundingMode.HALF_UP);
        }
    }

    /**
     * 计算方法，如果最早的订单时间大于计算过滤时间的开始时间。则认为数据量不足
     *
     * @return true 足够， false 不足
     */
    private Boolean checkOrderDataEnough(List<JsSyncAmazonOrder> jsSyncAmazonOrders, ZonedDateTime start) {
        // 因为排过序，所以第一个就是最时早的订单
        JsSyncAmazonOrder jsSyncAmazonOrder = jsSyncAmazonOrders.get(0);
        return jsSyncAmazonOrder.getPurchaseDate().getTime() <= start.toInstant().toEpochMilli();
    }

    private BigDecimal calOrderSale(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now, ZonedDateTime start) {
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        return jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() < now.toInstant().toEpochMilli())
                .filter(e -> e.getOrderTotal() != null)
                .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 将对象中的金额都转换为人民币
     *
     * 目前只支持 美元、日元、英镑
     *
     * @param obj 需要转换的对象
     * @param amount 获取该对象金额的方法引用
     * @param currencyCode 获取该对象币种的方法引用
     * @return
     */
    private <T> BigDecimal convertAmount(T obj, Function<? super T, ? extends BigDecimal> amount,
                                         Function<? super T, ? extends String> currencyCode) {
        if (amount.apply(obj) == null) {
            return BigDecimal.ZERO;
        } else {
            if ("USD".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("6.9387"));
            } else if ("GBP".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("8.4326"));
            } else if ("JPY".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("0.06510"));
            } else {
                log.error("目前不支持该币种：{}", currencyCode.apply(obj));
                throw LogicException.le(ResultEnum.ERROR);
            }
        }
    }
}
