package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.base.IBaseMapper;
import com.js.api.sync.service.*;
import com.js.common.constant.Constant;
import com.js.common.enums.ResultEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.ResultUtil;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.sync.job.AmazonDataAnalysisByKyc;
import com.js.sync.job.AmazonDataAnalysisByStore;
import com.js.sync.job.AmazonRiskQuota;
import com.js.sync.utils.JdbcUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.scheduling.annotation.Async;
import tk.mybatis.mapper.entity.Example;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.ZonedDateTime;
import java.util.Date;
import java.util.List;
import java.util.concurrent.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonDataSyncServiceImpl implements AmazonDataSyncService {

    /**
     * 等待同步数据的缓冲区，主要是为了防止相同 token 拉取触发亚马逊限流
     */
    private CopyOnWriteArrayList<KycStore> processingSyncList = new CopyOnWriteArrayList<>();
    private CopyOnWriteArrayList<KycStore> prepareToSyncList = new CopyOnWriteArrayList<>();

    @Autowired
    AmazonFinanceEventGroupService amazonFinanceEventGroupService;
    @Autowired
    AmazonFinanceEventService amazonFinanceEventService;
    @Autowired
    AmazonInventoryReportService amazonInventoryReportService;
//    @Autowired
//    AmazonOrderItemService amazonOrderItemService;
    @Autowired
    AmazonOrderService amazonOrderService;
    @Autowired
    AmazonSettlementReportService amazonSettlementReportService;
    @Autowired
    AmazonDataAnalysisByKyc amazonDataAnalysisByKyc;
    @Autowired
    AmazonDataAnalysisByStore amazonDataAnalysisByStore;
    @Autowired
    KycNaturalMapper kycNaturalMapper;
    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    @Qualifier("taskExecutor")
    Executor taskExecutor;
    @Autowired
    AmazonRiskQuota amazonRiskQuota;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonOrderAddressMapper jsSyncAmazonOrderAddressMapper;
    @Autowired
    JsSyncAmazonSettlementReportMapper jsSyncAmazonSettlementReportMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    JsSyncAmazonShipmentEventChargeMapper jsSyncAmazonShipmentEventChargeMapper;
    @Autowired
    JsSyncAmazonShipmentEventItemMapper jsSyncAmazonShipmentEventItemMapper;
    @Autowired
    SqlSessionFactory sqlSessionFactory;

    @Override
    public void startSyncStoreData(String storeId) {
        if (ObjectUtil.isEmpty(storeId)) {
            log.error("店铺 id 不允许为空");
            return;
        }
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (!checkStore(kycStore)) {
            log.error("店铺不符合数据同步条件");
            return;
        }
        // 循环所有在执行的店铺，如果没有 token 冲突的，则直接进入线程池开始执行并进入执行列表
        // 如果冲突了，进入待更新的列表。
        synchronized (this) {
            if (processingSyncList.isEmpty()) {
                processingSyncList.add(kycStore);
                startSync(kycStore);
            } else {
                Boolean duplicate = false;
                Boolean skip = false;
                for (KycStore store : processingSyncList) {
                    // 如果是同一个店铺，直接放弃新的同步请求。
                    if (store.getId().equals( kycStore.getId())) {
                        skip = true;
                    } else {
                        if (store.getSellId().equals(kycStore.getSellId())
                                && store.getMwsAuthToken().equals(kycStore.getMwsAuthToken())) {
                            // 发现有与正在执行相同 token 的店铺则让其进入待更新列表
                            prepareToSyncList.add(kycStore);
                            duplicate = true;
                        }
                    }
                }
                if (!duplicate && !skip) {
                    processingSyncList.add(kycStore);
                    startSync(kycStore);
                }
            }
        }
    }

    /**
     * 执行异步同步任务
     *
     * 在任务执行完毕后，会检查执行列表与待更新列表，如果存在符合条件的店铺。会再次调用该方法同步新的店铺
     * @param kycStore
     * @return
     */
    @Async("taskExecutor")
    CompletableFuture<KycStore> startSync(KycStore kycStore) {
        String storeId = kycStore.getId();
        CompletableFuture<Void> future1 = amazonOrderService.startSyncOrderWhole(storeId);
               /* .handleAsync((s, t) -> {
                    if (t == null) {
                        // 因为明细全量同步太慢了。这里先放弃同步而且订单还原也不需要历史订单。
                        amazonOrderItemService.startSyncOrderItemIncrement(storeId);
                    }
                    return s;
                },taskExecutor);*/
        CompletableFuture<Void> future2 = amazonFinanceEventGroupService.startSyncFinanceEventGroupWhole(storeId)
                .handleAsync((s, t) -> {
                    if (t == null) {
                        // 没有异常抛出，证明已经成功拉取完了事件组，开始同步事件明细。
                        CompletableFuture<Void> future = amazonFinanceEventService.startSyncFinanceEventWhole(storeId);
                        try {
                            // 这里需要阻塞线程不让同步件事件方法完毕。
                            future.get(1, TimeUnit.HOURS);
                        } catch (InterruptedException | ExecutionException | TimeoutException e) {
                            e.printStackTrace();
                        }
                    }
                    return s;
                }, taskExecutor);
        CompletableFuture<Void> future3 = amazonInventoryReportService.startSyncInventoryReportWhole(storeId);
        CompletableFuture<Void> future4 = amazonSettlementReportService.startSyncSettlementReportWhole(storeId);
        try {
            CompletableFuture.allOf(future1, future2, future3, future4).get(1, TimeUnit.HOURS);
            amazonDataAnalysisByStore.calStoreQuotaByStoreId(storeId);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            e.printStackTrace();
        }
        synchronized (this) {
            processingSyncList.remove(kycStore);
            if (!prepareToSyncList.isEmpty()) {
                if (processingSyncList.isEmpty()) {
                    KycStore remove = prepareToSyncList.remove(0);
                    processingSyncList.add(remove);
                    startSync(kycStore);
                } else {
                    for (KycStore store : prepareToSyncList) {
                        Boolean duplicate = false;
                        for (KycStore kycStore1 : processingSyncList) {
                            if (kycStore1.getSellId().equals(store.getSellId())
                                    && kycStore1.getMwsAuthToken().equals(store.getMwsAuthToken())) {
                                duplicate = true;
                            }
                        }
                        if (!duplicate) {
                            processingSyncList.add(store);
                            prepareToSyncList.remove(store);
                            startSync(store);
                        }
                    }
                }
            }
        }
        log.info("当前正在全量更新数：{}, 当前准备全量更新数：{}", processingSyncList.size(), prepareToSyncList.size());
        return CompletableFuture.completedFuture(kycStore);
    }

    /**
     * 校验店铺是否符合同步条件
     *
     * 启用状态
     * 未删除
     * 授权状态是"已经授权"或"授权且接到第一笔回款"
     * @param kycStore
     * @return 符合则为 true,否则为 false
     */
    private boolean checkStore(KycStore kycStore) {
        return kycStore.getEnable()
                && !kycStore.getDelFlag()
                && (StoreAuthStatusEnum.AUTH.ordinal() == kycStore.getAuthStatus()
                || StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal() == kycStore.getAuthStatus());
    }

    @Override
    public void reCalKycStatistics(String kycNaturalId) {
        // 固定只重算近一年半的信息
        ZonedDateTime utc = ZonedDateTime.now().withHour(0).withMinute(0).withSecond(0).withNano(0);
        for (int i = 0; i < 548; i++) {
            amazonDataAnalysisByKyc.calKycQuotaById(kycNaturalId, Date.from(utc.toInstant()));
            utc = utc.minusDays(1);
        }
    }

    @Override
    public ResponseMessage setAmazonDataDelFlag(String kycStoreId, boolean delFlag) {
        // 测试时使用店铺所有相关的亚马逊数据加总为 1.5W+ 执行时间为 20s 左右。其中更新数据库使用的时间大约为 5s
        // 如果将来还需要优化，第一优先考虑在 store_id 上增加索引来优化查询所需要使用的时间。
        // 如果仍需要优化更新语句可以考虑使用其它语法。见 com.js.sync.utils.JdbcUtil 说明
        if (ObjectUtil.isEmpty(kycStoreId)) {
            return ResultUtil.error(ResultEnum.ERROR);
        }
        List<JsSyncAmazonFinancialEventGroup> eventGroups = selectList(
                JsSyncAmazonFinancialEventGroup.class,
                jsSyncAmazonFinancialEventGroupMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql1 = "update js_sync_amazon_financial_event_group set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql1, getIds(eventGroups), delFlag);

        List<JsSyncAmazonInventory> inventoryList = selectList(
                JsSyncAmazonInventory.class,
                jsSyncAmazonInventoryMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql2 = "update js_sync_amazon_inventory set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql2, getIds(inventoryList), delFlag);

        List<JsSyncAmazonOrder> orders = selectList(
                JsSyncAmazonOrder.class,
                jsSyncAmazonOrderMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql3 = "update js_sync_amazon_order set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql3, getIds(orders), delFlag);

        List<String> addressIds = orders.stream().map(JsSyncAmazonOrder::getShippingAddressId).collect(Collectors.toList());
        String sql4 = "update js_sync_amazon_order_address set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql4, addressIds, delFlag);

        List<JsSyncAmazonSettlementReport> settlementReports = selectList(
                JsSyncAmazonSettlementReport.class,
                jsSyncAmazonSettlementReportMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql5 = "update js_sync_amazon_settlement_report set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql5, getIds(settlementReports), delFlag);

        List<JsSyncAmazonShipmentEvent> events = selectList(
                JsSyncAmazonShipmentEvent.class,
                jsSyncAmazonShipmentEventMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql6 = "update js_sync_amazon_shipment_event set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql6, getIds(events), delFlag);

        List<JsSyncAmazonShipmentEventCharge> eventCharges = selectList(
                JsSyncAmazonShipmentEventCharge.class,
                jsSyncAmazonShipmentEventChargeMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql7 = "update js_sync_amazon_shipment_event_charge set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql7, getIds(eventCharges), delFlag);

        List<JsSyncAmazonShipmentEventItem> eventItems = selectList(
                JsSyncAmazonShipmentEventItem.class,
                jsSyncAmazonShipmentEventItemMapper, e -> {
                    e.andEqualTo("storeId", kycStoreId);
                });
        String sql8 = "update js_sync_amazon_shipment_event_item set del_flag = ? where id = ?";
        updateAmazonDataDelFlag(sql8, getIds(eventItems), delFlag);
        return ResultUtil.success(ResultEnum.UPDATE_SUCCESS);
    }

    private void updateAmazonDataDelFlag(String sql, List<String> ids, boolean delFlag) {
        // 实测 7000 条更新大约花费 1.5 秒。 mysql jdbc url 上需要设置 rewriteBatchedStatements=true
        //long start = System.currentTimeMillis();
        JdbcUtil.execute(connection -> {
            try {
                PreparedStatement ps = connection.prepareStatement(sql);
                for (int i = 0; i < ids.size(); i++) {
                    ps.setBoolean(1, delFlag);
                    ps.setString(2, ids.get(i));
                    ps.addBatch();
                    if (i != 0 && i % 1000 == 0) {
                        ps.executeBatch();
                    } else if (i == ids.size() - 1) {
                        ps.executeBatch();
                    } else {
                        // nothing to do
                    }
                }
            } catch (SQLException e) {
                e.printStackTrace();
            }
        });
        //long end = System.currentTimeMillis();
        //log.info("更新使用时间： {}", end - start);
    }

    private List<String> getIds(List<? extends IBaseModel> list) {
        return list.stream().map(IBaseModel::getId).collect(Collectors.toList());
    }

    private <T> List<T> selectList(Class<? extends IBaseModel> clazz, IBaseMapper<T> mapper, Consumer<Example.Criteria> consumer) {
        Example example = new Example(clazz);
        Example.Criteria criteria = example.createCriteria();
        consumer.accept(criteria);
        return mapper.selectByExample(example);
    }
}
