package com.js.sync.job;

import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceClient;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceConfig;
import com.amazon.mws.finances._2015_05_01.model.ListFinancialEventGroupsRequest;
import com.amazon.mws.finances._2015_05_01.model.ListFinancialEventGroupsResponse;
import com.amazonservices.mws.client.MwsUtl;
import com.js.common.enums.*;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.KycStore;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.DBUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Example;

import javax.xml.datatype.XMLGregorianCalendar;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@Slf4j
@Component
public class AmazonCheckAuthToken {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    SqlSessionFactory sqlSessionFactory;

    /**
     * 检查全部已经成功授权的店铺的 token 有效性。如果出现问题发送报警
     * TODO sync 的报警功能。 by liuty
     */
    public void checkToken() {
        List<KycStore> kycStores = getKycStores();
        List<KycStore> updateStores = new ArrayList<>();
        for (KycStore kycStore : kycStores) {
            AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
            if (amazonEndpointEnum != null) {
                MWSFinancesServiceClient client = getClient(amazonEndpointEnum);
                ListFinancialEventGroupsRequest request = new ListFinancialEventGroupsRequest();
                request.setSellerId(kycStore.getSellId());
                request.setMWSAuthToken(kycStore.getMwsAuthToken());
                request.setMaxResultsPerPage(1);

                XMLGregorianCalendar financialEventGroupStartedAfter = MwsUtl.getDTF().newXMLGregorianCalendar();
                ZonedDateTime utcSearch = ZonedDateTime.ofInstant(Instant.now(), ZoneId.of("UTC")).minusMonths(1);
                financialEventGroupStartedAfter.setYear(utcSearch.getYear());
                financialEventGroupStartedAfter.setMonth(utcSearch.getMonthValue());
                financialEventGroupStartedAfter.setDay(1);
                request.setFinancialEventGroupStartedAfter(financialEventGroupStartedAfter);
                try {
                    ListFinancialEventGroupsResponse response = client.listFinancialEventGroups(request);
                    log.info("success storeId: {}", kycStore.getId());
                } catch (Exception e) {
                    // TODO 检测到后的逻辑怎么处理？by liuty
                    if (e.getMessage().startsWith("Invalid seller")) {
                        kycStore.setEnable(false);
                        //log.error("error1:{}", e.getMessage());
                    } else if (e.getMessage().startsWith("Missing required parameter")) {
                        kycStore.setEnable(false);
                        //log.error("error2:{}", e.getMessage());
                    } else if (e.getMessage().contains("denied")) {
                        // Access to Finances.ListFinancialEventGroups is denied
                        kycStore.setEnable(false);
                        //log.error("error3:{}", e.getMessage());
                    } else if (e.getMessage().startsWith("AuthToken is not valid for SellerId and AWSAccountId")) {
                        kycStore.setEnable(false);
                        //log.error("error4:{}", e.getMessage());
                    } else {
                        log.error("error5:{}", e.getMessage());
                    }
                    updateStores.add(kycStore);
                }
            } else {
                // nothing to do
            }
        }

        if (updateStores.size() > 0) {
            DBUtil.updateBatch(updateStores, KycStoreMapper.class, sqlSessionFactory);
        }
    }

    private MWSFinancesServiceClient getClient(AmazonEndpointEnum amazonEndpointEnum) {
        MWSFinancesServiceConfig config = new MWSFinancesServiceConfig();
        config.setServiceURL(amazonEndpointEnum.getEndpoint());
        String applicationName = "dmjishiyupay";
        String applicationVersion = "1.0";
        return new MWSFinancesServiceClient(amazonEndpointEnum.getAccessKey(),
                amazonEndpointEnum.getSecrectKey(), applicationName, applicationVersion, config);
    }

    private List<KycStore> getKycStores() {
        Example example = new Example(KycStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("enable", true);
        criteria.andIn("authStatus", Arrays.asList(
                StoreAuthStatusEnum.AUTH.ordinal(),
                StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal()));
        criteria.andEqualTo("delFlag", false);
        criteria.andEqualTo("storePlateformCode", StorePlateformEnum.AMAZON.name());
        return kycStoreMapper.selectByExample(example);
    }
}
