package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazonaws.mws.MarketplaceWebService;
import com.amazonaws.mws.MarketplaceWebServiceClient;
import com.amazonaws.mws.MarketplaceWebServiceConfig;
import com.amazonaws.mws.MarketplaceWebServiceException;
import com.amazonaws.mws.model.*;
import com.js.common.constant.Constant;
import com.js.common.enums.AmazonEndpointEnum;
import com.js.dal.dao.mapper.JsSyncAmazonInventoryMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.JsSyncAmazonInventory;
import com.js.dal.dao.model.KycStore;
import com.js.api.sync.service.AmazonInventoryReportService;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.DBUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.io.*;
import java.nio.charset.Charset;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.concurrent.CompletableFuture;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonInventoryReportServiceImpl implements AmazonInventoryReportService {

    private static final String REPORT_TYPE = "_GET_FLAT_FILE_OPEN_LISTINGS_DATA_";

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;
    @Autowired
    SqlSessionFactory sqlSessionFactory;

    @Override
    public CompletableFuture<Void> startSyncInventoryReportWhole(String storeId) {
        // 获取到当前店铺
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        // 获取调用亚马逊接口的客户端
        MarketplaceWebService service = getMarketplaceWebService(kycStore);
        Date statisticsTime = getConvertDate(Date.from(Instant.now()));
        // 报告只保留三个月，所以只抓最近四个月的。
        for (int i = 0; i < 4; i++) {
            // 获取请求
            GetReportListRequest request = getGetReportListRequest(kycStore, statisticsTime);
            try {
                GetReportListResponse reportListResponse = service.getReportList(request);
                if (reportListResponse.isSetGetReportListResult()) {
                    GetReportListResult getReportListResult = reportListResponse.getGetReportListResult();
                    if (getReportListResult.isSetReportInfoList()) {
                        // 每个月，默认直接取第一个
                        ReportInfo reportInfo = getReportListResult.getReportInfoList().get(0);
                        saveInventoryReport(kycStore, service, reportInfo.getReportId(), statisticsTime);
                        log.info("暂停一分钟");
                        Thread.sleep(60000L);
                    } else {
                        log.info("无库存报告信息，统计时间为：{}", statisticsTime);
                    }
                } else {
                    log.info("无库存报告信息，统计时间为：{}", statisticsTime);
                }
            } catch (MarketplaceWebServiceException | InterruptedException e) {
                e.printStackTrace();
                log.error("获取库存报告出错，店铺 id: {}, 统计时间为：{}", storeId, statisticsTime);

            }
            // 计算前一月的统计时间。
            statisticsTime = Date.from(ZonedDateTime.ofInstant(statisticsTime.toInstant(), ZoneId.of("UTC")).minusMonths(1).toInstant());
        }
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> startSyncInventoryReportIncrement(String storeId) {
        ZonedDateTime utc = ZonedDateTime.now(ZoneId.of("UTC"));
        Date statisticsTime = Date.from(utc.toInstant());
        try {
            startSyncInventoryReport(storeId, statisticsTime);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return CompletableFuture.completedFuture(null);
    }

    private GetReportListRequest getGetReportListRequest(KycStore kycStore, Date statisticsTime) {
        GetReportListRequest request = new GetReportListRequest()
                .withMerchant(kycStore.getSellId())
                .withMWSAuthToken(kycStore.getMwsAuthToken())
                .withReportTypeList(new TypeList(Arrays.asList(REPORT_TYPE)));
        // 设置时间，查询提近一个月的报告
        ZonedDateTime end = ZonedDateTime.ofInstant(statisticsTime.toInstant(), ZoneId.of("UTC"));
        ZonedDateTime start = end.minusMonths(1);

        DatatypeFactory df;
        try {
            df = DatatypeFactory.newInstance();
        } catch (DatatypeConfigurationException e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
        // 转换时间：
        XMLGregorianCalendar startDate = df
                .newXMLGregorianCalendar(new GregorianCalendar(start.getYear(), start.getMonthValue(), 1));
        XMLGregorianCalendar endDate = df
                .newXMLGregorianCalendar(new GregorianCalendar(end.getYear(), end.getMonthValue(), 1));
        request.setAvailableFromDate(startDate);
        request.setAvailableToDate(endDate);
        return request;
    }

    /**
     * 开始同步库存报告
     *
     * 由于亚马逊只可以请求到当前的最新的库存报告，所以参数中的统计时间主要作用是为了在数据库中做一个日期的标记。
     * 即求请的数据一直都是目前最新的库存信息。但存入数据时会被当作统计时间所标识的时间进行标记。
     *
     * @param storeId
     * @param statisticsTime 统计时间
     */
    public void startSyncInventoryReport(String storeId, Date statisticsTime) {
        statisticsTime = getConvertDate(statisticsTime);

        // 获取到当前店铺
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        // 获取调用亚马逊接口的客户端
        MarketplaceWebService service = getMarketplaceWebService(kycStore);
        // 设置请求报告 reportRequest
        RequestReportRequest reportRequest = getRequestReportRequest(kycStore, statisticsTime);
        // 发送库存报告申请请求，获取请求报告 id.
        String reportRequestId = getReportRequestId(service, reportRequest);
        // 获取请求报告列表
        GetReportRequestListRequest reportRequestListRequest = getGetReportRequestListRequest(kycStore, reportRequestId);
        // 获取报告 id
        String generatedReportId = getGenerateReportId(service, reportRequestListRequest, reportRequestId);
        if (generatedReportId == null) {
            log.info("库存报告 generatedReportId 为空, 取消该次同步");
            return;
        }
        // 保存报告
        saveInventoryReport(kycStore, service, generatedReportId, statisticsTime);
    }

    /**
     * 把日期转换成当日的 1点 0分 0秒
     *
     * @param statisticsTime
     * @return
     */
    private Date getConvertDate(Date statisticsTime) {
        ZonedDateTime utc = ZonedDateTime.ofInstant(statisticsTime.toInstant(), ZoneId.of("UTC"));
        statisticsTime = Date.from(utc.withHour(1).withMinute(0).withSecond(0).withNano(0).toInstant());
        return statisticsTime;
    }

    private void saveInventoryReport(KycStore kycStore, MarketplaceWebService service, String generatedReportId, Date statisticsTime) {
        String filename = getReport(kycStore, service, generatedReportId);
        List<JsSyncAmazonInventory> insertList = new ArrayList<>();
        List<JsSyncAmazonInventory> updateList = new ArrayList<>();
        List<JsSyncAmazonInventory> jsSyncAmazonInventories = getJsSyncAmazonInventories(kycStore, statisticsTime);
        try {
            String charsetName = calCharsetName(kycStore);
            File tempFile = new File(filename);
            BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(tempFile), Charset.forName(charsetName)));
            String line;
            int k = 0;
            HashMap<String, Integer> index = new HashMap<>();
            while ((line = br.readLine()) != null) { //循环读取行
                // 实际测试时，因为列的顺序可能会发生改变，所以需要动态的进行列名的判断。
                String[] segments = line.split("\t");
                if (k == 0) {
                    for (int i = 0; i < segments.length; i++) {
                        index.put(convertIndexName(kycStore, segments[i]), i);
                    }
                } else {
                    boolean isExist = false;
                    for (JsSyncAmazonInventory jsSyncAmazonInventory : jsSyncAmazonInventories) {
                        if (segments[index.get("asin")].equals(jsSyncAmazonInventory.getAsin())) {
                            isExist = true;
                            jsSyncAmazonInventory.setSellersku(getSegmProp(index, segments, "sku"));
                            jsSyncAmazonInventory.setQuantity(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "quantity")));
                            jsSyncAmazonInventory.setPrice(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "price")));
                            jsSyncAmazonInventory.setUpdateId("sys");
                            jsSyncAmazonInventory.setUpdateDts(Date.from(Instant.now()));
                            jsSyncAmazonInventory.setUpdateName("sys");
                            updateList.add(jsSyncAmazonInventory);
                        }
                    }
                    if (!isExist) {
                        JsSyncAmazonInventory jsSyncAmazonInventory = new JsSyncAmazonInventory();
                        AmaMWSCommonUtil.initBaseData(jsSyncAmazonInventory);
                        jsSyncAmazonInventory.setStoreId(kycStore.getId());
                        jsSyncAmazonInventory.setUserNaturalId(kycStore.getKycNaturalId());
                        jsSyncAmazonInventory.setAsin(getSegmProp(index, segments, "asin"));
                        jsSyncAmazonInventory.setSellersku(getSegmProp(index, segments, "sku"));
                        jsSyncAmazonInventory.setQuantity(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "quantity")));
                        jsSyncAmazonInventory.setPrice(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "price")));
                        jsSyncAmazonInventory.setStatisticsTime(statisticsTime);
                        insertList.add(jsSyncAmazonInventory);
                    }
                }
                k++;
            }
            br.close();
            tempFile.delete();
        } catch (IOException e) {
            e.printStackTrace();
            log.error("fileName: {} 的文件没有找到！", filename);
            return;
        }

        if (ObjectUtil.isNotEmpty(insertList)) {
            jsSyncAmazonInventoryMapper.insertListWithKey(insertList);
            log.info("库存保存成功 店铺 id: {}, 统计时间: {}", kycStore.getId(), statisticsTime);
        }
        if (ObjectUtil.isNotEmpty(updateList)) {
            DBUtil.updateBatch(updateList, JsSyncAmazonInventoryMapper.class, sqlSessionFactory);
            log.info("库存更新成功 店铺 id: {}, 统计时间: {}", kycStore.getId(), statisticsTime);
        }
    }

    /**
     * 根据不同地区，转换库存报告的表头
     */
    private String convertIndexName(KycStore kycStore, String indexName) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        if (AmazonEndpointEnum.JAPAN.equals(amazonEndpointEnum)) {
            if ("出品者SKU".equals(indexName)) {
                return "sku";
            } else if ("ASIN".equals(indexName)) {
                return "asin";
            } else if ("価格".equals(indexName)) {
                return "price";
            } else if ("数量".equals(indexName)) {
                return "quantity";
            } else {
                return indexName;
            }
        } else {
            return indexName;
        }
    }

    private String calCharsetName(KycStore kycStore) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        if (AmazonEndpointEnum.JAPAN.equals(amazonEndpointEnum)) {
            return "SHIFT-JIS";
        } else {
            return "UTF-8";
        }
    }

    private String getSegmProp(HashMap<String, Integer> index, String[] segments, String key) {
        if (index.get(key) == null || index.get(key) > segments.length - 1) {
            return null;
        } else {
            return segments[index.get(key)];
        }
    }

    /**
     * 获取当天库存报告信息。
     *
     * 注意，因为日期经过了格式化，所以统计出来的数据都是当天的。但是什么时间的是不一定的。
     * @param kycStore
     * @param statisticsTime
     * @return
     */
    private List<JsSyncAmazonInventory> getJsSyncAmazonInventories(KycStore kycStore, Date statisticsTime) {
        Example example = new Example(JsSyncAmazonInventory.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", kycStore.getId());
        criteria.andEqualTo("statisticsTime", statisticsTime);
        return jsSyncAmazonInventoryMapper.selectByExample(example);
    }

    private String getReport(KycStore kycStore, MarketplaceWebService service, String generatedReportId) {
        GetReportRequest request = new GetReportRequest();
        request.setMerchant(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());

        request.setReportId(generatedReportId);
        FileOutputStream report;
        String filename = "InventoryReport" + generatedReportId;
        try {
            report = new FileOutputStream(filename);
            request.setReportOutputStream( report );
            GetReportResponse inventoryReport = service.getReport(request);
            report.close();
            return filename;
        } catch (MarketplaceWebServiceException | IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    private String getGenerateReportId(MarketplaceWebService service, GetReportRequestListRequest reportRequestListRequest, String reportRequestId) {
        try {
            while (true) {
                GetReportRequestListResponse reportRequestList = service.getReportRequestList(reportRequestListRequest);
                if (reportRequestList.isSetGetReportRequestListResult()) {
                    GetReportRequestListResult getReportRequestListResult = reportRequestList.getGetReportRequestListResult();
                    if (getReportRequestListResult.isSetReportRequestInfoList()) {
                        List<ReportRequestInfo> reportRequestInfoList = getReportRequestListResult.getReportRequestInfoList();
                        for (ReportRequestInfo reportRequestInfo : reportRequestInfoList) {
                            if (reportRequestInfo.getReportRequestId().equals(reportRequestId)) {
                                String reportProcessingStatus1 = reportRequestInfo.getReportProcessingStatus();
                                if ("_DONE_".equals(reportProcessingStatus1)
                                        || "_DONE_NO_DATA_".equals(reportProcessingStatus1)
                                        || "_CANCELLED_".equals(reportProcessingStatus1)) {
                                    log.debug("库存报告申请结束，状态：{}", reportProcessingStatus1);
                                    return reportRequestInfo.getGeneratedReportId();
                                }
                            } else {
                                // 按道理这里应该永远走不进来，但借鉴的代码却加上了这个 if 判断
                                log.error("getReportRequestInfoList 出现了不合要求的 reportRequestInfo。搜索条件的 " +
                                        "reportRequestId: {}, 当前搜索到的 reportRequestId: {}", reportRequestId, reportRequestInfo.getReportRequestId());
                            }
                        }
                        log.info("尚未准备完毕，暂停一分钟");
                        Thread.sleep(60000L);
                    }
                    // 因为指定了 id, 所以应该不会有 nextToken 选项了。
                }
            }
        } catch (MarketplaceWebServiceException | InterruptedException e) {
            e.printStackTrace();
            return null;
        }
    }

    private GetReportRequestListRequest getGetReportRequestListRequest(KycStore kycStore, String reportRequestId) {
        GetReportRequestListRequest reportRequestListRequest = new GetReportRequestListRequest();
        reportRequestListRequest.setMerchant(kycStore.getSellId());
        reportRequestListRequest.setMWSAuthToken(kycStore.getMwsAuthToken());
        reportRequestListRequest.setReportRequestIdList(new IdList().withId(reportRequestId));
        return reportRequestListRequest;
    }

    private String getReportRequestId(MarketplaceWebService service, RequestReportRequest request) {
        // ReportRequestInfo 这个类注意一下。
        try {
            RequestReportResponse requestReportResponse = service.requestReport(request);
            if (requestReportResponse.isSetRequestReportResult()) {
                RequestReportResult requestReportResult = requestReportResponse.getRequestReportResult();
                if (requestReportResult.isSetReportRequestInfo()) {
                    return requestReportResult.getReportRequestInfo().getReportRequestId();
                } else {
                    throw new MarketplaceWebServiceException("ReportRequestInfo is null !");
                }
            } else {
                throw new MarketplaceWebServiceException("RequestReportResult is null !");
            }
        } catch (MarketplaceWebServiceException e) {
            e.printStackTrace();
            log.error("库存报告申请失败");
            return null;
        }
    }

    private RequestReportRequest getRequestReportRequest(KycStore kycStore, Date statisticsTime) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        // 库存报告，使用完整版的。
        RequestReportRequest request = new RequestReportRequest()
                .withMerchant(kycStore.getSellId())
                .withMarketplaceIdList(new IdList(Arrays.asList(amazonEndpointEnum.getMarketPlaceId())))
                .withReportType(REPORT_TYPE)
                .withMWSAuthToken(kycStore.getMwsAuthToken());
        return request;
    }

    private MarketplaceWebService getMarketplaceWebService(KycStore kycStore) {
        AmazonEndpointEnum amazonEndpointEnum = AmaMWSCommonUtil.calEndpointEnum(kycStore);
        MarketplaceWebServiceConfig config = new MarketplaceWebServiceConfig();
        config.setServiceURL(amazonEndpointEnum.getEndpoint());

        return new MarketplaceWebServiceClient(
                amazonEndpointEnum.getAccessKey(), amazonEndpointEnum.getSecrectKey(), "dmjishiyupay", "1.0", config);
    }


}
