package com.js.sync.job;

import com.js.common.JsException.LogicException;
import com.js.common.enums.AmazonEndpointEnum;
import com.js.common.enums.AmazonEventTypeEnum;
import com.js.common.enums.ResultEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.sync.utils.AmaMWSCommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 亚马逊风控指标
 *
 * 参考指标及算法以风控给出的 excel 为准
 *
 * 设计上计算指标前需要先去亚马逊拉取一次数据，保证计算时数据是最新的。
 * 指标的计算均不依赖指标表里的数据。只要源数据存在即可。
 * 这样在异步调用时就不需控制计算顺序了。
 * 传递给风控进行模型计算时，只传递最新计算的结果。
 *
 * @author liutianyu
 */
@Slf4j
@Component
public class AmazonRiskQuota {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    JsSyncAmazonShipmentEventChargeMapper jsSyncAmazonShipmentEventChargeMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;
    @Autowired
    PlatformAmazonRiskQuotaMapper platformAmazonRiskQuotaMapper;

    /**
     * 指标计算，计算前需要拉取亚马逊的数据。
     * 根据情况不同，计算出的数据可能入库可能不入库。
     *
     * @param kycNaturalId 用户的 kyc id
     */
    public PlatformAmazonRiskQuota calQuota(String kycNaturalId) {
        // 获取该用户下全部店铺
        List<KycStore> kycStores = getKycStoresByKycNaturalId(kycNaturalId);

        // 获取授权成功的店铺
        List<KycStore> authStores = kycStores.stream()
                .filter(e -> (StoreAuthStatusEnum.AUTH.ordinal() == e.getAuthStatus())
                        || StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal() == e.getAuthStatus())
                .collect(Collectors.toList());

        ZonedDateTime now = ZonedDateTime.now();
        ZonedDateTime start = now.minusYears(2);

        // 获取该用户下近两年的全部的店铺订单
        // 正常情况下可以拉到用户一年半的订单数据。
        // 为了防止将来订单数据量越来越大，只取最近两年的订单。
        List<JsSyncAmazonOrder> jsSyncAmazonOrders = getJsSyncAmazonOrders(kycNaturalId, start, now);
        if (jsSyncAmazonOrders.size() < 1) {
            log.info("该用户下无订单数据。风控指标计算停止");
            return null;
        }
        // 获取该用户近两年的事件组
        List<JsSyncAmazonFinancialEventGroup> eventGroups = getJsSyncAmazonFinancialEventGroups(kycNaturalId, start, now);

        // 获取对应的事件
        List<JsSyncAmazonShipmentEvent> events = getJsSyncAmazonShipmentEvents(kycNaturalId, eventGroups);

        // 获取事件明细消费
        List<JsSyncAmazonShipmentEventCharge> eventCharges = getJsSyncAmazonShipmentEventsCharge(kycNaturalId, events);

        // 获取商口库存信息
        List<JsSyncAmazonInventory> inventories = getJsSyncAmazonInventories(kycNaturalId, start, now);

        // 授权成功的店铺数量
        int authAndAuthWithAccountNum = authStores.size();

        // 亚马逊美国店铺近半年销售额占比（授权成功）
        BigDecimal amazonUSSaleRatio = calAmazonUSSaleRatio(authStores, jsSyncAmazonOrders, events, now);

        // 授权成功店铺月均销售额（美元）
        BigDecimal monthlySale = calMonthlySale(jsSyncAmazonOrders, events, now);

        // 退款率
//        BigDecimal refundRate = calRefundRate(eventGroups, events, now);

        // 退款率环比波动率
        BigDecimal refundChainRate = calRefundChainRate(eventGroups, events, now);

        // 在售商品数量
//        BigDecimal availableProductNum = calAvailableProductNum(inventories, now);

        // 在售商品数量环比波动率
        BigDecimal availableProductNumChainRate = calAvailableProductNumChainRate(inventories, now);

        // 绑定回款账户与授权店铺的比例
        BigDecimal authWithAccountAndAuthRatio = calAuthWithAccountAndAuthRatio(authStores);

        LinkedHashMap<String, BigDecimal> itemTotal = calOrderItemSaleRank(jsSyncAmazonOrders, eventCharges, inventories, now);

        // 销售额前10名商品占比
        BigDecimal salesTopTenRatio;

        // 销售额前3名商品占比
        BigDecimal salesTopThreeRatio;

        if (itemTotal == null) {
            salesTopTenRatio = BigDecimal.ZERO;
            salesTopThreeRatio = BigDecimal.ZERO;
        } else {
            salesTopTenRatio = calSalesTopTenRatio(itemTotal);
            salesTopThreeRatio = calSalesTopThreeRatio(itemTotal);
        }

        // 月均入账金额（美元）
        BigDecimal monthlyIncome;
        if (authWithAccountAndAuthRatio.compareTo(new BigDecimal(0)) == 0) {
            monthlyIncome = BigDecimal.ZERO;
        } else {
            monthlyIncome = calMonthlyIncome(eventGroups);
        }

        // 绑定回款账户时长,单位为月
        BigDecimal bindTime = calBindTime(authStores, now);

        // 近一个月入账金额环比增长率
        BigDecimal incomeChainRate;
        if (authWithAccountAndAuthRatio.compareTo(new BigDecimal(0)) == 0) {
            incomeChainRate = BigDecimal.ZERO;
        } else {
            incomeChainRate = calIncomeChainRate(eventGroups, now);
        }

        // 近一个月入账金额与前3个月平均入账金额比例
        BigDecimal incomeThreeRatio;
        if (authWithAccountAndAuthRatio.compareTo(new BigDecimal(0)) == 0) {
            incomeThreeRatio = BigDecimal.ZERO;
        } else {
            incomeThreeRatio = calIncomeThreeRatio(eventGroups, now);
        }

        // 近一个月入账金额与前6个月平均入账金额比例
        BigDecimal incomeSixRatio;
        if (authWithAccountAndAuthRatio.compareTo(new BigDecimal(0)) == 0) {
            incomeSixRatio = BigDecimal.ZERO;
        } else {
            incomeSixRatio = calIncomeSixRatio(eventGroups, now);
        }

        // 入账金额同比增长率
        BigDecimal incomeGrowthRateOnYear;
        if (authWithAccountAndAuthRatio.compareTo(new BigDecimal(0)) == 0) {
            incomeGrowthRateOnYear = BigDecimal.ZERO;
        } else {
            incomeGrowthRateOnYear = calIncomeGrowthRateOnYear(eventGroups, now);
        }

        // TODO 高风险行业。 by liuty

        // 近一月内笔均入账金额
        BigDecimal averageIncome;
        if (authWithAccountAndAuthRatio.compareTo(new BigDecimal(0)) == 0) {
            averageIncome = BigDecimal.ZERO;
        } else {
            averageIncome = calAverageIncome(eventGroups);
        }

        PlatformAmazonRiskQuota platformAmazonRiskQuota = new PlatformAmazonRiskQuota();
        platformAmazonRiskQuota.setKycNaturalId(kycNaturalId);
        platformAmazonRiskQuota.setAuthAndAuthWithAccountNum(authAndAuthWithAccountNum);
        platformAmazonRiskQuota.setAmazonUsSaleRatio(amazonUSSaleRatio);
        platformAmazonRiskQuota.setMonthlySale(monthlySale);
        platformAmazonRiskQuota.setRefundChainRate(refundChainRate);
        platformAmazonRiskQuota.setAvailableProductNumChainRate(availableProductNumChainRate);
        platformAmazonRiskQuota.setAuthWithAccountAndAuthRatio(authWithAccountAndAuthRatio);
        platformAmazonRiskQuota.setSalesTopTenRatio(salesTopTenRatio);
        platformAmazonRiskQuota.setSalesTopThreeRatio(salesTopThreeRatio);
        platformAmazonRiskQuota.setMonthlyIncome(monthlyIncome);
        platformAmazonRiskQuota.setBindTime(bindTime.intValue());
        platformAmazonRiskQuota.setIncomeChainRate(incomeChainRate);
        platformAmazonRiskQuota.setIncomeThreeRatio(incomeThreeRatio);
        platformAmazonRiskQuota.setIncomeSixRatio(incomeSixRatio);
        platformAmazonRiskQuota.setIncomeGrowthRateOnYear(incomeGrowthRateOnYear);
        platformAmazonRiskQuota.setAverageIncome(averageIncome);
        platformAmazonRiskQuota.setDelFlag(false);
        // 如果需要入库，将时间统一。
        ZonedDateTime statisticsTime = now.withHour(1).withMinute(0).withSecond(0).withNano(0);

        List<PlatformAmazonRiskQuota> platformAmazonRiskQuotas = getPlatformAmazonRiskQuotas(kycNaturalId, statisticsTime);
        if (platformAmazonRiskQuotas.size() == 0) {
            // 新数据插入数据库
            AmaMWSCommonUtil.initBaseData(platformAmazonRiskQuota);
            platformAmazonRiskQuota.setStatisticsTime(Date.from(statisticsTime.toInstant()));
            platformAmazonRiskQuotaMapper.insertSelective(platformAmazonRiskQuota);
            log.info("风控指标插入成功. kycNaturalId: {}, statisticsTime: {}", kycNaturalId, statisticsTime);
        } else if (platformAmazonRiskQuotas.size() == 1) {
            // 更新数据库
            PlatformAmazonRiskQuota quota = platformAmazonRiskQuotas.get(0);
            platformAmazonRiskQuota.setId(quota.getId());
            platformAmazonRiskQuota.setCreateId(quota.getCreateId());
            platformAmazonRiskQuota.setCreateName(quota.getCreateName());
            platformAmazonRiskQuota.setCreateDts(quota.getCreateDts());
            platformAmazonRiskQuota.setUpdateId("sys");
            platformAmazonRiskQuota.setUpdateName("sys");
            platformAmazonRiskQuota.setUpdateDts(new Date());
            platformAmazonRiskQuota.setStatisticsTime(quota.getStatisticsTime());
            platformAmazonRiskQuotaMapper.updateByPrimaryKey(platformAmazonRiskQuota);
            log.info("风控指标更新成功 kycNaturalId: {}, statisticsTime: {}", kycNaturalId, statisticsTime);
        } else {
            // 数据异常
            log.error("风控指标数据条目异常，kycNaturalId: {}, stattisticsTime: {}", kycNaturalId, statisticsTime);
            return null;
        }
        return platformAmazonRiskQuota;
    }

    private List<PlatformAmazonRiskQuota> getPlatformAmazonRiskQuotas(String kycNaturalId, ZonedDateTime statisticsTime) {
        Example example = new Example(PlatformAmazonRiskQuota.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andEqualTo("statisticsTime", Date.from(statisticsTime.toInstant()));
        return platformAmazonRiskQuotaMapper.selectByExample(example);
    }

    private BigDecimal calAvailableProductNumChainRate(List<JsSyncAmazonInventory> inventories, ZonedDateTime now) {
        // 本月在售商品数量
        BigDecimal productNum1 = calAvailableProductNum(inventories, now);
        // 计算出上个月末的最后一天的 23：59：59
        ZonedDateTime end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusSeconds(1);
        // 上月在售商品数量
        BigDecimal productNum2 = calAvailableProductNum(inventories, end);

        // 计算在售商品数量环比波动率
        if (productNum1.compareTo(new BigDecimal("0")) == 0) {
            // 正常使用时应该进不到这里
            return BigDecimal.ZERO;
        } else {
            // (productNum1 - productNum2) / productNum1
            return productNum1.subtract(productNum2).divide(productNum1, 4, RoundingMode.HALF_UP);
        }
    }

    private BigDecimal calAverageIncome(List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        // 获取最近一笔回账的日期，因为事件组排过序。所以第一个就是最近的。
        if (eventGroups.size() == 0) {
            return BigDecimal.ZERO;
        }
        JsSyncAmazonFinancialEventGroup eventGroup = eventGroups.get(0);
        ZonedDateTime fundTransferDate = ZonedDateTime.ofInstant(eventGroup.getFundTransferDate().toInstant(), ZoneId.of("UTC"));
        // 向前推 1 个月
        ZonedDateTime start = fundTransferDate.minusMonths(1);
        // 计算在该时间范围的入账事件组
        List<JsSyncAmazonFinancialEventGroup> groups = eventGroups.stream()
                .filter(e -> e.getFundTransferDate() != null)
                .filter(e -> e.getFundTransferDate().getTime() <= fundTransferDate.toInstant().toEpochMilli())
                .filter(e -> e.getFundTransferDate().getTime() >= start.toInstant().toEpochMilli())
                .collect(Collectors.toList());

        // 计算总入账金额
        BigDecimal sum = groups.stream()
                .map(e -> convertAmount(e, JsSyncAmazonFinancialEventGroup::getOriginalTotal,
                        JsSyncAmazonFinancialEventGroup::getOriginalTotalCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 计算笔均入账金额
        return sum.divide(new BigDecimal(groups.size()), 4, RoundingMode.HALF_UP);
    }

    private BigDecimal calIncomeGrowthRateOnYear(List<JsSyncAmazonFinancialEventGroup> eventGroups, ZonedDateTime now) {
        ZonedDateTime end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        ZonedDateTime start = end.minusMonths(1);
        ZonedDateTime end2 = end.minusMonths(12);
        ZonedDateTime start2 = end.minusMonths(13);

        if (checkEventGroupDataEnough(eventGroups, start2)) {
            // 上个月的入账总金额
            BigDecimal sum1 = getEventGroupSumByStartAndEnd(eventGroups, start, end);
            // 去年该月的入账总金额
            BigDecimal sum2 = getEventGroupSumByStartAndEnd(eventGroups, start2, end2);
            // sum1 / sum2
            return sum1.divide(sum2, 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private BigDecimal calIncomeSixRatio(List<JsSyncAmazonFinancialEventGroup> eventGroups, ZonedDateTime now) {
        ZonedDateTime end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        ZonedDateTime middle = end.minusMonths(1);
        ZonedDateTime start = end.minusMonths(7);

        if (checkEventGroupDataEnough(eventGroups, start)) {
            // 上个月的入账总金额
            BigDecimal sum1 = getEventGroupSumByStartAndEnd(eventGroups, middle, end);
            // 前两个月到前七个月的入账总金额
            BigDecimal sum2 = getEventGroupSumByStartAndEnd(eventGroups, start, middle);
            // sum1 / (sum2 / 6)
            return sum1.divide(sum2.divide(new BigDecimal(6), 4, RoundingMode.HALF_UP), 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private BigDecimal calIncomeThreeRatio(List<JsSyncAmazonFinancialEventGroup> eventGroups, ZonedDateTime now) {
        ZonedDateTime end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        ZonedDateTime middle = end.minusMonths(1);
        ZonedDateTime start = end.minusMonths(4);

        if (checkEventGroupDataEnough(eventGroups, start)) {
            // 上个月的入账总金额
            BigDecimal sum1 = getEventGroupSumByStartAndEnd(eventGroups, middle, end);
            // 前两个月到前四个月的入账总金额
            BigDecimal sum2 = getEventGroupSumByStartAndEnd(eventGroups, start, middle);
            // sum1 / (sum2 / 3)
            return sum1.divide(sum2.divide(new BigDecimal(3), 4, RoundingMode.HALF_UP), 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private BigDecimal calIncomeChainRate(List<JsSyncAmazonFinancialEventGroup> eventGroups, ZonedDateTime now) {
        ZonedDateTime end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        ZonedDateTime middle = end.minusMonths(1);
        ZonedDateTime start = end.minusMonths(2);

        if (checkEventGroupDataEnough(eventGroups, start)) {
            // 上个月的入账总金额
            BigDecimal sum1 = getEventGroupSumByStartAndEnd(eventGroups, middle, end);
            // 大上一个月的入账总金额
            BigDecimal sum2 = getEventGroupSumByStartAndEnd(eventGroups, start, middle);
            // (sum1 - sum2) / sum2
            return sum1.subtract(sum2).divide(sum2, 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    /**
     * 计算指定时间区间的事件组总额。
     *
     * 非美元计算时会自动转换成美元
     *
     * @param eventGroups
     * @param end
     * @param start
     * @return
     */
    private BigDecimal getEventGroupSumByStartAndEnd(List<JsSyncAmazonFinancialEventGroup> eventGroups,
                                                     ZonedDateTime start, ZonedDateTime end) {
        return eventGroups.stream()
                .filter(e -> e.getFundTransferDate() != null)
                .filter(e -> e.getFundTransferDate().getTime() < end.toInstant().toEpochMilli())
                .filter(e -> e.getFundTransferDate().getTime() >= start.toInstant().toEpochMilli())
                .map(e -> convertAmount(e, JsSyncAmazonFinancialEventGroup::getOriginalTotal,
                        JsSyncAmazonFinancialEventGroup::getOriginalTotalCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private BigDecimal calBindTime(List<KycStore> authStores, ZonedDateTime now) {
        // 以最早签署时间为准
        Optional<KycStore> kycStore = authStores.stream()
                .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal())
                .filter(e -> e.getAuthWithAccountDate() != null)
                .filter(e -> e.getAuthWithAccountDate().getTime() > 0)
                .sorted(Comparator.comparing(KycStore::getAuthWithAccountDate))
                .findFirst();
        if (kycStore.isPresent()) {
            ZonedDateTime authWithAccountDate = ZonedDateTime.ofInstant(kycStore.get().getAuthWithAccountDate().toInstant(),
                    ZoneId.systemDefault());
            // 计算绑定时间。单位为月
            long months = authWithAccountDate.until(now, ChronoUnit.MONTHS);
            return new BigDecimal(months);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private BigDecimal calMonthlyIncome(List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        // 获取最近一笔回账的日期，因为事件组排过序。所以第一个就是最近的。
        if (eventGroups.size() == 0) {
            return BigDecimal.ZERO;
        }
        JsSyncAmazonFinancialEventGroup eventGroup = eventGroups.get(0);
        ZonedDateTime fundTransferDate = ZonedDateTime.ofInstant(eventGroup.getFundTransferDate().toInstant(), ZoneId.of("UTC"));
        // 向前推 12 个月
        ZonedDateTime start = fundTransferDate.minusMonths(12);
        // 计算在该时间范围内总入账金额
        BigDecimal sum = eventGroups.stream()
                .filter(e -> e.getFundTransferDate() != null)
                .filter(e -> e.getFundTransferDate().getTime() <= fundTransferDate.toInstant().toEpochMilli())
                .filter(e -> e.getFundTransferDate().getTime() >= start.toInstant().toEpochMilli())
                .map(e -> convertAmount(e, JsSyncAmazonFinancialEventGroup::getOriginalTotal, JsSyncAmazonFinancialEventGroup::getOriginalTotalCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        if (checkEventGroupDataEnough(eventGroups, start)) {
            return sum.divide(new BigDecimal(12), 4, RoundingMode.HALF_UP);
        } else {
            List<JsSyncAmazonFinancialEventGroup> groups = eventGroups.stream()
                    .filter(e -> e.getFundTransferDate() != null)
                    .sorted(Comparator.comparing(JsSyncAmazonFinancialEventGroup::getFundTransferDate))
                    .collect(Collectors.toList());
            ZonedDateTime newStart = ZonedDateTime.ofInstant(groups.get(0).getFundTransferDate().toInstant(), ZoneId.of("UTC"));
            long days = newStart.until(fundTransferDate, ChronoUnit.DAYS);
            return sum.divide(new BigDecimal(days)
                    .divide(new BigDecimal("30"), 4, RoundingMode.HALF_UP), 4, RoundingMode.HALF_UP);
        }
    }

    private List<JsSyncAmazonShipmentEventCharge> getJsSyncAmazonShipmentEventsCharge(String kycNaturalId, List<JsSyncAmazonShipmentEvent> events) {
        // 获取事件对应的订单 id
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        if (orderIds.size() > 0) {
            Example example = new Example(JsSyncAmazonShipmentEventCharge.class);
            Example.Criteria criteria = example.createCriteria();
            criteria.andIn("orderId", orderIds);
            criteria.andEqualTo("userNaturalId", kycNaturalId);
            return jsSyncAmazonShipmentEventChargeMapper.selectByExample(example);
        } else {
            return Collections.emptyList();
        }
    }

    private BigDecimal calSalesTopTenRatio(LinkedHashMap<String, BigDecimal> itemTotal) {
        // 月总销售额
        BigDecimal sumSaleTotal = itemTotal.entrySet().stream()
                .map(Map.Entry::getValue)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        // 销售额前10名商品占比
        if (itemTotal.entrySet().size() < 10) {
            return new BigDecimal(1);
        } else {
            BigDecimal salesToeTen = BigDecimal.ZERO;
            int i = 0;
            for (Map.Entry<String, BigDecimal> stringBigDecimalEntry : itemTotal.entrySet()) {
                i++;
                salesToeTen = salesToeTen.add(stringBigDecimalEntry.getValue());
                if (i > 9) {
                    break;
                }
            }
            return salesToeTen.divide(sumSaleTotal, 4, RoundingMode.HALF_UP);
        }
    }

    private BigDecimal calSalesTopThreeRatio(LinkedHashMap<String, BigDecimal> itemTotal) {
        // 月总销售额
        BigDecimal sumSaleTotal = itemTotal.entrySet().stream()
                .map(Map.Entry::getValue)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        // 销售额前 3 名商品占比
        if (itemTotal.entrySet().size() < 3) {
            return new BigDecimal(1);
        } else {
            BigDecimal salesToeTen = BigDecimal.ZERO;
            int i = 0;
            for (Map.Entry<String, BigDecimal> stringBigDecimalEntry : itemTotal.entrySet()) {
                i++;
                salesToeTen = salesToeTen.add(stringBigDecimalEntry.getValue());
                if (i > 2) {
                    break;
                }
            }
            return salesToeTen.divide(sumSaleTotal, 4, RoundingMode.HALF_UP);
        }
    }

    /**
     * 计算商品销售额排名
     *
     * @param jsSyncAmazonOrders
     * @param eventCharges
     * @param inventories
     * @param now
     * @return
     */
    private LinkedHashMap<String, BigDecimal> calOrderItemSaleRank(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEventCharge> eventCharges, List<JsSyncAmazonInventory> inventories, ZonedDateTime now) {
        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime end;
        ZonedDateTime start;
        List<String> orderIds;
        if (dayOfMonth >= 20) {
            // 如果当月大于等于 20 号，则使用当月
            end = now;
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        } else {
            // 当月不足 20 天 使用上月的。
            end = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
            start = now.minusMonths(1).withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        }

        if (checkOrderDataEnough(jsSyncAmazonOrders, start)) {
            orderIds = jsSyncAmazonOrders.stream()
                    .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                    .filter(e -> e.getPurchaseDate().getTime() <= end.toInstant().toEpochMilli())
                    .filter(e -> "Shipped".equals(e.getOrderStatus()))
                    .map(JsSyncAmazonOrder::getAmazonOrderId)
                    .collect(Collectors.toList());
        } else {
            return null;
        }

        // 取出对应订单的消费事件,统计各种商品的总销售额
        Map<String, BigDecimal> groupByASIN = eventCharges.stream()
                .filter(e -> orderIds.contains(e.getOrderId()))
                .filter(e -> "Principal".equals(e.getChargeType()))
                .collect(Collectors.groupingBy(e -> getAsinBySku(e.getSellersku(), inventories),
                        Collectors.mapping(e -> convertAmount(e,
                                JsSyncAmazonShipmentEventCharge::getChargeAmount,
                                JsSyncAmazonShipmentEventCharge::getChargeAmountCode),
                                Collectors.reducing(BigDecimal.ZERO, BigDecimal::add))));

        // 以销售额进行降序分组
        LinkedHashMap<String, BigDecimal> itemTotal = groupByASIN.entrySet().stream()
                .sorted((a, b) -> b.getValue().subtract(a.getValue()).intValue())
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue,
                        (u, v) -> {
                            throw new IllegalStateException(String.format("Duplicate key %s", u));
                        },
                        LinkedHashMap::new));
        return itemTotal;
    }

    private BigDecimal calAuthWithAccountAndAuthRatio(List<KycStore> authStores) {
        // 获取收到回款的店铺数量
        long num = authStores.stream()
                .filter(e -> StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal() == e.getAuthStatus())
                .count();
        return new BigDecimal(num).divide(new BigDecimal(authStores.size()), 4, RoundingMode.HALF_UP);
    }

    private List<JsSyncAmazonInventory> getJsSyncAmazonInventories(String kycNaturalId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonInventory.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("userNaturalId", kycNaturalId);
        criteria.andGreaterThanOrEqualTo("statisticsTime", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("statisticsTime", Date.from(now.toInstant()));

        return jsSyncAmazonInventoryMapper.selectByExample(example);
    }

    private BigDecimal calAvailableProductNum(List<JsSyncAmazonInventory> inventories, ZonedDateTime now) {
        // 以当前日期为开始一至到本月一号，所有库存曾经大于0的商品
        Date start = AmaMWSCommonUtil.convertToTheFristDayOfTheMonth(Date.from(now.toInstant()));

        // 取出本月份的库存快照并计算商品种类数据。因为拉取的库存报告类型本身就只显示库存大于0的。所以不需要做库存大于0的判断。
        long num = inventories.stream()
                .filter(e -> e.getStatisticsTime().getTime() > start.getTime())
                .filter(e -> e.getStatisticsTime().getTime() <= now.toInstant().toEpochMilli())
                .map(JsSyncAmazonInventory::getAsin)
                .distinct()
                .count();
        return new BigDecimal(num);
    }

    private List<JsSyncAmazonShipmentEvent> getJsSyncAmazonShipmentEvents(String kycNaturalId, List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        List<String> groupIds = eventGroups.stream()
                .map(JsSyncAmazonFinancialEventGroup::getFinancialEventGroupId)
                .collect(Collectors.toList());
        Example example = new Example(JsSyncAmazonShipmentEvent.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andIn("financialEventGroupId", groupIds);
        return jsSyncAmazonShipmentEventMapper.selectByExample(example);
    }

    private List<JsSyncAmazonFinancialEventGroup> getJsSyncAmazonFinancialEventGroups(String kycNaturalId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonFinancialEventGroup.class);
        example.setOrderByClause("fund_transfer_date desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("userNaturalId", kycNaturalId);
        criteria.andEqualTo("processingStatus", "Closed");
        criteria.andEqualTo("fundTransferStatus", "Succeeded");
        criteria.andGreaterThanOrEqualTo("fundTransferDate", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("fundTransferDate", Date.from(now.toInstant()));
        Example.Criteria criteria1 = example.or();
        criteria1.andEqualTo("userNaturalId", kycNaturalId);
        criteria1.andEqualTo("processingStatus", "Open");
        return jsSyncAmazonFinancialEventGroupMapper.selectByExample(example);
    }

    private BigDecimal calRefundRate(List<JsSyncAmazonFinancialEventGroup> eventGroups, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        // 时间设为本月一号到上月一号
        Date start = AmaMWSCommonUtil.convertToTheFristDayOfTheMonth(Date.from(now.minusMonths(1).toInstant()));
        Date end = AmaMWSCommonUtil.convertToTheFristDayOfTheMonth(Date.from(now.toInstant()));

        // 取出上个月的事件组id
        List<String> groupIds = eventGroups.stream()
                .filter(e -> e.getFundTransferDate() != null)
                .filter(e -> e.getFundTransferDate().getTime() >= start.getTime())
                .filter(e -> e.getFundTransferDate().getTime() < end.getTime())
                .map(JsSyncAmazonFinancialEventGroup::getFinancialEventGroupId)
                .collect(Collectors.toList());
        // 取出上个月的事件
        List<JsSyncAmazonShipmentEvent> lastMonthEvents = events.stream()
                .filter(e -> groupIds.contains(e.getFinancialEventGroupId()))
                .collect(Collectors.toList());
        // 上个月的退款事件数
        long refounds = lastMonthEvents.stream()
                .filter(e -> AmazonEventTypeEnum.REFOUN.ordinal() == e.getEventType())
                .count();
        // 上个月正常的订单数
        long shipments = lastMonthEvents.stream()
                .filter(e -> AmazonEventTypeEnum.SHIPMENT.ordinal() == e.getEventType())
                .count();
        // 计算退款率
        return new BigDecimal(refounds).divide(new BigDecimal(shipments + refounds), 4, RoundingMode.HALF_UP);
    }

    private BigDecimal calRefundChainRate(List<JsSyncAmazonFinancialEventGroup> eventGroups, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        // 计算上一个月的退款率
        BigDecimal refundRate1 = calRefundRate(eventGroups, events, now);
        // 计算大上一个月的退款率
        BigDecimal refundRate2 = calRefundRate(eventGroups, events, now.minusMonths(1));
        // 退款率环比波动率
        // (refundRate1 - refundRate2) / refundRate2
        return refundRate1.subtract(refundRate2).divide(refundRate2, 4, RoundingMode.HALF_UP);
    }

    private BigDecimal calMonthlySale(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        // 计算过滤时间。以发起时间为准，向前推一年
        ZonedDateTime start = now.minusYears(1);

        Boolean enough = checkOrderDataEnough(jsSyncAmazonOrders, start);
        // 计算时间段内全部订单的价格
        BigDecimal sum = calOrderSale(jsSyncAmazonOrders, events, now, start);
        if (enough) {
            return sum.divide(new BigDecimal("12"), 4, RoundingMode.HALF_UP);
        } else {
            ZonedDateTime purchaseDate = ZonedDateTime.ofInstant(jsSyncAmazonOrders.get(0).getPurchaseDate().toInstant(), ZoneId.of("UTC"));
            long days = purchaseDate.until(now, ChronoUnit.DAYS);
            BigDecimal months = new BigDecimal(days).divide(new BigDecimal("30"), 4, RoundingMode.HALF_UP);
            return sum.divide(months, 4, RoundingMode.HALF_UP);
        }
    }

    private BigDecimal calAmazonUSSaleRatio(List<KycStore> authStores, List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now) {
        List<KycStore> storesUS = authStores.stream()
                .filter(e -> AmaMWSCommonUtil.calEndpointEnum(e).equals(AmazonEndpointEnum.US))
                .collect(Collectors.toList());
        List<String> storesUSIds = storesUS.stream().map(KycStore::getId).collect(Collectors.toList());

        // 计算过滤时间。以发起时间为准，向前推半年
        ZonedDateTime start = now.minusMonths(6);

        // 查看订单数据量是否足够半年
        if(!checkOrderDataEnough(jsSyncAmazonOrders, start)) {
            log.info("用户 id：{}, 订单数据量不足。美国亚马逊近半年销售销占比指标固定为 0");
            return BigDecimal.ZERO;
        }

        // 计算美国亚马逊近半年的销售额
        BigDecimal sumUS = jsSyncAmazonOrders.stream()
                .filter(e -> storesUSIds.contains(e.getStoreId()))
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() < now.toInstant().toEpochMilli())
                .filter(e -> e.getOrderTotal() != null)
                .map(JsSyncAmazonOrder::getOrderTotal)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 计算该用户全部店铺的近半年的销售额，非美元币种需要转成美元
        BigDecimal sum = calOrderSale(jsSyncAmazonOrders, events, now, start);
        return sumUS.divide(sum, 4, RoundingMode.HALF_UP);
    }

    private BigDecimal calOrderSale(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events, ZonedDateTime now, ZonedDateTime start) {
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        return jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getPurchaseDate().getTime() >= start.toInstant().toEpochMilli())
                .filter(e -> e.getPurchaseDate().getTime() < now.toInstant().toEpochMilli())
                .filter(e -> e.getOrderTotal() != null)
                .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private List<JsSyncAmazonOrder> getJsSyncAmazonOrders(String kycNaturalId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonOrder.class);
        example.setOrderByClause("purchase_date asc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andGreaterThanOrEqualTo("purchaseDate", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("purchaseDate", Date.from(now.toInstant()));
        return jsSyncAmazonOrderMapper.selectByExample(example);
    }

    /**
     * 计算方法，如果最早的订单时间大于计算过滤时间的开始时间。则认为数据量不足
     *
     * @return true 足够， false 不足
     */
    private Boolean checkOrderDataEnough(List<JsSyncAmazonOrder> jsSyncAmazonOrders, ZonedDateTime start) {
        // 因为排过序，所以第一个就是最时早的订单
        JsSyncAmazonOrder jsSyncAmazonOrder = jsSyncAmazonOrders.get(0);
        return jsSyncAmazonOrder.getPurchaseDate().getTime() <= start.toInstant().toEpochMilli();
    }

    private Boolean checkEventGroupDataEnough(List<JsSyncAmazonFinancialEventGroup> eventGroups, ZonedDateTime start) {
        List<JsSyncAmazonFinancialEventGroup> groups = eventGroups.stream()
                .filter(e -> e.getFundTransferDate() != null)
                .sorted(Comparator.comparing(JsSyncAmazonFinancialEventGroup::getFundTransferDate))
                .collect(Collectors.toList());
        // 因为排过序，所以第一个就是最时早的订单
        JsSyncAmazonFinancialEventGroup group = groups.get(0);
        return group.getFundTransferDate().getTime() <= start.toInstant().toEpochMilli();
    }

    /**
     * 将对象中的金额都转换为美金
     *
     * 目前只支持 美元、日元、英镑
     *
     * @param obj 需要转换的对象
     * @param amount 获取该对象金额的方法引用
     * @param currencyCode 获取该对象币种的方法引用
     * @return
     */
    private <T> BigDecimal convertAmount(T obj, Function<? super T, ? extends BigDecimal> amount,
                                             Function<? super T, ? extends String> currencyCode) {
        if (amount.apply(obj) == null) {
            return BigDecimal.ZERO;
        } else {
            if ("USD".equals(currencyCode.apply(obj))) {
                return amount.apply(obj);
            } else if ("GBP".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("1.238"));
            } else if ("JPY".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("0.009203"));
            } else {
                log.error("目前不支持该币种：{}", currencyCode.apply(obj));
                throw LogicException.le(ResultEnum.ERROR);
            }
        }
    }

    private List<KycStore> getKycStoresByKycNaturalId(String kycNaturalId) {
        Example example = new Example(KycStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        return kycStoreMapper.selectByExample(example);
    }

    /**
     * 通过商品的 sku 取到获取对应的 asin 号
     *
     * 如果取不到则返回 sku 并打印日志进行提示。
     */
    private String getAsinBySku(String sku, List<JsSyncAmazonInventory> jsSyncAmazonInventory) {
        for (JsSyncAmazonInventory syncAmazonInventory : jsSyncAmazonInventory) {
            if (syncAmazonInventory.getSellersku().equals(sku)) {
                return syncAmazonInventory.getAsin();
            }
        }
        log.info("库存记录不全，没有找到 sku: {} 对应的 asin", sku);
        return sku;
    }

}
