package com.js.sync.job;

import cn.hutool.core.date.DateUtil;
import com.js.common.JsException.LogicException;
import com.js.common.enums.AmazonEventTypeEnum;
import com.js.common.enums.ResultEnum;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.AveragingBigDecimal;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.*;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 总店铺指标。
 *
 * 算法以风控给出的 excel 为准
 * 指标的计算均不依赖指标表里的数据。只要源数据存在即可。
 * 这样在异步调用时就不需控制计算顺序了。
 *
 * 显示趋图时，对于完整的月份，直接取该月份的最后一天。对于当月，则取目前的最新日期。
 *
 * @author liutianyu
 */
@Slf4j
@Component
public class AmazonDataAnalysisByKyc {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    JsSyncAmazonShipmentEventChargeMapper jsSyncAmazonShipmentEventChargeMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;
    @Autowired
    PlatformStatisticsByKycMapper platformStatisticsByKycMapper;

    // 只统计指定时间的当月
    public PlatformStatisticsByKyc calKycQuotaById(String kycNaturalId, Date time) {
        ZonedDateTime now;
        if (time == null) {
            now = ZonedDateTime.now();
        } else {
            now = ZonedDateTime.ofInstant(time.toInstant(), ZoneId.systemDefault());
        }

        ZonedDateTime start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0);

        // 获取该用户下当月全部的店铺订单
        List<JsSyncAmazonOrder> jsSyncAmazonOrders = getJsSyncAmazonOrders(kycNaturalId, start, now);
        if (jsSyncAmazonOrders.size() < 1) {
            log.info("该用户下无订单数据。指标计算停止");
            return null;
        }
        // 获取该用户近两年的事件组
        List<JsSyncAmazonFinancialEventGroup> eventGroups = getJsSyncAmazonFinancialEventGroups(kycNaturalId, start, now);

        // 获取对应的事件
        List<JsSyncAmazonShipmentEvent> events = getJsSyncAmazonShipmentEvents(kycNaturalId, eventGroups);

        // 获取事件明细消费
        List<JsSyncAmazonShipmentEventCharge> eventCharges = getJsSyncAmazonShipmentEventsCharge(kycNaturalId, events);

        // 获取商口库存信息
        List<JsSyncAmazonInventory> inventories = getJsSyncAmazonInventories(kycNaturalId, start, now);

        // 入账金额
        BigDecimal monthlyIncome = calMonthlyIncome(eventGroups);

        // 销售额
        BigDecimal monthlySale = calMonthlySale(jsSyncAmazonOrders, events);

        // 税率
        BigDecimal taxRate = calTaxRate(jsSyncAmazonOrders, eventCharges);

        // 订单数量
        BigDecimal monthlyOrderNum = calMonthlyOrderNum(jsSyncAmazonOrders, events);

        // 客单价
        BigDecimal perCustomerTransaction = calPerCustomerTransaction(jsSyncAmazonOrders, events);

        // SKU
        BigDecimal averageSkuNum = calAverageSkuNum(inventories);

        // 退款率
        BigDecimal monthlyRefundRate = calMonthlyRefundRate(events);

        PlatformStatisticsByKyc platformStatisticsByKyc = new PlatformStatisticsByKyc();
        platformStatisticsByKyc.setKycNaturalId(kycNaturalId);
        platformStatisticsByKyc.setMonthlyIncome(monthlyIncome);
        platformStatisticsByKyc.setMonthlySale(monthlySale);
        platformStatisticsByKyc.setTaxRate(taxRate);
        platformStatisticsByKyc.setMonthlyOrderNum(monthlyOrderNum);
        platformStatisticsByKyc.setPerCustomerTransaction(perCustomerTransaction);
        platformStatisticsByKyc.setAverageSkuNum(averageSkuNum);
        platformStatisticsByKyc.setMonthlyRefundRate(monthlyRefundRate);

        ZonedDateTime statisticsTime = now.withHour(1).withMinute(0).withSecond(0).withNano(0);

        Example example = new Example(PlatformStatisticsByStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andEqualTo("statisticsTime", Date.from(statisticsTime.toInstant()));
        List<PlatformStatisticsByKyc> platformStatisticsByKycs = platformStatisticsByKycMapper.selectByExample(example);
        if (platformStatisticsByKycs.size() == 0) {
            AmaMWSCommonUtil.initBaseData(platformStatisticsByKyc);
            platformStatisticsByKyc.setStatisticsTime(Date.from(statisticsTime.toInstant()));
            platformStatisticsByKycMapper.insertSelective(platformStatisticsByKyc);
            log.info("总店铺指标插入成功 kycNaturalId: {}, stataisticsTime: {}", kycNaturalId, statisticsTime);
        } else if (platformStatisticsByKycs.size() == 1) {
            PlatformStatisticsByKyc statistics = platformStatisticsByKycs.get(0);
            platformStatisticsByKyc.setId(statistics.getId());
            platformStatisticsByKyc.setCreateId(statistics.getCreateId());
            platformStatisticsByKyc.setCreateName(statistics.getCreateName());
            platformStatisticsByKyc.setCreateDts(statistics.getCreateDts());
            platformStatisticsByKyc.setUpdateId("sys");
            platformStatisticsByKyc.setUpdateName("sys");
            platformStatisticsByKyc.setUpdateDts(new Date());
            platformStatisticsByKyc.setStatisticsTime(Date.from(statisticsTime.toInstant()));
            log.info("总店铺指标更新成功 kycNaturalId: {}, stataisticsTime: {}", kycNaturalId, statisticsTime);
        } else {
            log.error("总店铺指标数据条目异常，kycNaturalId: {}, stattisticsTime: {}", kycNaturalId, statisticsTime);
            return null;
        }
        return platformStatisticsByKyc;
    }

    private BigDecimal calMonthlyRefundRate(List<JsSyncAmazonShipmentEvent> events) {
        // 计算总笔数
        long count = events.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal()
                        || e.getEventType() == AmazonEventTypeEnum.REFOUN.ordinal())
                .count();
        // 计算退款笔数数
        long refound = events.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.REFOUN.ordinal())
                .count();
        if (count == 0) {
            return BigDecimal.ZERO;
        } else {
            return new BigDecimal(refound).divide(new BigDecimal(count), 4, RoundingMode.HALF_UP);
        }
    }

    private BigDecimal calAverageSkuNum(List<JsSyncAmazonInventory> inventories) {
        // 库存报告一天只存一份，同一天库存报告的条数就等于 sku 数
        Map<String, Long> map = inventories.stream()
                .collect(Collectors.groupingBy(e -> DateUtil.format(e.getStatisticsTime(), "yyyy-MM-dd"),
                        Collectors.counting()));
        long count = map.entrySet().stream()
                .mapToLong(Map.Entry::getValue)
                .count();
        if (map.keySet().size() != 0) {
            return new BigDecimal(count).divide(new BigDecimal(map.keySet().size()), 4, RoundingMode.HALF_UP);
        } else {
            return BigDecimal.ZERO;
        }
    }

    private BigDecimal calPerCustomerTransaction(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events) {
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        return jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                .collect(new AveragingBigDecimal());
    }

    private BigDecimal calMonthlyOrderNum(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events) {
        List<String> orderIds = events.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal())
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        long count = jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .count();
        return new BigDecimal(count);
    }

    private BigDecimal calTaxRate(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEventCharge> eventCharges) {
        List<String> orderIds = eventCharges.stream()
                .map(JsSyncAmazonShipmentEventCharge::getOrderId)
                .collect(Collectors.toList());
        // 计算销售额
        BigDecimal sum = jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getOrderTotal() != null)
                .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 计算税额
        BigDecimal tax = eventCharges.stream()
                .filter(e -> "Tax".equals(e.getChargeType()))
                .map(e -> convertAmount(e, JsSyncAmazonShipmentEventCharge::getChargeAmount, JsSyncAmazonShipmentEventCharge::getChargeAmountCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        // 计算税率
        if (sum.compareTo(BigDecimal.ZERO) == 0) {
            return BigDecimal.ZERO;
        } else {
            return tax.divide(sum, 4, RoundingMode.HALF_UP);
        }
    }

    private BigDecimal calMonthlySale(List<JsSyncAmazonOrder> jsSyncAmazonOrders, List<JsSyncAmazonShipmentEvent> events) {
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        return jsSyncAmazonOrders.stream()
                .filter(e -> "Shipped".equals(e.getOrderStatus()))
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getOrderTotal() != null)
                .map(e -> convertAmount(e, JsSyncAmazonOrder::getOrderTotal, JsSyncAmazonOrder::getCurrencyCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private BigDecimal calMonthlyIncome(List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        return eventGroups.stream()
                .filter(e -> "Succeeded".equals(e.getFundTransferStatus()))
                .map(e -> convertAmount(e, JsSyncAmazonFinancialEventGroup::getOriginalTotal, JsSyncAmazonFinancialEventGroup::getOriginalTotalCode))
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private List<JsSyncAmazonInventory> getJsSyncAmazonInventories(String kycNaturalId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonInventory.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("userNaturalId", kycNaturalId);
        criteria.andGreaterThanOrEqualTo("statisticsTime", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("statisticsTime", Date.from(now.toInstant()));

        return jsSyncAmazonInventoryMapper.selectByExample(example);
    }

    private List<JsSyncAmazonShipmentEventCharge> getJsSyncAmazonShipmentEventsCharge(String kycNaturalId, List<JsSyncAmazonShipmentEvent> events) {
        if (events.size() == 0) {
            return Collections.emptyList();
        }
        // 获取事件对应的订单 id
        List<String> orderIds = events.stream()
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        Example example = new Example(JsSyncAmazonShipmentEventCharge.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andIn("orderId", orderIds);
        criteria.andEqualTo("userNaturalId", kycNaturalId);

        return jsSyncAmazonShipmentEventChargeMapper.selectByExample(example);
    }

    private List<JsSyncAmazonShipmentEvent> getJsSyncAmazonShipmentEvents(String kycNaturalId, List<JsSyncAmazonFinancialEventGroup> eventGroups) {
        List<String> groupIds = eventGroups.stream()
                .map(JsSyncAmazonFinancialEventGroup::getFinancialEventGroupId)
                .collect(Collectors.toList());
        Example example = new Example(JsSyncAmazonShipmentEvent.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andIn("financialEventGroupId", groupIds);
        return jsSyncAmazonShipmentEventMapper.selectByExample(example);
    }

    private List<JsSyncAmazonFinancialEventGroup> getJsSyncAmazonFinancialEventGroups(String kycNaturalId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonFinancialEventGroup.class);
        example.setOrderByClause("fund_transfer_date desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("userNaturalId", kycNaturalId);
        criteria.andEqualTo("processingStatus", "Closed");
        criteria.andEqualTo("fundTransferStatus", "Succeeded");
        criteria.andGreaterThanOrEqualTo("fundTransferDate", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("fundTransferDate", Date.from(now.toInstant()));
        Example.Criteria criteria1 = example.or();
        criteria1.andEqualTo("userNaturalId", kycNaturalId);
        criteria1.andEqualTo("processingStatus", "Open");
        return jsSyncAmazonFinancialEventGroupMapper.selectByExample(example);
    }

    private List<JsSyncAmazonOrder> getJsSyncAmazonOrders(String kycNaturalId, ZonedDateTime start, ZonedDateTime now) {
        Example example = new Example(JsSyncAmazonOrder.class);
        example.setOrderByClause("purchase_date asc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andGreaterThanOrEqualTo("purchaseDate", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("purchaseDate", Date.from(now.toInstant()));
        return jsSyncAmazonOrderMapper.selectByExample(example);
    }

    /**
     * 将对象中的金额都转换为人民币
     *
     * 目前只支持 美元、日元、英镑
     *
     * @param obj 需要转换的对象
     * @param amount 获取该对象金额的方法引用
     * @param currencyCode 获取该对象币种的方法引用
     * @return
     */
    private <T> BigDecimal convertAmount(T obj, Function<? super T, ? extends BigDecimal> amount,
                                         Function<? super T, ? extends String> currencyCode) {
        if (amount.apply(obj) == null) {
            return BigDecimal.ZERO;
        } else {
            if ("USD".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("6.9387"));
            } else if ("GBP".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("8.4326"));
            } else if ("JPY".equals(currencyCode.apply(obj))) {
                return amount.apply(obj).multiply(new BigDecimal("0.06510"));
            } else {
                log.error("目前不支持该币种：{}", currencyCode.apply(obj));
                throw LogicException.le(ResultEnum.ERROR);
            }
        }
    }

}
