package com.js.sync.utils;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.exceptions.UtilException;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.core.util.ReflectUtil;
import cn.hutool.core.util.StrUtil;
import com.base.IBaseMapper;
import com.js.dal.dao.model.IBaseModel;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.session.RowBounds;
import tk.mybatis.mapper.entity.Example;

import javax.xml.datatype.XMLGregorianCalendar;
import java.math.BigDecimal;
import java.time.*;
import java.util.Date;
import java.util.List;

/**
 * 亚马逊拉取服务通用工具类
 *
 * @author liutianyu
 */
@Slf4j
public class AmaMWSCommonUtil {

    /**
     * 将字符串转为 BigDecimal
     */
    public static BigDecimal convertBigDecimal(String number) {
        if (ObjectUtil.isEmpty(number)) {
            return null;
        } else {
            return new BigDecimal(number);
        }
    }

    /**
     * 将 xmlGregorianCalender 转化为 date
     */
    public static Date convertXMLGregorianCalendarToDate(XMLGregorianCalendar xmlGregorianCalendar) {
        if (ObjectUtil.isEmpty(xmlGregorianCalendar)) {
            return null;
        }
        int timezone = xmlGregorianCalendar.getTimezone();
        int year = xmlGregorianCalendar.getYear();
        int month = xmlGregorianCalendar.getMonth();
        int day = xmlGregorianCalendar.getDay();
        int hour = xmlGregorianCalendar.getHour();
        int minute = xmlGregorianCalendar.getMinute();
        int second = xmlGregorianCalendar.getSecond();
        int millisecond = xmlGregorianCalendar.getMillisecond(); // 这个数据亚马逊有时会传负值过来。
        ZonedDateTime utc = ZonedDateTime.of(year, month, day, hour, minute, second, 0, ZoneId.of("UTC"));
        return Date.from(utc.toInstant());
    }

    /**
     * 将 UTC 格式的时间字符串转对日期对象
     *
     * @param dateString
     * @return
     */
    public static Date convertISOStringToDate(String dateString) {
        String dateFormatter = "yyyy-MM-dd'T'HH:mm:ssX";
        if (ObjectUtil.isNotEmpty(dateString)) {
            return DateUtil.parse(dateString, dateFormatter);
        } else {
            return null;
        }
    }


    /**
     * 初始化实体的一些基本属性
     */
    public static <T extends IBaseModel> void initBaseData(T record) {
        Date currDate = Date.from(Instant.now());
        if (StrUtil.isEmpty(record.getId())) {
            record.setId(IdUtil.simpleUUID());
        }
        if (StrUtil.isEmpty(record.getCreateName())) {
            record.setCreateName("sys");
        }
        if (StrUtil.isEmpty(record.getCreateId())) {
            record.setCreateId("sys");
        }
        if (record.getCreateDts() == null) {
            record.setCreateDts(currDate);
        }
        if (record.getUpdateDts() == null) {
            record.setUpdateDts(currDate);
        }
    }

    /**
     * 获取同步时间
     *
     * @param storeId 店铺 id
     * @param clazz 保存时的实体类
     * @param iBaseMapper 保存时使用的 mapper 需要指定泛型
     *
     */
    public static <T> Date getSyncDate(String storeId, Class clazz, IBaseMapper<T> iBaseMapper) {
        Date syncDate;
        Example example = new Example(clazz);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        example.setOrderByClause("sync_date desc");
        List<T> groups = iBaseMapper.selectByExampleAndRowBounds(example, new RowBounds(0, 1));
        if (groups.size() == 1) {
            try {
                syncDate = ReflectUtil.invoke(groups.get(0), "getSyncDate");
            } catch (UtilException e) {
                log.error("同步时间获取失败，不存在 syncDate");
                e.printStackTrace();
                throw e;
            }
        } else {
            syncDate = Date.from(LocalDateTime.of(1, 1, 1, 0, 0, 0).toInstant(ZoneOffset.UTC));
        }
        return syncDate;
    }

    /**
     * 将日期转为当月的 1 号零点零分零秒。
     *
     * @param date
     * @return
     */
    public static Date convertToTheFristDayOfTheMonth(Date date) {
        ZonedDateTime utc = ZonedDateTime.ofInstant(date.toInstant(), ZoneId.of("UTC"))
                .withDayOfMonth(1)
                .withHour(0).withMinute(0).withSecond(0).withNano(0);
        return Date.from(utc.toInstant());
    }

}
