package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazonaws.mws.MarketplaceWebService;
import com.amazonaws.mws.MarketplaceWebServiceClient;
import com.amazonaws.mws.MarketplaceWebServiceConfig;
import com.amazonaws.mws.MarketplaceWebServiceException;
import com.amazonaws.mws.model.*;
import com.js.api.sync.service.AmazonSettlementReportService;
import com.js.common.constant.Constant;
import com.js.dal.dao.mapper.JsSyncAmazonSettlementReportMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.JsSyncAmazonSettlementReport;
import com.js.dal.dao.model.KycStore;
import com.js.sync.enums.Shop;
import com.js.sync.utils.AmaMWSCommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import java.io.*;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonSettlementReportServiceImpl implements AmazonSettlementReportService {

    private static final String REPORT_TYPE = "_GET_V2_SETTLEMENT_REPORT_DATA_FLAT_FILE_";

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonSettlementReportMapper jsSyncAmazonSettlementReportMapper;
    @Autowired
    SqlSessionFactory sqlSessionFactory;

    @Override
    public void startSyncSettlementReportWhole(String storeId) {
        startSyncSettlementReport(storeId, null);
    }
    @Override
    public void startSyncSettlementReportIncrement(String storeId) {
        Date date = Date.from(ZonedDateTime.now().minusMonths(1).toInstant());
        startSyncSettlementReport(storeId, date);
    }

    private void startSyncSettlementReport(String storeId, Date date) {
        if (ObjectUtil.isEmpty(storeId)) {
            return;
        }
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (ObjectUtil.isEmpty(kycStore)) {
            return;
        }
        // 获取调用亚马逊接口的客户端
        MarketplaceWebService service = getMarketplaceWebService(kycStore);
        // 获取请求
        GetReportListRequest request = getGetReportListRequest(kycStore);

        if (ObjectUtil.isNotEmpty(date)) {
            // 设置拉取时间。
            DatatypeFactory df;
            try {
                df = DatatypeFactory.newInstance();
            } catch (DatatypeConfigurationException e) {
                e.printStackTrace();
                throw new RuntimeException(e);
            }
            ZonedDateTime start = ZonedDateTime.ofInstant(date.toInstant(), ZoneId.of("UTC"));
            // 转换时间：
            XMLGregorianCalendar startDate = df
                    .newXMLGregorianCalendar(new GregorianCalendar(start.getYear(), start.getMonthValue(), start.getDayOfMonth()));
            request.setAvailableFromDate(startDate);
        }

        Example example = new Example(JsSyncAmazonSettlementReport.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        List<JsSyncAmazonSettlementReport> jsSyncAmazonSettlementReports = jsSyncAmazonSettlementReportMapper.selectByExample(example);
        List<String> reportIds = jsSyncAmazonSettlementReports.stream()
                .map(JsSyncAmazonSettlementReport::getReportId)
                .distinct()
                .collect(Collectors.toList());
        try {
            GetReportListResponse response = service.getReportList(request);
            GetReportListResult getReportListResult = response.getGetReportListResult();
            List<ReportInfo> reportInfoList = getReportListResult.getReportInfoList();

            for (ReportInfo reportInfo : reportInfoList) {
                if (reportIds.contains(reportInfo.getReportId())) {
                    log.info("已经同步 reportId: {},跳过", reportInfo.getReportId());
                } else {
                    saveSettlementReport(kycStore, service, reportInfo.getReportId());
                }
            }
            if (getReportListResult.isSetNextToken()) {
                getReportListByNextToken(kycStore, service, getReportListResult.getNextToken(), reportIds);
            }
        } catch (MarketplaceWebServiceException | InterruptedException | IOException e) {
            e.printStackTrace();
            log.error("结算报告同步出错,请重新拉取 storeId: {} 的数据", storeId);
        }
        log.info("结算报告，同步完成");
    }

    private void getReportListByNextToken(KycStore kycStore, MarketplaceWebService service, String nextToken, List<String> reportIds) throws MarketplaceWebServiceException, InterruptedException {
        log.info("有 nextToken: {} 暂停一分钟", nextToken);
        Thread.sleep(60000L);
        GetReportListByNextTokenRequest getReportListByNextTokenRequest = new GetReportListByNextTokenRequest();
        getReportListByNextTokenRequest.setMerchant(kycStore.getSellId());
        getReportListByNextTokenRequest.setMWSAuthToken(kycStore.getMwsAuthToken());
        getReportListByNextTokenRequest.setNextToken(nextToken);
        GetReportListByNextTokenResponse reportListByNextToken = service.getReportListByNextToken(getReportListByNextTokenRequest);
        GetReportListByNextTokenResult getReportListByNextTokenResult = reportListByNextToken.getGetReportListByNextTokenResult();
        List<ReportInfo> reportInfoList = getReportListByNextTokenResult.getReportInfoList();
        for (ReportInfo reportInfo : reportInfoList) {
            try {
                if (reportIds.contains(reportInfo.getReportId())) {
                    log.info("已经同步 reportId: {},跳过", reportInfo.getReportId());
                } else {
                    saveSettlementReport(kycStore, service, reportInfo.getReportId());
                }
            } catch (IOException e) {
                e.printStackTrace();
                log.info("结算报告同步出错，使用 nextToken：{}, 重新发送", nextToken);
            }
        }
        if (getReportListByNextTokenResult.isSetNextToken()) {
            getReportListByNextToken(kycStore, service, getReportListByNextTokenResult.getNextToken(), reportIds);
        }
    }

    private void saveSettlementReport(KycStore kycStore, MarketplaceWebService service, String reportId)
            throws InterruptedException, IOException {
        String fileName = getReport(kycStore, service, reportId);
        if (ObjectUtil.isEmpty(fileName)) {
            return;
        }
        List<JsSyncAmazonSettlementReport> insertList = new ArrayList<>();
        try {
            File tempFile = new File(fileName);
            BufferedReader br = new BufferedReader(new FileReader(tempFile));
            String line;
            int k = 0;
            HashMap<String, Integer> index = new HashMap<>();
            while ((line = br.readLine()) != null) { //循环读取行
                String[] segments = line.split("\t");
                if (k == 0) {
                    for (int i = 0; i < segments.length; i++) {
                        index.put(segments[i], i);
                    }
                } else {
                    JsSyncAmazonSettlementReport jsSyncAmazonSettlementReport = setJsSyncAmazonSettlementReportProperties(kycStore, index, segments);
                    jsSyncAmazonSettlementReport.setReportId(reportId);
                    insertList.add(jsSyncAmazonSettlementReport);
                }
                k++;
            }
            tempFile.delete();
        } catch (IOException e) {
            e.printStackTrace();
            log.error("fileName: {} 的文件没有找到！reportId: {}", fileName, reportId);
            throw e;
        }
        if (ObjectUtil.isNotEmpty(insertList)) {
            jsSyncAmazonSettlementReportMapper.insertListWithKey(insertList);
            log.info("保存成功：{} 条结算报告。settlementId:{}", insertList.size(), insertList.get(0).getSettlementId());
        }
        log.info("暂停一分钟");
        Thread.sleep(60000L);
    }

    private JsSyncAmazonSettlementReport setJsSyncAmazonSettlementReportProperties(KycStore kycStore, HashMap<String, Integer> index, String[] segments) {
        JsSyncAmazonSettlementReport jsSyncAmazonSettlementReport = new JsSyncAmazonSettlementReport();
        AmaMWSCommonUtil.initBaseData(jsSyncAmazonSettlementReport);
        jsSyncAmazonSettlementReport.setKycNaturalId(kycStore.getKycNaturalId());
        jsSyncAmazonSettlementReport.setStoreId(kycStore.getId());
        jsSyncAmazonSettlementReport.setSettlementId(getSegmProp(index, segments, "settlement-id"));
        jsSyncAmazonSettlementReport.setSettlementStartDate(AmaMWSCommonUtil.convertISOStringToDate(getSegmProp(index, segments, "settlement-start-date")));
        jsSyncAmazonSettlementReport.setSettlementEndDate(AmaMWSCommonUtil.convertISOStringToDate(getSegmProp(index, segments, "settlement-end-date")));
        jsSyncAmazonSettlementReport.setDepositDate(AmaMWSCommonUtil.convertISOStringToDate(getSegmProp(index, segments, "deposit-date")));
        jsSyncAmazonSettlementReport.setTotalAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "total-amount")));
        jsSyncAmazonSettlementReport.setCurrency(getSegmProp(index, segments, "currency"));
        jsSyncAmazonSettlementReport.setTransactionType(getSegmProp(index, segments, "transaction-type"));
        jsSyncAmazonSettlementReport.setOrderId(getSegmProp(index, segments, "order-id"));
        jsSyncAmazonSettlementReport.setMerchantOrderId(getSegmProp(index, segments, "merchant-order-id"));
        jsSyncAmazonSettlementReport.setShipmentId(getSegmProp(index, segments, "shipment-id"));
        jsSyncAmazonSettlementReport.setMarketplaceName(getSegmProp(index, segments, "marketplace-name"));
        jsSyncAmazonSettlementReport.setShipmentFeeType(getSegmProp(index, segments, "shipment-fee-type"));
        jsSyncAmazonSettlementReport.setShipmentFeeAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "shipment-fee-amount")));
        jsSyncAmazonSettlementReport.setOrderFeeType(getSegmProp(index, segments, "order-fee-type"));
        jsSyncAmazonSettlementReport.setOrderFeeAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "order-fee-amount")));
        jsSyncAmazonSettlementReport.setFulfillmentId(getSegmProp(index, segments, "fulfillment-id"));
        jsSyncAmazonSettlementReport.setPostedDate(AmaMWSCommonUtil.convertISOStringToDate(getSegmProp(index, segments, "posted-date")));
        jsSyncAmazonSettlementReport.setOrderItemCode(getSegmProp(index, segments, "order-item-code"));
        jsSyncAmazonSettlementReport.setMerchantOrderItemId(getSegmProp(index, segments, "merchant-order-item-id"));
        jsSyncAmazonSettlementReport.setMerchantAdjustmentItemId(getSegmProp(index, segments, "merchant-adjustment-item-id"));
        jsSyncAmazonSettlementReport.setSku(getSegmProp(index, segments, "sku"));
        jsSyncAmazonSettlementReport.setQuantityPurchased(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "quantity-purchased")));
        jsSyncAmazonSettlementReport.setPriceType(getSegmProp(index, segments, "price-type"));
        jsSyncAmazonSettlementReport.setPriceAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "price-amount")));
        jsSyncAmazonSettlementReport.setItemRelatedFeeType(getSegmProp(index, segments, "item-related-fee-type"));
        jsSyncAmazonSettlementReport.setItemRelatedFeeAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "item-related-fee-amount")));
        jsSyncAmazonSettlementReport.setMiscFeeAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "misc-fee-amount")));
        jsSyncAmazonSettlementReport.setOtherAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "other-fee-amount")));
        jsSyncAmazonSettlementReport.setOtherFeeReasonDescription(getSegmProp(index, segments, "other-fee-reason-description"));
        jsSyncAmazonSettlementReport.setDirectPaymentType(getSegmProp(index, segments, "direct-payment-type"));
        jsSyncAmazonSettlementReport.setDirectPaymentAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "direct-payment-amount")));
        jsSyncAmazonSettlementReport.setOtherAmount(AmaMWSCommonUtil.convertBigDecimal(getSegmProp(index, segments, "other-amount")));
        jsSyncAmazonSettlementReport.setStatisticsTime(AmaMWSCommonUtil.convertToTheFristDayOfTheMonth(new Date()));
        return jsSyncAmazonSettlementReport;
    }

    private String getSegmProp(HashMap<String, Integer> index, String[] segments, String key) {
        if (index.get(key) > segments.length - 1) {
            return null;
        } else {
            return segments[index.get(key)];
        }
    }

    private String getReport(KycStore kycStore, MarketplaceWebService service, String reportId) {
        GetReportRequest request = new GetReportRequest();
        request.setMerchant(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());

        request.setReportId(reportId);
        FileOutputStream report;
        String filename = "SettlementReport" + reportId;
        try {
            report = new FileOutputStream(filename);
            request.setReportOutputStream( report );
            GetReportResponse settlementReport = service.getReport(request);
            report.flush();
            return filename;
        } catch (MarketplaceWebServiceException | IOException e) {
            e.printStackTrace();
            log.error("结算报告：reportId:{} 需要手动同步更新", reportId);
            return null;
        }
    }

    private GetReportListRequest getGetReportListRequest(KycStore kycStore) {
        GetReportListRequest request = new GetReportListRequest()
                .withMerchant(kycStore.getSellId())
                .withMWSAuthToken(kycStore.getMwsAuthToken())
                .withReportTypeList(new TypeList(Arrays.asList(REPORT_TYPE)));
        return request;
    }

    private MarketplaceWebService getMarketplaceWebService(KycStore kycStore) {
        Shop shop = new Shop();
        MarketplaceWebServiceConfig config = new MarketplaceWebServiceConfig();
        config.setServiceURL(kycStore.getAmaServiceUrl());

        return new MarketplaceWebServiceClient(
                shop.getAmazonAccessKey(), shop.getAmazonSecrectKey(), "dmjsy", "1.0", config);
    }
}
