package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazonservices.mws.orders._2013_09_01.MarketplaceWebServiceOrdersAsyncClient;
import com.amazonservices.mws.orders._2013_09_01.MarketplaceWebServiceOrdersClient;
import com.amazonservices.mws.orders._2013_09_01.MarketplaceWebServiceOrdersConfig;
import com.amazonservices.mws.orders._2013_09_01.model.*;
import com.js.common.constant.Constant;
import com.js.dal.dao.mapper.JsSyncAmazonOrderItemMapper;
import com.js.dal.dao.mapper.JsSyncAmazonOrderMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.JsSyncAmazonOrder;
import com.js.dal.dao.model.JsSyncAmazonOrderItem;
import com.js.dal.dao.model.KycStore;
import com.js.sync.enums.Shop;
import com.js.api.sync.service.AmazonOrderItemService;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.DBUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;


/**
 * 同步订单明细
 *
 * 由于亚马逊节流限制，该接口同步数据速度较慢。
 */
@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
@Deprecated
public class AmazonOrderItemServiceImpl implements AmazonOrderItemService {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonOrderItemMapper jsSyncAmazonOrderItemMapper;
    @Autowired
    SqlSessionFactory sqlSessionFactory;

    @Override
    public void startSyncOrderItemWhole(String storeId) {

        // 获取对应店铺
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (ObjectUtil.isEmpty(kycStore)) {
            log.error("店铺 id 错误: {}", storeId);
            return;
        }
        // 计算需要同步的时间
        ZonedDateTime now = ZonedDateTime.now(ZoneId.of("UTC"));
        Date syncDate = Date.from(now.minusYears(2).toInstant());
        syncOrderItemBySyncDate(storeId, kycStore, syncDate);

    }

    @Override
    public void startSyncOrderItemIncrement(String storeId) {
        // 获取对应店铺
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (ObjectUtil.isEmpty(kycStore)) {
            log.error("店铺 id 错误: {}", storeId);
            return;
        }
        Date previousSyncDate = AmaMWSCommonUtil.getSyncDate(storeId, JsSyncAmazonOrder.class, jsSyncAmazonOrderMapper);
        if (previousSyncDate.getTime() < 0) {
            log.info("亚马逊店铺订单尚未进行过全量抓取，请进行进行全量抓取后再执行明细增量抓取。店铺 id: {}", storeId);
            return;
        } else {
            log.info("准备增量抓取，抓取最近两个月的订单。使用时间为：{}", previousSyncDate);
            Date syncDate = Date.from(ZonedDateTime.ofInstant(previousSyncDate.toInstant(), ZoneId.of("UTC")).minusMonths(2).toInstant());
            syncOrderItemBySyncDate(storeId, kycStore, syncDate);
            return;
        }

    }

    private void syncOrderItemBySyncDate(String storeId, KycStore kycStore, Date syncDate) {
        // 获取符合要求的全部订单
        List<JsSyncAmazonOrder> jsSyncAmazonOrders = getJsSyncAmazonOrders(storeId, syncDate);
        // 获取数据库中已有的商品明细信息
        List<String> orderIds = jsSyncAmazonOrders.stream().map(JsSyncAmazonOrder::getAmazonOrderId).collect(Collectors.toList());
        Example example = new Example(JsSyncAmazonOrderItem.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andIn("orderId", orderIds);
        List<JsSyncAmazonOrderItem> jsSyncAmazonOrderItems = jsSyncAmazonOrderItemMapper.selectByExample(example);
        // 获取亚马逊客户端
        MarketplaceWebServiceOrdersClient client = getClient(kycStore);

        List<JsSyncAmazonOrderItem> insertList = new ArrayList<JsSyncAmazonOrderItem>();
        List<JsSyncAmazonOrderItem> updateList = new ArrayList<JsSyncAmazonOrderItem>();

        for (JsSyncAmazonOrder jsSyncAmazonOrder : jsSyncAmazonOrders) {
            try {
                log.info("开始同步订单 id 为: {} 的明细，暂停3秒钟", jsSyncAmazonOrder.getAmazonOrderId());
                Thread.sleep(3000L);

                ListOrderItemsRequest request = getRequest(kycStore, jsSyncAmazonOrder.getAmazonOrderId());
                ListOrderItemsResponse listOrderItemsResponse = client.listOrderItems(request);
                ListOrderItemsResult listOrderItemsResult = listOrderItemsResponse.getListOrderItemsResult();
                List<OrderItem> orderItems = listOrderItemsResult.getOrderItems();
                saveOrderItems(jsSyncAmazonOrder, orderItems, jsSyncAmazonOrderItems, insertList, updateList);

                if (listOrderItemsResult.isSetNextToken()) {
                    getOrderItemByNextToken(client, kycStore, listOrderItemsResult.getNextToken(), jsSyncAmazonOrder,
                            jsSyncAmazonOrderItems, insertList, updateList);
                }
                if (insertList.size() > 100) {
                    jsSyncAmazonOrderItemMapper.insertListWithKey(insertList);
                    log.info("新增订单明细数据:{} 条", insertList.size());
                    insertList.clear();
                }
                if (updateList.size() > 100) {
                    DBUtil.updateBatch(updateList, JsSyncAmazonOrderItemMapper.class, sqlSessionFactory);
                    log.info("更新订单明细数据:{} 条", updateList.size());
                    updateList.clear();
                }
            } catch (InterruptedException e) {
                e.printStackTrace();
                log.error("订单同步失败 order id: {}", jsSyncAmazonOrder.getAmazonOrderId());
            }

        }

        if (ObjectUtil.isNotEmpty(insertList)) {
            jsSyncAmazonOrderItemMapper.insertListWithKey(insertList);
            log.info("新增订单明细数据:{} 条", insertList.size());
        }
        if (ObjectUtil.isNotEmpty(updateList)) {
            DBUtil.updateBatch(updateList, JsSyncAmazonOrderItemMapper.class, sqlSessionFactory);
            log.info("更新订单明细数据:{} 条", updateList.size());
        }
    }



    private void getOrderItemByNextToken(MarketplaceWebServiceOrdersClient client, KycStore kycStore, String nextToken,
                                         JsSyncAmazonOrder jsSyncAmazonOrder, List<JsSyncAmazonOrderItem> jsSyncAmazonOrderItems,
                                         List<JsSyncAmazonOrderItem> insertList, List<JsSyncAmazonOrderItem> updateList) {
        try {
            log.info("有 nextToken, 等待3秒钟");
            Thread.sleep(3000L);

            ListOrderItemsByNextTokenRequest request = getNextTokenRequest(kycStore, nextToken);
            ListOrderItemsByNextTokenResponse response = client.listOrderItemsByNextToken(request);
            ListOrderItemsByNextTokenResult listOrderItemsByNextTokenResult = response.getListOrderItemsByNextTokenResult();
            List<OrderItem> orderItems = listOrderItemsByNextTokenResult.getOrderItems();

            saveOrderItems(jsSyncAmazonOrder, orderItems, jsSyncAmazonOrderItems, insertList, updateList);

            if (listOrderItemsByNextTokenResult.isSetNextToken()) {
                getOrderItemByNextToken(client, kycStore, listOrderItemsByNextTokenResult.getNextToken(),
                        jsSyncAmazonOrder, jsSyncAmazonOrderItems, insertList, updateList);
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
            log.error("getOrderItemByNextToken nextToken: {}", nextToken);
        }
    }

    private ListOrderItemsByNextTokenRequest getNextTokenRequest(KycStore kycStore, String nextToken) {
        ListOrderItemsByNextTokenRequest request = new ListOrderItemsByNextTokenRequest();
        request.setSellerId(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());
        request.setNextToken(nextToken);
        return request;
    }

    private List<JsSyncAmazonOrder> getJsSyncAmazonOrders(String storeId, Date syncDate) {
        Example example = new Example(JsSyncAmazonOrder.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andGreaterThanOrEqualTo("purchaseDate", syncDate);
        return jsSyncAmazonOrderMapper.selectByExample(example);
    }

    /**
     * 保存商品明细
     *
     * 注意出于性能考虑，这里只是将要保存、更新的商品象放到了插入与更新列表中。
     */
    private void saveOrderItems(JsSyncAmazonOrder jsSyncAmazonOrder, List<OrderItem> orderItems,
                                List<JsSyncAmazonOrderItem> jsSyncAmazonOrderItems,
                                List<JsSyncAmazonOrderItem> insertList, List<JsSyncAmazonOrderItem> updateList) {
        for (OrderItem orderItem : orderItems) {
            boolean isExist = false;
            for (JsSyncAmazonOrderItem jsSyncAmazonOrderItem : jsSyncAmazonOrderItems) {
                if (jsSyncAmazonOrderItem.getOrderItemId().equals(orderItem.getOrderItemId())) {
                    isExist = true;
                    jsSyncAmazonOrderItem.setUpdateId("sys");
                    jsSyncAmazonOrderItem.setUpdateName("sys");
                    jsSyncAmazonOrderItem.setUpdateDts(Date.from(Instant.now()));
                    setJsSyncAmazonOrderItemProperties(orderItem, jsSyncAmazonOrderItem);
                    updateList.add(jsSyncAmazonOrderItem);
                }
            }
            if (!isExist) {
                JsSyncAmazonOrderItem jsSyncAmazonOrderItem = new JsSyncAmazonOrderItem();
                AmaMWSCommonUtil.initBaseData(jsSyncAmazonOrderItem);
                jsSyncAmazonOrderItem.setUserNaturalId(jsSyncAmazonOrder.getKycNaturalId());
                jsSyncAmazonOrderItem.setOrderId(jsSyncAmazonOrder.getAmazonOrderId());
                jsSyncAmazonOrderItem.setStoreId(jsSyncAmazonOrder.getStoreId());
                setJsSyncAmazonOrderItemProperties(orderItem, jsSyncAmazonOrderItem);
                insertList.add(jsSyncAmazonOrderItem);
            }
        }
    }

    private void setJsSyncAmazonOrderItemProperties(OrderItem orderItem, JsSyncAmazonOrderItem jsSyncAmazonOrderItem) {
        jsSyncAmazonOrderItem.setAsin(orderItem.getASIN());
        jsSyncAmazonOrderItem.setSellersku(orderItem.getSellerSKU());
        jsSyncAmazonOrderItem.setOrderItemId(orderItem.getOrderItemId());
        jsSyncAmazonOrderItem.setTitle(orderItem.getTitle());
        jsSyncAmazonOrderItem.setQuantityOrdered(new BigDecimal(orderItem.getQuantityOrdered()));
        jsSyncAmazonOrderItem.setQuantityShipped(new BigDecimal(orderItem.getQuantityShipped()));
        if (ObjectUtil.isNotEmpty(orderItem.getItemPrice())) {
            jsSyncAmazonOrderItem.setItemPriceCode(orderItem.getItemPrice().getCurrencyCode());
            jsSyncAmazonOrderItem.setItemPrice(AmaMWSCommonUtil.convertBigDecimal(orderItem.getItemPrice().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getShippingPrice())) {
            jsSyncAmazonOrderItem.setItemPriceCode(orderItem.getShippingPrice().getCurrencyCode());
            jsSyncAmazonOrderItem.setItemPrice(AmaMWSCommonUtil.convertBigDecimal(orderItem.getShippingPrice().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getGiftWrapPrice())) {
            jsSyncAmazonOrderItem.setGiftWrapPriceCode(orderItem.getGiftWrapPrice().getCurrencyCode());
            jsSyncAmazonOrderItem.setGiftWrapPrice(AmaMWSCommonUtil.convertBigDecimal(orderItem.getGiftWrapPrice().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getItemTax())) {
            jsSyncAmazonOrderItem.setItemTaxCode(orderItem.getItemTax().getCurrencyCode());
            jsSyncAmazonOrderItem.setItemTax(AmaMWSCommonUtil.convertBigDecimal(orderItem.getItemTax().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getShippingTax())) {
            jsSyncAmazonOrderItem.setShippingTaxCode(orderItem.getShippingTax().getCurrencyCode());
            jsSyncAmazonOrderItem.setShippingTax(AmaMWSCommonUtil.convertBigDecimal(orderItem.getShippingTax().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getGiftWrapTax())) {
            jsSyncAmazonOrderItem.setGiftWrapTaxCode(orderItem.getGiftWrapTax().getCurrencyCode());
            jsSyncAmazonOrderItem.setGiftWrapTax(AmaMWSCommonUtil.convertBigDecimal(orderItem.getGiftWrapTax().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getShippingDiscount())) {
            jsSyncAmazonOrderItem.setShippingDiscountCode(orderItem.getShippingDiscount().getCurrencyCode());
            jsSyncAmazonOrderItem.setShippingDiscount(AmaMWSCommonUtil.convertBigDecimal(orderItem.getShippingDiscount().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getShippingDiscountTax())) {
            jsSyncAmazonOrderItem.setShippingDiscountTaxCode(orderItem.getShippingDiscountTax().getCurrencyCode());
            jsSyncAmazonOrderItem.setShippingDiscountTax(AmaMWSCommonUtil.convertBigDecimal(orderItem.getShippingDiscountTax().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getPromotionDiscount())) {
            jsSyncAmazonOrderItem.setPromotionDiscountCode(orderItem.getPromotionDiscount().getCurrencyCode());
            jsSyncAmazonOrderItem.setPromotionDiscount(AmaMWSCommonUtil.convertBigDecimal(orderItem.getPromotionDiscount().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getPromotionDiscountTax())) {
            jsSyncAmazonOrderItem.setPromotionDiscountTaxCode(orderItem.getPromotionDiscountTax().getCurrencyCode());
            jsSyncAmazonOrderItem.setPromotionDiscountTax(AmaMWSCommonUtil.convertBigDecimal(orderItem.getPromotionDiscountTax().getAmount()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getCODFee())) {
            jsSyncAmazonOrderItem.setCodFeeCode(orderItem.getCODFee().getCurrencyCode());
            jsSyncAmazonOrderItem.setCodFee(AmaMWSCommonUtil.convertBigDecimal(orderItem.getCODFee().getCurrencyCode()));
        }
        if (ObjectUtil.isNotEmpty(orderItem.getCODFeeDiscount())) {
            jsSyncAmazonOrderItem.setCodFeeDiscountCode(orderItem.getCODFeeDiscount().getCurrencyCode());
            jsSyncAmazonOrderItem.setCodFeeDiscount(AmaMWSCommonUtil.convertBigDecimal(orderItem.getCODFeeDiscount().getAmount()));
        }
        jsSyncAmazonOrderItem.setIsGift(orderItem.getIsGift());
        jsSyncAmazonOrderItem.setGiftMessageText(orderItem.getGiftMessageText());
        jsSyncAmazonOrderItem.setGiftWrapLevel(orderItem.getGiftWrapLevel());
        jsSyncAmazonOrderItem.setConditionNote(orderItem.getConditionNote());
        jsSyncAmazonOrderItem.setConditionId(orderItem.getConditionId());
        jsSyncAmazonOrderItem.setConditionSubtypeId(orderItem.getConditionSubtypeId());
        jsSyncAmazonOrderItem.setPriceDesignation(orderItem.getPriceDesignation());
    }

    private ListOrderItemsRequest getRequest(KycStore kycStore, String orderId) {
        // Create a request.
        ListOrderItemsRequest request = new ListOrderItemsRequest();
        request.setSellerId(kycStore.getSellId());
        request.setMWSAuthToken(kycStore.getMwsAuthToken());
        request.setAmazonOrderId(orderId);
        return request;
    }

    private MarketplaceWebServiceOrdersClient getClient(KycStore kycStore) {
        MarketplaceWebServiceOrdersClient client;
        Shop shop = new Shop();

        MarketplaceWebServiceOrdersConfig config = new MarketplaceWebServiceOrdersConfig();
        config.setServiceURL(kycStore.getAmaServiceUrl());
        client = new MarketplaceWebServiceOrdersAsyncClient(shop.getAmazonAccessKey(), shop.getAmazonSecrectKey(),
                "replaceWithAppName", "replaceWithAppVersion", config, null);
        return client;
    }
}
