package com.js.sync.service.impl;

import cn.hutool.core.util.ObjectUtil;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceClient;
import com.amazon.mws.finances._2015_05_01.MWSFinancesServiceConfig;
import com.amazon.mws.finances._2015_05_01.model.*;
import com.amazonservices.mws.client.MwsUtl;
import com.js.common.constant.Constant;
import com.js.dal.dao.mapper.JsSyncAmazonFinancialEventGroupMapper;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.model.JsSyncAmazonFinancialEventGroup;
import com.js.dal.dao.model.KycStore;
import com.js.sync.enums.Shop;
import com.js.api.sync.service.AmazonFinanceEventGroupService;
import com.js.sync.utils.AmaMWSCommonUtil;
import com.js.sync.utils.DBUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.SqlSessionFactory;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import javax.xml.datatype.XMLGregorianCalendar;
import java.time.*;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class AmazonFinanceEventGroupServiceImpl implements AmazonFinanceEventGroupService {

    @Autowired
    SqlSessionFactory sqlSessionFactory;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    KycStoreMapper kycStoreMapper;

    @Override
    public void startSyncFinanceEventGroupWhole(String storeId) {
        // 抓取条件设为当前时间的两年之前
        Date syncDate = Date.from(ZonedDateTime.of(LocalDateTime.now(), ZoneId.of("UTC")).minusYears(2).toInstant());
        log.info("准备全量抓取，抓取最近两年的付款账单。使用时间为：{}", syncDate);
        startSyncFinanceEventGroupBySyncDate(storeId, syncDate);

    }

    @Override
    public void startSyncFinanceEventGroupIncrement(String storeId) {
        Date previousSyncDate = AmaMWSCommonUtil.getSyncDate(storeId, JsSyncAmazonFinancialEventGroup.class, jsSyncAmazonFinancialEventGroupMapper);
        if (previousSyncDate.getTime() < 0) {
            log.info("亚马逊店铺订单尚未进行过全量抓取，请进行进行全量抓取后再执行增量抓取。店铺 id: {}", storeId);
            return;
        } else {
            log.info("准备增量抓取，抓取最近两个月的订单。使用时间为：{}", previousSyncDate);
            Date syncDate = Date.from(ZonedDateTime.ofInstant(previousSyncDate.toInstant(), ZoneId.of("UTC")).minusMonths(2).toInstant());
            startSyncFinanceEventGroupBySyncDate(storeId, syncDate);
            return;
        }
    }

    private void startSyncFinanceEventGroupBySyncDate(String storeId, Date syncDate) {
        if (ObjectUtil.isEmpty(storeId) || ObjectUtil.isEmpty(syncDate)) {
            log.error("亚马逊付款账单同步失败 storeId, syncDate 不可以为空");
            return;
        }
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (ObjectUtil.isEmpty(kycStore)) {
            log.error("付款账单同步失败，无此店铺 id：{}", storeId);
            return;
        }
        MWSFinancesServiceClient client = getClient(kycStore.getAmaServiceUrl());

        ListFinancialEventGroupsRequest request = new ListFinancialEventGroupsRequest();
        String sellerId = kycStore.getSellId();
        request.setSellerId(sellerId);
        String mwsAuthToken = kycStore.getMwsAuthToken();
        request.setMWSAuthToken(mwsAuthToken);
        Example example = new Example(JsSyncAmazonFinancialEventGroup.class);
        Example.Criteria criteria = example.createCriteria();
        // TODO　目前亚马逊这个　api 有 bug ,时间范围限定的和查出来的不符最终会导致更新的数据和老数据对不上。所以直接把库里数据全部取出来。
        criteria.andEqualTo("storeId", kycStore.getId());
        List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventOldGroups =
                jsSyncAmazonFinancialEventGroupMapper.selectByExample(example);

        // 设置一页显示几条
//        Integer maxResultsPerPage = 30;
//        request.setMaxResultsPerPage(maxResultsPerPage);
        // 设置抓取的时间范围
        XMLGregorianCalendar financialEventGroupStartedAfter = MwsUtl.getDTF().newXMLGregorianCalendar();
        ZonedDateTime utcSearch = ZonedDateTime.ofInstant(syncDate.toInstant(), ZoneId.of("UTC"));
        financialEventGroupStartedAfter.setYear(utcSearch.getYear());
        financialEventGroupStartedAfter.setMonth(utcSearch.getMonthValue());
        financialEventGroupStartedAfter.setDay(utcSearch.getDayOfMonth());
        request.setFinancialEventGroupStartedAfter(financialEventGroupStartedAfter);
        log.info("准备同步付款账单数据，同步使用的时间为：{}", utcSearch);
        // 获取查询结果
        ListFinancialEventGroupsResponse response = client.listFinancialEventGroups(request);
        ListFinancialEventGroupsResult listFinancialEventGroupsResult = response.getListFinancialEventGroupsResult();
        List<FinancialEventGroup> financialEventGroupList = listFinancialEventGroupsResult.getFinancialEventGroupList();

        // 保存付款账单
        updateFinanceEventGroup(kycStore, financialEventGroupList, jsSyncAmazonFinancialEventOldGroups);
        // 如果没传完隔一分钟后继续
        if (listFinancialEventGroupsResult.isSetNextToken()) {
            getFinanceEventGroupsByNextToken(client, kycStore, sellerId, mwsAuthToken,
                    listFinancialEventGroupsResult.getNextToken(), jsSyncAmazonFinancialEventOldGroups);
        } else {
            log.info("付款账单拉取完毕");
        }

    }

    private MWSFinancesServiceClient getClient(String serviceUrl) {
        Shop shop2 = new Shop();
        MWSFinancesServiceConfig config = new MWSFinancesServiceConfig();
        config.setServiceURL(serviceUrl);
        String applicationName = "dmjishiyupay";
        String applicationVersion = "1.0";
        return new MWSFinancesServiceClient(shop2.getAmazonAccessKey(),
                shop2.getAmazonSecrectKey(), applicationName, applicationVersion, config);
    }

    private void updateFinanceEventGroup(KycStore kycStore, List<FinancialEventGroup> financialEventGroupList, List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventOldGroups) {
        if (ObjectUtil.isEmpty(jsSyncAmazonFinancialEventOldGroups)) {
            saveFinanceEventGroup(kycStore, financialEventGroupList);
        } else {
            List<JsSyncAmazonFinancialEventGroup> updateList = new ArrayList<>();
            List<FinancialEventGroup> insertList = new ArrayList<>();
            Date syncDate = Date.from(Instant.now());
            for (FinancialEventGroup financialEventGroup : financialEventGroupList) {
                boolean isExist = false;
                for (JsSyncAmazonFinancialEventGroup jsSyncAmazonFinancialEventOldGroup : jsSyncAmazonFinancialEventOldGroups) {
                    if (jsSyncAmazonFinancialEventOldGroup.getFinancialEventGroupId().equals(financialEventGroup.getFinancialEventGroupId())) {
                        isExist = true;
                        jsSyncAmazonFinancialEventOldGroup.setSyncDate(syncDate);
                        setJsSyncFinancialEventGroupProperties(jsSyncAmazonFinancialEventOldGroup, financialEventGroup);
                        updateList.add(jsSyncAmazonFinancialEventOldGroup);
                        break;
                    }
                }
                if (!isExist) {
                    insertList.add(financialEventGroup);
                }
            }
            saveFinanceEventGroup(kycStore, insertList);
            DBUtil.updateBatch(updateList, JsSyncAmazonFinancialEventGroupMapper.class, sqlSessionFactory);
        }
    }

    private void saveFinanceEventGroup(KycStore kycStore, List<FinancialEventGroup> financialEventGroupList) {
        if (ObjectUtil.isEmpty(financialEventGroupList) || ObjectUtil.isEmpty(kycStore)) {
            return;
        }
        List<JsSyncAmazonFinancialEventGroup> insertList = new ArrayList<>();
        Date syncDate = Date.from(Instant.now());
        for (FinancialEventGroup financialEventGroup : financialEventGroupList) {
            JsSyncAmazonFinancialEventGroup jsSyncAmazonFinancialEventGroup = new JsSyncAmazonFinancialEventGroup();
            AmaMWSCommonUtil.initBaseData(jsSyncAmazonFinancialEventGroup);
            jsSyncAmazonFinancialEventGroup.setStoreId(kycStore.getId());
            jsSyncAmazonFinancialEventGroup.setUserNaturalId(kycStore.getKycNaturalId());
            jsSyncAmazonFinancialEventGroup.setSyncDate(syncDate);
            setJsSyncFinancialEventGroupProperties(jsSyncAmazonFinancialEventGroup, financialEventGroup);
            insertList.add(jsSyncAmazonFinancialEventGroup);
        }
        jsSyncAmazonFinancialEventGroupMapper.insertListWithKey(insertList);
    }

    private void getFinanceEventGroupsByNextToken(MWSFinancesServiceClient client, KycStore kycStore, String sellerId,
                                                  String mwsAuthToken, String nextToken,
                                                  List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventOldGroups) {
        ListFinancialEventGroupsByNextTokenResult listFinancialEventGroupsByNextTokenResult;
        try {
            log.info("付款列表有 nextToken 暂停一分钟");
            Thread.sleep(60000);

            ListFinancialEventGroupsByNextTokenRequest request = new ListFinancialEventGroupsByNextTokenRequest();
            request.setSellerId(sellerId);
            request.setMWSAuthToken(mwsAuthToken);
            request.setNextToken(nextToken);
            ListFinancialEventGroupsByNextTokenResponse response = client.listFinancialEventGroupsByNextToken(request);
            listFinancialEventGroupsByNextTokenResult = response.getListFinancialEventGroupsByNextTokenResult();
            List<FinancialEventGroup> financialEventGroupList = listFinancialEventGroupsByNextTokenResult.getFinancialEventGroupList();
            // 保存付款账单
            updateFinanceEventGroup(kycStore, financialEventGroupList, jsSyncAmazonFinancialEventOldGroups);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("拉取亚马逊付款数据时出错：{}, 当前正拉取的 nextToken：{}", e.getMessage(), nextToken);
            return;
        }

        // 如果没有传完，递归。
        if (listFinancialEventGroupsByNextTokenResult.isSetNextToken()) {
            getFinanceEventGroupsByNextToken(client,kycStore, sellerId, mwsAuthToken,
                    listFinancialEventGroupsByNextTokenResult.getNextToken(), jsSyncAmazonFinancialEventOldGroups);
        } else {
            log.info("付款账单拉取完毕");
        }
    }

    /**
     * 设置实体属性
     */
    private void setJsSyncFinancialEventGroupProperties(JsSyncAmazonFinancialEventGroup jsSyncAmazonFinancialEventGroup,
                                                        FinancialEventGroup financialEventGroup) {
        jsSyncAmazonFinancialEventGroup.setUpdateDts(null);
        jsSyncAmazonFinancialEventGroup.setFinancialEventGroupId(financialEventGroup.getFinancialEventGroupId());
        jsSyncAmazonFinancialEventGroup.setProcessingStatus(financialEventGroup.getProcessingStatus());
        jsSyncAmazonFinancialEventGroup.setFundTransferStatus(financialEventGroup.getFundTransferStatus());
        if (ObjectUtil.isNotEmpty(financialEventGroup.getOriginalTotal())) {
            jsSyncAmazonFinancialEventGroup.setOriginalTotalCode(financialEventGroup.getOriginalTotal().getCurrencyCode());
            jsSyncAmazonFinancialEventGroup.setOriginalTotal(financialEventGroup.getOriginalTotal().getCurrencyAmount());
        }
        if (ObjectUtil.isNotEmpty(financialEventGroup.getConvertedTotal())) {
            jsSyncAmazonFinancialEventGroup.setConvertedTotalCode(financialEventGroup.getConvertedTotal().getCurrencyCode());
            jsSyncAmazonFinancialEventGroup.setConvertedTotal(financialEventGroup.getConvertedTotal().getCurrencyAmount());
        }
        jsSyncAmazonFinancialEventGroup.setFundTransferDate(
                AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(financialEventGroup.getFundTransferDate()));
        jsSyncAmazonFinancialEventGroup.setTraceId(financialEventGroup.getTraceId());
        jsSyncAmazonFinancialEventGroup.setAccountTail(financialEventGroup.getAccountTail());
        if (ObjectUtil.isNotEmpty(financialEventGroup.getBeginningBalance())) {
            jsSyncAmazonFinancialEventGroup.setBeginningBalanceCode(financialEventGroup.getBeginningBalance().getCurrencyCode());
            jsSyncAmazonFinancialEventGroup.setBeginningBalance(financialEventGroup.getBeginningBalance().getCurrencyAmount());
        }
        jsSyncAmazonFinancialEventGroup.setFinancialEventGroupStart(
                AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(financialEventGroup.getFinancialEventGroupStart()));
        jsSyncAmazonFinancialEventGroup.setFinancialEventGroupEnd(
                AmaMWSCommonUtil.convertXMLGregorianCalendarToDate(financialEventGroup.getFinancialEventGroupEnd()));
    }

}
