package com.js.sync.job;

import cn.hutool.core.util.ObjectUtil;
import com.js.common.enums.AmazonEventTypeEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.sync.service.impl.AmazonStatisticsServiceImpl;
import com.js.sync.utils.AmaMWSCommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 订单数据分析店铺维度。
 */
@Slf4j
@Component
public class AmazonDataAnalysisByStore {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    PlatformStatisticsByStoreMapper platformStatisticsByStoreMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;
    @Autowired
    AmazonStatisticsServiceImpl amazonStatisticsService;
    @Autowired
    JsSyncAmazonShipmentEventChargeMapper jsSyncAmazonShipmentEventChargeMapper;

    /**
     * 开始分析数据
     */
    public void startAnalysis(String storeId, Date analysisDate) {
        // 获取当前店铺
        KycStore kycStore = kycStoreMapper.selectByPrimaryKey(storeId);
        if (!(kycStore.getAuthStatus().equals(StoreAuthStatusEnum.AUTH.ordinal())
                || kycStore.getAuthStatus().equals(StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal()))) {
            log.info("店铺 id: {} 尚未成功授权，数据分析失败", storeId);
            return;
        }
        //计算绑定时间
        BigDecimal bindTime;
        if (kycStore.getAuthWithAccountDate().getTime() < 0) {
            bindTime = new BigDecimal(0);
        } else {
            bindTime = new BigDecimal(ZonedDateTime.ofInstant(kycStore.getAuthWithAccountDate().toInstant(), ZoneId.systemDefault())
                    .until(ZonedDateTime.now(), ChronoUnit.MONTHS));
        }
        // 计算时间区间。（上个月到其前13个月的）
        ZonedDateTime border = ZonedDateTime.ofInstant(analysisDate.toInstant(), ZoneId.of("UTC")).plusMonths(1);
        ZonedDateTime dateEnd = ZonedDateTime.of(border.getYear(), border.getMonthValue(), 1, 0, 0, 0, 0, ZoneId.of("UTC"));
        ZonedDateTime dateStart = dateEnd.minusMonths(13);
        List<JsSyncAmazonOrder> jsSyncAmazonOrders =
                amazonStatisticsService.getJsSyncAmazonOrders(Arrays.asList(storeId), dateStart, dateEnd);
        if (jsSyncAmazonOrders == null) {
            return;
        }
        JsSyncAmazonOrder jsSyncAmazonOldestOrder = jsSyncAmazonOrders.get(jsSyncAmazonOrders.size() - 1);
        ZonedDateTime acturalTime = ZonedDateTime.ofInstant(jsSyncAmazonOldestOrder.getPurchaseDate().toInstant(), ZoneId.of("UTC"));
        // 计算实际可以使用的完整月份。
        long months = acturalTime.until(dateEnd, ChronoUnit.MONTHS);
        if (months < 1) {
            log.error("数据量不足，亚马逊店铺维度数据分析失败");
            return;
        }
        ZonedDateTime dateStartNew = dateEnd.minusMonths(months);

        // 获取对应时间的事件组(现在只是美元)
        List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventGroups =
                amazonStatisticsService.getJsSyncAmazonFinancialEventGroups(Arrays.asList(storeId), dateStartNew, dateEnd);
        // 获取对应时间的付款事件
        List<JsSyncAmazonShipmentEvent> jsSyncAmazonShipmentEvents =
                amazonStatisticsService.getJsSyncShipmentEvents(jsSyncAmazonFinancialEventGroups);
        // 获取到全部有效订单的 id
        List<String> orderIds = jsSyncAmazonShipmentEvents.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal())
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        // 总销售额
        BigDecimal sumTotal = jsSyncAmazonOrders.stream()
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getOrderTotal() != null)
                .map(JsSyncAmazonOrder::getOrderTotal)
                .reduce(new BigDecimal(0), BigDecimal::add);

        // 月均销售额
        BigDecimal monthlySale = sumTotal.divide(BigDecimal.valueOf(months), 4, RoundingMode.HALF_UP);

        // 得到本月的事件组 id 列表
        ZonedDateTime monthStart = dateEnd.minusMonths(1);
        List<String> financialEventGroupMonthIds = jsSyncAmazonFinancialEventGroups.stream()
                .filter(e -> e.getFundTransferDate().getTime() < dateEnd.toInstant().toEpochMilli())
                .filter(e -> e.getFundTransferDate().getTime() >= monthStart.toInstant().toEpochMilli())
                .map(JsSyncAmazonFinancialEventGroup::getFinancialEventGroupId)
                .collect(Collectors.toList());
        // 本月的有效交易订单 id
        List<String> orderMonthIds = jsSyncAmazonShipmentEvents.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal())
                .filter(e -> financialEventGroupMonthIds.contains(e.getFinancialEventGroupId()))
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        if (ObjectUtil.isEmpty(orderMonthIds)) {
            orderMonthIds.add("");
        }
        // 本月退款的订单 id
        List<String> refoundOrderIds = jsSyncAmazonShipmentEvents.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.REFOUN.ordinal())
                .filter(e -> financialEventGroupMonthIds.contains(e.getFinancialEventGroupId()))
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        // 退款率
        BigDecimal refundRate = new BigDecimal(refoundOrderIds.size())
                .divide(new BigDecimal(orderIds.size()).add(new BigDecimal(refoundOrderIds.size()))
                        , 6, RoundingMode.HALF_UP);
        // 退款率环比波动率
        BigDecimal refundChainRate;
        List<PlatformStatisticsByStore> preOnePlatformStatisticsByStore = getPrePlatformStatisticsByStore(storeId, dateEnd, 1);
        if (ObjectUtil.isNotEmpty(preOnePlatformStatisticsByStore)) {
            if (refundRate.intValue() == 0) {
                refundChainRate = refundRate.subtract(preOnePlatformStatisticsByStore.get(0).getRefundRate());
            } else {
                refundChainRate = refundRate.subtract(preOnePlatformStatisticsByStore.get(0).getRefundRate())
                        .divide(refundRate, 4, RoundingMode.HALF_UP);
            }
        } else {
            refundChainRate = new BigDecimal(0);
        }

        // 在售商品数量
        List<JsSyncAmazonInventory> jsSyncAmazonInventory = getJsSyncAmazonInventory(storeId, Date.from(dateEnd.toInstant()));
        BigDecimal availableProductNum = new BigDecimal(jsSyncAmazonInventory.size());
        // 在售商品数量环比波动率
        BigDecimal availableProductNumChainRate;
        if (ObjectUtil.isNotEmpty(preOnePlatformStatisticsByStore) && availableProductNum.intValue() != 0) {
            availableProductNumChainRate = availableProductNum.subtract(preOnePlatformStatisticsByStore.get(0).getAvailableProductNum())
                    .divide(availableProductNum, 4, RoundingMode.HALF_UP);
        } else {
            availableProductNumChainRate = new BigDecimal(0);
        }
        // 月入账金额(以实际到账时间为准)
        BigDecimal income = jsSyncAmazonFinancialEventGroups.stream()
                .filter(e -> financialEventGroupMonthIds.contains(e.getFinancialEventGroupId()))
                .map(JsSyncAmazonFinancialEventGroup::getOriginalTotal)
                .reduce(new BigDecimal(0), BigDecimal::add);
        // 笔均入账金额
        BigDecimal averageIncome;
        if (financialEventGroupMonthIds.size() > 0) {
            averageIncome = income.divide(new BigDecimal(financialEventGroupMonthIds.size()), 4, RoundingMode.HALF_UP);
        } else {
            averageIncome = new BigDecimal(0);
        }
        // 月均入账金额
        BigDecimal monthlyIncome = jsSyncAmazonFinancialEventGroups.stream()
                .map(JsSyncAmazonFinancialEventGroup::getOriginalTotal)
                .reduce(new BigDecimal(0), BigDecimal::add).divide(new BigDecimal(months), 2, RoundingMode.HALF_UP);
        // 获取本月全部商品明细 id
        List<JsSyncAmazonShipmentEventCharge> jsSyncAmazonShipmentEventCharges = getJsSyncAmazonShipmentEventCharges(orderMonthIds);
        // 根据 sku 取到对应商品的 asin
        // 以商品对应的 asin 进行分组，统计销售额。
        AtomicReference<Map<String, BigDecimal>> groupByASIN = new AtomicReference<>(jsSyncAmazonShipmentEventCharges.stream()
                .filter(e -> "Principal".equals(e.getChargeType()))
                .collect(Collectors.groupingBy(e -> getAsinBySku(e.getSellersku(), jsSyncAmazonInventory),
                        Collectors.mapping(JsSyncAmazonShipmentEventCharge::getChargeAmount,
                                Collectors.reducing(new BigDecimal(0), BigDecimal::add)))));
        // 以销售额进行降序分组
        LinkedHashMap<String, BigDecimal> itemTotal = groupByASIN.get().entrySet().stream()
                .sorted((a, b) -> b.getValue().subtract(a.getValue()).intValue())
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue,
                        (u, v) -> {
                            throw new IllegalStateException(String.format("Duplicate key %s", u));
                        },
                        LinkedHashMap::new));
        // 月总销售额
        BigDecimal sumSaleTotal = new BigDecimal(0);
        for (Map.Entry<String, BigDecimal> stringBigDecimalEntry : itemTotal.entrySet()) {
            sumSaleTotal = sumSaleTotal.add(stringBigDecimalEntry.getValue());
        }
        // 销售额前10名商品占比
        BigDecimal salesTopTenRatio;
        if (itemTotal.entrySet().size() < 10) {
            salesTopTenRatio = new BigDecimal(1);
        } else {
            BigDecimal salesToeTen = new BigDecimal(0);
            int i = 0;
            for (Map.Entry<String, BigDecimal> stringBigDecimalEntry : itemTotal.entrySet()) {
                i++;
                salesToeTen = salesToeTen.add(stringBigDecimalEntry.getValue());
                if (i > 9) {
                    break;
                }
            }
            salesTopTenRatio = salesToeTen.divide(sumSaleTotal, 4, RoundingMode.HALF_UP);
        }
        // 销售额前3名商品占比
        BigDecimal salesTopThreeRatio;
        if (itemTotal.entrySet().size() < 3) {
            salesTopThreeRatio = new BigDecimal(1);
        } else {
            BigDecimal salesThreeTen = new BigDecimal(0);
            int i = 0;
            for (Map.Entry<String, BigDecimal> stringBigDecimalEntry : itemTotal.entrySet()) {
                i++;
                salesThreeTen = salesThreeTen.add(stringBigDecimalEntry.getValue());
                if (i > 2) {
                    break;
                }
            }
            salesTopThreeRatio = salesThreeTen.divide(sumSaleTotal, 4, RoundingMode.HALF_UP);
        }
        // 绑定回款账户时长

        // 近一个月入账金额环比增长率
        BigDecimal incomeChainRate;
        if (ObjectUtil.isNotEmpty(preOnePlatformStatisticsByStore) && income.intValue() != 0) {
            incomeChainRate = income.subtract(preOnePlatformStatisticsByStore.get(0).getIncome()).divide(income, 4, RoundingMode.HALF_UP);
        } else {
            incomeChainRate = new BigDecimal(0);
        }
        // 近一个月入账金额与前3个月平均入账金额比例
        BigDecimal incomeThreeRatio = calIncomeRatio(storeId, dateEnd, income, 3);
        // 近一个月入账金额与前6个月平均入账金额比例
        BigDecimal incomeSixRatio = calIncomeRatio(storeId, dateEnd, income, 6);
        // 入账金额同比增长率
        BigDecimal incomeGrowthRateOnYear;
        List<PlatformStatisticsByStore> preTwelvePlatformStatisticsByStore = getPrePlatformStatisticsByStore(storeId, dateEnd, 12);
        if (ObjectUtil.isNotEmpty(preTwelvePlatformStatisticsByStore)
                && preOnePlatformStatisticsByStore.get(0).getIncome().intValue() != 0) {
            incomeGrowthRateOnYear = income.subtract(preTwelvePlatformStatisticsByStore.get(0).getIncome())
                    .divide(preTwelvePlatformStatisticsByStore.get(0).getIncome(), 4, RoundingMode.HALF_UP);
        } else {
            incomeGrowthRateOnYear = new BigDecimal(0);
        }

        Example example = new Example(PlatformStatisticsByStore.class);
        example.setOrderByClause("statistics_time desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andEqualTo("statisticsTime", Date.from(dateEnd.toInstant()));
        PlatformStatisticsByStore platformStatisticsByStore;
        List<PlatformStatisticsByStore> platformStatisticsByStores = platformStatisticsByStoreMapper.selectByExample(example);
        boolean isNew;
        if (platformStatisticsByStores.size() > 1) {
            log.error("店铺维度数据条目异常 storeId: {}", storeId);
            return;
        } else if (platformStatisticsByStores.size() == 1) {
            isNew = false;
            platformStatisticsByStore = platformStatisticsByStores.get(0);
            platformStatisticsByStore.setUpdateId("sys");
            platformStatisticsByStore.setUpdateName("sys");
            platformStatisticsByStore.setUpdateDts(Date.from(Instant.now()));
        } else {
            isNew = true;
            platformStatisticsByStore = new PlatformStatisticsByStore();
            AmaMWSCommonUtil.initBaseData(platformStatisticsByStore);
        }
        platformStatisticsByStore.setStoreId(storeId);
        platformStatisticsByStore.setMonthlySale(monthlySale);
//        platformStatisticsByStore.setMonthlySaleRatio()
        platformStatisticsByStore.setIncome(income);
        platformStatisticsByStore.setAverageIncome(averageIncome);
        platformStatisticsByStore.setMonthlyIncome(monthlyIncome);
        platformStatisticsByStore.setIncomeChainRate(incomeChainRate);
        platformStatisticsByStore.setIncomeGrowthRateOnYear(incomeGrowthRateOnYear);
//        platformStatisticsByStore.setPerCustomerTransaction()
//        platformStatisticsByStore.setPerCusTransGrowthRateOnYear()
//        platformStatisticsByStore.setMonthlyOrderNum()
        platformStatisticsByStore.setRefundRate(refundRate);
//        platformStatisticsByStore.setRefundGrowthRateOnYear()
        platformStatisticsByStore.setBindTime(bindTime);
        platformStatisticsByStore.setRefundChainRate(refundChainRate);
        platformStatisticsByStore.setAvailableProductNum(availableProductNum);
        platformStatisticsByStore.setAvailableProductNumChainRate(availableProductNumChainRate);
        platformStatisticsByStore.setSalesTopTenRatio(salesTopTenRatio);
        platformStatisticsByStore.setSalesTopThreeRatio(salesTopThreeRatio);
        platformStatisticsByStore.setIncomeThreeRatio(incomeThreeRatio);
        platformStatisticsByStore.setIncomeSixRatio(incomeSixRatio);
        platformStatisticsByStore.setStatisticsTime(Date.from(dateEnd.toInstant()));
        if (isNew) {
            platformStatisticsByStoreMapper.insertSelective(platformStatisticsByStore);
            log.info("保存店铺维度信息成功");
        } else {
            platformStatisticsByStoreMapper.updateByPrimaryKey(platformStatisticsByStore);
            log.info("更新店铺维度信息成功");
        }
    }

    /**
     * 计算入账与前 n 个月平均入账所占比例。
     * @param storeId
     * @param dateEnd
     * @param income
     * @param month 统计本月与前 month 的比例
     * @return
     */
    private BigDecimal calIncomeRatio(String storeId, ZonedDateTime dateEnd, BigDecimal income, int month) {
        List<PlatformStatisticsByStore> prePlatformStatisticsByStores = getPrePlatformStatisticsByStore(storeId, dateEnd, month);
        if (ObjectUtil.isNotEmpty(prePlatformStatisticsByStores)) {
            BigDecimal preMonthlySale = prePlatformStatisticsByStores.stream()
                    .map(PlatformStatisticsByStore::getIncome)
                    .reduce(new BigDecimal(0), BigDecimal::add).divide(new BigDecimal(month), 4, RoundingMode.HALF_UP);
            if (preMonthlySale.intValue() != 0) {
                return income.divide(preMonthlySale, 4, RoundingMode.HALF_UP);
            } else {
                return new BigDecimal(0);
            }
        } else {
            return new BigDecimal(0);
        }
    }

    private List<JsSyncAmazonShipmentEventCharge> getJsSyncAmazonShipmentEventCharges(List<String> orderMonthIds) {
        Example example = new Example(JsSyncAmazonShipmentEventCharge.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andIn("orderId", orderMonthIds);
        return jsSyncAmazonShipmentEventChargeMapper.selectByExample(example);
    }

    /**
     * 通过商品的 sku 取到获取对应的 asin 号
     *
     * 如果取不到则返回 sku 并打印日志进行提示。
     */
    private String getAsinBySku(String sku, List<JsSyncAmazonInventory> jsSyncAmazonInventory) {
        for (JsSyncAmazonInventory syncAmazonInventory : jsSyncAmazonInventory) {
            if (syncAmazonInventory.getSellersku().equals(sku)) {
                return syncAmazonInventory.getAsin();
            }
        }
        log.error("库存记录不全，没有找到 sku: {} 对应的 asin", sku);
        return sku;
    }

    private List<JsSyncAmazonInventory> getJsSyncAmazonInventory(String storeId, Date statisticsTime) {
        Example example = new Example(JsSyncAmazonInventory.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andEqualTo("statisticsTime", statisticsTime);
        return jsSyncAmazonInventoryMapper.selectByExample(example);
    }

    /**
     * 根据所给日期，获取其前 n 个月的统计信息
     *
     * @param storeId
     * @param date
     * @param month 前几个月，不可以为 0
     */
    private List<PlatformStatisticsByStore> getPrePlatformStatisticsByStore(String storeId, ZonedDateTime date, int month) {
        Example example = new Example(PlatformStatisticsByStore.class);
        example.setOrderByClause("statistics_time desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("storeId", storeId);
        criteria.andLessThanOrEqualTo("statisticsTime", Date.from(date.minusMonths(1).toInstant()));
        criteria.andGreaterThanOrEqualTo("statisticsTime", Date.from(date.minusMonths(month).toInstant()));
        List<PlatformStatisticsByStore> platformStatisticsByStores = platformStatisticsByStoreMapper.selectByExample(example);
        if (platformStatisticsByStores.size() > month) {
            log.error("店铺统计信息数据条目错误。店铺 id: {}, 统计时间：{}", storeId, Date.from(date.minusMonths(1).toInstant()));
            return null;
        } else if (platformStatisticsByStores.size() == month) {
            return platformStatisticsByStores;
        } else {
            log.info("店铺 id: {}, 数据不足{}个月，获取失败", storeId, month);
            return null;
        }
    }
}
