package com.js.sync.job;

import cn.hutool.core.util.ObjectUtil;
import com.js.common.enums.AmazonEventTypeEnum;
import com.js.common.enums.SiteEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import com.js.sync.service.impl.AmazonStatisticsServiceImpl;
import com.js.sync.utils.AmaMWSCommonUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.*;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 订单数据分析。
 */
@Slf4j
@Component
public class AmazonDataAnalysisByKyc {

    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    JsSyncAmazonOrderMapper jsSyncAmazonOrderMapper;
    @Autowired
    JsSyncAmazonFinancialEventGroupMapper jsSyncAmazonFinancialEventGroupMapper;
    @Autowired
    JsSyncAmazonShipmentEventMapper jsSyncAmazonShipmentEventMapper;
    @Autowired
    PlatformStatisticsByKycMapper platformStatisticsByKycMapper;
    @Autowired
    JsSyncAmazonInventoryMapper jsSyncAmazonInventoryMapper;

    @Autowired
    AmazonStatisticsServiceImpl amazonStatisticsService;


    /**
     * 以 kyc 维度分析数据
     *
     * @param kycNaturalId kyc 用户的 id
     * @param analysisDate 需要分析的时间（主要用于确认分析的年月）
     */
    public void startAnalysis(String kycNaturalId, Date analysisDate) {

        List<KycStore> kycStores = getKycStores(kycNaturalId);
        // 获取平台数
//        BigDecimal platFormNum = BigDecimal.valueOf(kycStores.stream().collect(Collectors.groupingBy(KycStore::getStorePlateformCode)).keySet().size());
        // 获取店铺数
        BigDecimal storeNum = BigDecimal.valueOf(kycStores.size());
        if (storeNum.intValue() == 0) {
            log.error("没有获取到有效店铺，亚马逊 kyc 维度数据分析失败");
            return;
        }
        // 统计亚马逊美国店铺数量
        List<KycStore> amazonUSStores = kycStores.stream().filter(e -> SiteEnum.NORTH_AMERICA_SITE.name().equals(e.getStoreTheSiteCode())).collect(Collectors.toList());
//        // 马逊美国店铺数量占比
//        BigDecimal amazonusshopRatio = BigDecimal.valueOf(amazonUSStores.size()).divide(storeNum, 4, RoundingMode.HALF_UP);

        // 计算时间区间。（上个月到其前13个月的）
        ZonedDateTime border = ZonedDateTime.ofInstant(analysisDate.toInstant(), ZoneId.of("UTC")).plusMonths(1);
        ZonedDateTime dateEnd = ZonedDateTime.of(border.getYear(), border.getMonthValue(), 1, 0, 0, 0, 0, ZoneId.of("UTC"));
        ZonedDateTime dateStart = dateEnd.minusMonths(13);
        List<String> storeIds = kycStores.stream().map(KycStore::getId).collect(Collectors.toList());
        // 获取对应时间的订单
        List<JsSyncAmazonOrder> jsSyncAmazonOrders =
                amazonStatisticsService.getJsSyncAmazonOrders(storeIds, dateStart, dateEnd);
        if (ObjectUtil.isEmpty(jsSyncAmazonOrders)) {
            log.error("数据量不足，亚马逊 kyc 维度数据分析失败");
            return;
        }
        JsSyncAmazonOrder jsSyncAmazonOldestOrder = jsSyncAmazonOrders.get(jsSyncAmazonOrders.size() - 1);
        ZonedDateTime acturalTime = ZonedDateTime.ofInstant(jsSyncAmazonOldestOrder.getPurchaseDate().toInstant(), ZoneId.of("UTC"));
        // 计算实际可以使用的完整月份。
        long months = acturalTime.until(dateEnd, ChronoUnit.MONTHS);
        if (months < 1) {
            log.error("数据量不足，亚马逊 kyc 维度数据分析失败");
            return;
        }
        ZonedDateTime dateStartNew = dateEnd.minusMonths(months);

        // 获取对应时间的事件组(现在只是美元)
        List<JsSyncAmazonFinancialEventGroup> jsSyncAmazonFinancialEventGroups =
                amazonStatisticsService.getJsSyncAmazonFinancialEventGroups(storeIds, dateStartNew, dateEnd);
        // 获取对应时间的付款事件
        List<JsSyncAmazonShipmentEvent> jsSyncAmazonShipmentEvents =
                amazonStatisticsService.getJsSyncShipmentEvents(jsSyncAmazonFinancialEventGroups);
        // 计算亚马逊美国店铺近半年销售额占比
        BigDecimal halfYearSaleRatio;
        if (months >= 6) {
            ZonedDateTime dateHalfYear = dateEnd.minusMonths(6);
            List<JsSyncAmazonFinancialEventGroup> halfYearFinancialEventGroups = jsSyncAmazonFinancialEventGroups.stream()
                    .filter(e -> e.getFundTransferDate().getTime() > dateHalfYear.toInstant().toEpochMilli())
                    .collect(Collectors.toList());
            List<String> eventGroupIds = halfYearFinancialEventGroups.stream()
                    .map(JsSyncAmazonFinancialEventGroup::getFinancialEventGroupId)
                    .collect(Collectors.toList());
            List<String> halfYearOrderIds = jsSyncAmazonShipmentEvents.stream()
                    .filter(e -> eventGroupIds.contains(e.getFinancialEventGroupId()))
                    .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                    .collect(Collectors.toList());
            // 近半年美国店铺售额
            BigDecimal usHalfSaleTotal = jsSyncAmazonOrders.stream()
                    .filter(e -> halfYearOrderIds.contains(e.getAmazonOrderId()))
                    .filter(e -> e.getOrderTotal() != null)
                    .map(JsSyncAmazonOrder::getOrderTotal)
                    .reduce(new BigDecimal(0), BigDecimal::add);
            // 近半年总销售额（TODO 目前和上面的是一样的）
            BigDecimal halfSaleTotal = jsSyncAmazonOrders.stream()
                    .filter(e -> halfYearOrderIds.contains(e.getAmazonOrderId()))
                    .filter(e -> e.getOrderTotal() != null)
                    .map(JsSyncAmazonOrder::getOrderTotal)
                    .reduce(new BigDecimal(0), BigDecimal::add);
            if (halfSaleTotal.intValue() != 0) {
                halfYearSaleRatio =  usHalfSaleTotal.divide(halfSaleTotal, 4, RoundingMode.HALF_UP);
            } else {
                halfYearSaleRatio = new BigDecimal(0);
            }
        } else {
            halfYearSaleRatio = new BigDecimal(0);
        }
        // 获取到全部有效订单的 id
        List<String> orderIds = jsSyncAmazonShipmentEvents.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal())
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());

        // 总销售额
        BigDecimal sumTotal = jsSyncAmazonOrders.stream()
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getOrderTotal() != null)
                .map(JsSyncAmazonOrder::getOrderTotal)
                .reduce(new BigDecimal(0), BigDecimal::add);
        // 美国亚马逊销售总额 TODO 目前同上，，还需再考虑
        List<String> amazonUSStoreIds = amazonUSStores.stream().map(KycStore::getId).collect(Collectors.toList());
//        BigDecimal sumUSTotal = jsSyncAmazonOrders.stream()
//                .filter(e -> amazonUSStoreIds.contains(e.getStoreId()))
//                .filter(e -> "USD".equals(e.getCurrencyCode()))
//                .filter(e -> e.getOrderTotal() != null)
//                .filter(e -> e.getPurchaseDate().getTime() >= dateStartNew.toInstant().toEpochMilli())
//                .map(JsSyncAmazonOrder::getOrderTotal)
//                .reduce(new BigDecimal(0), BigDecimal::add);
        BigDecimal sumUSTotal = jsSyncAmazonOrders.stream()
                .filter(e -> orderIds.contains(e.getAmazonOrderId()))
                .filter(e -> e.getOrderTotal() != null)
                .map(JsSyncAmazonOrder::getOrderTotal)
                .reduce(new BigDecimal(0), BigDecimal::add);

//        // 亚马逊美国店铺销售额占比
//        BigDecimal amazonussaleRatio = sumUSTotal.divide(sumTotal, 4, RoundingMode.HALF_UP);
        // 月均销售额
        BigDecimal monthlySale = sumTotal.divide(BigDecimal.valueOf(months), 4, RoundingMode.HALF_UP);

        // 得到本月的事件组 id 列表
        ZonedDateTime monthStart = dateEnd.minusMonths(1);
        List<String> financialEventGroupMonthIds = jsSyncAmazonFinancialEventGroups.stream()
                .filter(e -> e.getFundTransferDate().getTime() < dateEnd.toInstant().toEpochMilli())
                .filter(e -> e.getFundTransferDate().getTime() >= monthStart.toInstant().toEpochMilli())
                .map(e -> e.getFinancialEventGroupId())
                .collect(Collectors.toList());
        // 入账金额，只计本月的(以实际到账时间为准)。
        BigDecimal income = jsSyncAmazonFinancialEventGroups.stream()
                .filter(e -> financialEventGroupMonthIds.contains(e.getFinancialEventGroupId()))
                .map(JsSyncAmazonFinancialEventGroup::getOriginalTotal)
                .reduce(new BigDecimal(0), BigDecimal::add);
        // 月均入账金额
        BigDecimal monthlyIncome = jsSyncAmazonFinancialEventGroups.stream()
                .filter(e -> e.getOriginalTotal() != null)
                .map(JsSyncAmazonFinancialEventGroup::getOriginalTotal)
                .reduce(new BigDecimal(0), BigDecimal::add)
                .divide(new BigDecimal(months), 4, RoundingMode.HALF_UP);

        // 本月的有效交易订单 id
        List<String> orderMonthIds = jsSyncAmazonShipmentEvents.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.SHIPMENT.ordinal())
                .filter(e -> financialEventGroupMonthIds.contains(e.getFinancialEventGroupId()))
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        // 取本月所有店铺完成的订单
        List<JsSyncAmazonOrder> jsSyncAmazonOrdersMonth = jsSyncAmazonOrders.stream()
                .filter(e -> orderMonthIds.contains(e.getAmazonOrderId()))
                .collect(Collectors.toList());
        // 本月销售总额
        BigDecimal sumTotalMonth = jsSyncAmazonOrdersMonth.stream()
                .map(JsSyncAmazonOrder::getOrderTotal).reduce(new BigDecimal(0), BigDecimal::add);
        // 月订单量
        BigDecimal monthlyOrderNum = BigDecimal.valueOf(jsSyncAmazonOrdersMonth.size());
        // 客单价
        BigDecimal perCustomerTransaction;
        if (monthlyOrderNum.intValue() == 0) {
            log.info("sumTotalMonth: {}", sumTotalMonth);
            perCustomerTransaction = new BigDecimal(0);
        } else {
            perCustomerTransaction = sumTotalMonth.divide(monthlyOrderNum, 4, RoundingMode.HALF_UP);
        }

        // 月均退款率
        // 获取到全部退款的订单 id
        List<String> refoundOrderIds = jsSyncAmazonShipmentEvents.stream()
                .filter(e -> e.getEventType() == AmazonEventTypeEnum.REFOUN.ordinal())
                .map(JsSyncAmazonShipmentEvent::getAmazonOrderId)
                .collect(Collectors.toList());
        BigDecimal monthlyRefundRate = new BigDecimal(refoundOrderIds.size()).divide(new BigDecimal(orderIds.size()), 6, RoundingMode.HALF_UP);

        // 在售商品数量。
        List<JsSyncAmazonInventory> jsSyncAmazonInventories = getJsSyncAmazonInventories(kycNaturalId, dateEnd);
        BigDecimal availableProductNum = new BigDecimal(jsSyncAmazonInventories.stream().map(JsSyncAmazonInventory::getAsin).distinct().count());

        // 计算同比相关指标
        // 获取去年同一月份的统计信息
        PlatformStatisticsByKyc preTwelvePlatformStatistics;
        List<PlatformStatisticsByKyc> prePlatformStatistics = getPrePlatformStatisticsByKyc(kycNaturalId, dateEnd, 12);
        if (prePlatformStatistics != null) {
            preTwelvePlatformStatistics = prePlatformStatistics.get(0);
        } else {
            preTwelvePlatformStatistics = null;
        }
        List<PlatformStatisticsByKyc> preOnePlatformStatisticsByKyc = getPrePlatformStatisticsByKyc(kycNaturalId, dateEnd, 1);
        // 近一个月入账金额环比增长率
        BigDecimal incomeChainRate;
        if (ObjectUtil.isNotEmpty(preOnePlatformStatisticsByKyc) && income.intValue() != 0) {
            incomeChainRate = income.subtract(preOnePlatformStatisticsByKyc.get(0).getIncome()).divide(income, 4, RoundingMode.HALF_UP);
        } else {
            incomeChainRate = new BigDecimal(0);
        }

        // 近一个月入账金额与前3个月平均入账金额比例
        BigDecimal incomeThreeRatio = calIncomeRatio(kycNaturalId, dateEnd, income, 3);
        // 近一个月入账金额与前6个月平均入账金额比例
        BigDecimal incomeSixRatio = calIncomeRatio(kycNaturalId, dateEnd, income, 6);

        // 入账金额同比增长率
        BigDecimal incomeGrowthRateOnYear;
        // 退款率同比增长率
        BigDecimal monthlyRefundGrowthRateOnYear;
        // 客单价同比增长率
        BigDecimal perCusTransGrowthRateOnYear;

        if (ObjectUtil.isEmpty(preTwelvePlatformStatistics)) {
            incomeGrowthRateOnYear = new BigDecimal(0);
            monthlyRefundGrowthRateOnYear = new BigDecimal(0);
            perCusTransGrowthRateOnYear = new BigDecimal(0);
        } else {
            incomeGrowthRateOnYear = getIncomeGrowthRateOnYear(income, preTwelvePlatformStatistics.getIncome());
            monthlyRefundGrowthRateOnYear = getMonthlyRefundGrowthRateOnYear(monthlyRefundRate, preTwelvePlatformStatistics.getMonthlyRefundRate());
            perCusTransGrowthRateOnYear = getPerCusTransGrowthRateOnYear(perCustomerTransaction, preTwelvePlatformStatistics.getPerCustomerTransaction());
        }


        // 统计时间
        PlatformStatisticsByKyc searchCondition = new PlatformStatisticsByKyc();
        Date statisticsTime = Date.from(dateEnd.toInstant());
        searchCondition.setKycNaturalId(kycNaturalId);
        searchCondition.setStatisticsTime(statisticsTime);
        List<PlatformStatisticsByKyc> platformStatisticsByKycs = platformStatisticsByKycMapper.select(searchCondition);
        Date time = Date.from(Instant.now());
        boolean isNew;
        PlatformStatisticsByKyc platformStatisticsByKyc;
        if (platformStatisticsByKycs.size() > 1) {
            log.error("数据库条目异常: kycNatualId: {}, statisticsTime: {}", kycNaturalId, statisticsTime);
            return;
        } else if (platformStatisticsByKycs.size() == 1) {
            isNew = false;
            platformStatisticsByKyc = platformStatisticsByKycs.get(0);
            platformStatisticsByKyc.setUpdateId("sys");
            platformStatisticsByKyc.setUpdateName("sys");
            platformStatisticsByKyc.setUpdateDts(time);
        } else if (platformStatisticsByKycs.size() == 0) {
            isNew = true;
            platformStatisticsByKyc = new PlatformStatisticsByKyc();
            AmaMWSCommonUtil.initBaseData(platformStatisticsByKyc);
        } else {
            log.error("结果异常: {}", platformStatisticsByKycs);
            return;
        }
        platformStatisticsByKyc.setKycNaturalId(kycNaturalId);
        platformStatisticsByKyc.setHalfYearSaleRatio(halfYearSaleRatio);
//        platformStatisticsByKyc.setAmazonusshopRatio(amazonusshopRatio);
//        platformStatisticsByKyc.setAmazonussaleRatio(amazonussaleRatio);
        platformStatisticsByKyc.setMonthlySale(monthlySale);
        platformStatisticsByKyc.setIncome(income);
        platformStatisticsByKyc.setIncomeChainRate(incomeChainRate);
        platformStatisticsByKyc.setMonthlyIncome(monthlyIncome);
        platformStatisticsByKyc.setIncomeThreeRatio(incomeThreeRatio);
        platformStatisticsByKyc.setIncomeSixRatio(incomeSixRatio);
        platformStatisticsByKyc.setIncomeGrowthRateOnYear(incomeGrowthRateOnYear);
        platformStatisticsByKyc.setPerCustomerTransaction(perCustomerTransaction);
        platformStatisticsByKyc.setPerCusTransGrowthRateOnYear(perCusTransGrowthRateOnYear);
        platformStatisticsByKyc.setMonthlyOrderNum(monthlyOrderNum);
        platformStatisticsByKyc.setMonthlyRefundRate(monthlyRefundRate);
        platformStatisticsByKyc.setMonthlyRefundGrowthRateOnYear(monthlyRefundGrowthRateOnYear);
        platformStatisticsByKyc.setAvailableProductNum(availableProductNum);
        platformStatisticsByKyc.setStatisticsTime(statisticsTime);
        if (isNew) {
            platformStatisticsByKycMapper.insertSelective(platformStatisticsByKyc);
        } else {
            platformStatisticsByKycMapper.updateByPrimaryKey(platformStatisticsByKyc);
        }
    }

    private List<JsSyncAmazonInventory> getJsSyncAmazonInventories(String kycNaturalId, ZonedDateTime dateEnd) {
        Example example = new Example(JsSyncAmazonInventory.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("userNaturalId", kycNaturalId);
        criteria.andEqualTo("statisticsTime", Date.from(dateEnd.toInstant()));
        return jsSyncAmazonInventoryMapper.selectByExample(example);
    }

    /**
     * 计算客单价同比增长率
     *
     * @param perCustomerTransaction 本月客单价
     * @param prePerCustomerTransaction 去年本月客单价
     */
    private BigDecimal getPerCusTransGrowthRateOnYear(BigDecimal perCustomerTransaction, BigDecimal prePerCustomerTransaction) {
        if (ObjectUtil.isEmpty(perCustomerTransaction) || ObjectUtil.isEmpty(prePerCustomerTransaction) || prePerCustomerTransaction.intValue() == 0) {
            return new BigDecimal(0);
        } else {
            return perCustomerTransaction.subtract(prePerCustomerTransaction).divide(prePerCustomerTransaction, 4, RoundingMode.HALF_UP);
        }
    }

    /**
     * 计算同比月退款率
     *
     * @param monthlyRefundRate 本月退款率
     * @param preMonthlyRefundRate 去年本月退款率
     */
    private BigDecimal getMonthlyRefundGrowthRateOnYear(BigDecimal monthlyRefundRate, BigDecimal preMonthlyRefundRate) {
        if (ObjectUtil.isEmpty(monthlyRefundRate) || ObjectUtil.isEmpty(preMonthlyRefundRate) || preMonthlyRefundRate.intValue() == 0) {
            return new BigDecimal(0);
        } else {
            return monthlyRefundRate.subtract(preMonthlyRefundRate).divide(preMonthlyRefundRate, 4, RoundingMode.HALF_UP);
        }
    }

    /**
     * 获取入账金额同比增长率
     *
     * @param income 本月的入账金额
     * @param preIncome 去年的入账金额
     *
     */
    private BigDecimal getIncomeGrowthRateOnYear(BigDecimal income, BigDecimal preIncome) {
        if (ObjectUtil.isEmpty(income) || ObjectUtil.isEmpty(preIncome) || preIncome.intValue() == 0) {
            return new BigDecimal(0);
        } else {
            return (income.subtract(preIncome)).divide(preIncome, 4, RoundingMode.HALF_UP);
        }
    }

    /**
     * 计算入账与前 n 个月平均入账所占比例。
     *
     * @param kycNaturalId
     * @param dateEnd
     * @param income
     * @param month 统计本月与前 month 的比例
     * @return
     */
    private BigDecimal calIncomeRatio(String kycNaturalId, ZonedDateTime dateEnd, BigDecimal income, int month) {
        List<PlatformStatisticsByKyc> prePlatformStatisticsByKycs = getPrePlatformStatisticsByKyc(kycNaturalId, dateEnd, month);
        if (ObjectUtil.isNotEmpty(prePlatformStatisticsByKycs)) {
            BigDecimal preMonthlySale = prePlatformStatisticsByKycs.stream()
                    .map(PlatformStatisticsByKyc::getIncome)
                    .reduce(new BigDecimal(0), BigDecimal::add).divide(new BigDecimal(month), 4, RoundingMode.HALF_UP);
            if (preMonthlySale.intValue() != 0) {
                return income.divide(preMonthlySale, 4, RoundingMode.HALF_UP);
            } else {
                return new BigDecimal(0);
            }
        } else {
            return new BigDecimal(0);
        }
    }

    /**
     * 根据所给日期，获取其前 n 个月的统计信息
     *
     * @param kycNaturalId
     * @param date
     * @param month 前几个月，不可以为 0
     */
    private List<PlatformStatisticsByKyc> getPrePlatformStatisticsByKyc(String kycNaturalId, ZonedDateTime date, int month) {
        Example example = new Example(PlatformStatisticsByKyc.class);
        example.setOrderByClause("statistics_time desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andLessThanOrEqualTo("statisticsTime", Date.from(date.minusMonths(1).toInstant()));
        criteria.andGreaterThanOrEqualTo("statisticsTime", Date.from(date.minusMonths(month).toInstant()));
        List<PlatformStatisticsByKyc> prePlatformStatisticsByKycs = platformStatisticsByKycMapper.selectByExample(example);
        if (prePlatformStatisticsByKycs.size() > month) {
            log.error("用户统计信息数据条目错误。用户 id: {}, 统计时间：{}", kycNaturalId, Date.from(date.minusMonths(1).toInstant()));
            return null;
        } else if (prePlatformStatisticsByKycs.size() == month) {
            return prePlatformStatisticsByKycs;
        } else {
            log.info("用户 id: {}, 数据不足{}个月，获取失败", kycNaturalId, month);
            return null;
        }
    }

    /**
     * 查出所相关店铺
     */
    private List<KycStore> getKycStores(String kycNaturalId) {
        Example example = new Example(KycStore.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        List<Integer> authStatusCondition = new ArrayList<>();
        authStatusCondition.add(StoreAuthStatusEnum.AUTH.ordinal());
        authStatusCondition.add(StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal());
        criteria.andCondition("auth_status in ", authStatusCondition);
        return kycStoreMapper.selectByExample(example);
    }


}
