package com.js.pay.utils;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringUtils;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @Description: 寻汇加密验签工具
 * @author liuh
 * @date 2018/5/18
 */
public class SignatureUtil {
    /**
     * UTF-8字符集
     **/
    private static final String CHARSET_UTF8 = "UTF-8";

    private static final String SIGN_TYPE_RSA = "RSA";
    private static final String SIGN_ALGORITHMS = "SHA1WithRSA";

    /**
     * sha256WithRsa 算法请求类型
     */
    private static final String SIGN_TYPE_RSA2 = "RSA2";
    private static final String SIGN_SHA256RSA_ALGORITHMS = "SHA256WithRSA";
    private static final String SIGN = "sign";

    /**
     * 获取签名值,默认utf-8 RSA2
     *
     * @param params     签名参数
     * @param privateKey 私钥
     * @return String
     */
    public static String rsaSign(Map<String, String> params, String privateKey) {
        return rsaSign(params, privateKey, CHARSET_UTF8, SIGN_TYPE_RSA2);
    }

    /**
     * 获取签名值
     *
     * @param params     签名参数
     * @param privateKey 私钥
     * @param charset    字符编码
     * @param signType   签名类型 RSA  RSA2
     * @return String
     */
    public static String rsaSign(Map<String, String> params, String privateKey, String charset, String signType) {
        String signContent = getSignContent(params);
        return rsaSign(signContent, privateKey, charset, signType);
    }


    /**
     * sha1WithRsa 加签
     *
     * @param content    签名内容
     * @param privateKey 私钥
     * @param charset    字符编码
     * @param signType   签名类型 RSA  RSA2
     * @return String
     */
    public static String rsaSign(String content, String privateKey, String charset, String signType) {
        String algorithm;
        if (SIGN_TYPE_RSA.equals(signType)) {
            algorithm = SIGN_ALGORITHMS;
        } else if (SIGN_TYPE_RSA2.equals(signType)) {
            algorithm = SIGN_SHA256RSA_ALGORITHMS;
        } else {
            throw new RuntimeException("Sign Type is Not Support : signType=" + signType);
        }

        try {
            PrivateKey priKey = getPrivateKeyFromPKCS8(SIGN_TYPE_RSA, new ByteArrayInputStream(privateKey.getBytes()));

            java.security.Signature signature = java.security.Signature.getInstance(algorithm);
            signature.initSign(priKey);

            if (StringUtils.isEmpty(charset)) {
                signature.update(content.getBytes());
            } else {
                signature.update(content.getBytes(charset));
            }

            byte[] signed = signature.sign();

            return new String(Base64.encodeBase64(signed));
        } catch (InvalidKeySpecException ie) {
            throw new RuntimeException("RSA私钥格式不正确，请检查是否正确配置了PKCS8格式的私钥", ie);
        } catch (Exception e) {
            throw new RuntimeException("RSAcontent = " + content + "; charset = " + charset, e);
        }
    }

    /**
     * 获取签名内容
     * 1、去除sign
     * 2、对参数key按照ASCII排序
     *
     * @param params 参数
     * @return String
     */
    private static String getSignContent(Map<String, String> params) {
        if (params == null) {
            return null;
        }
        params.remove(SIGN);
        StringBuilder content = new StringBuilder();
        List<String> keys = new ArrayList<>(params.keySet());
        Collections.sort(keys);

        for (int i = 0; i < keys.size(); i++) {
            String key = keys.get(i);
            String value = params.get(key);
            if (!StringUtils.isAnyBlank(key, value)) {
                content.append(i == 0 ? "" : "&").append(key).append("=").append(value);
            }
        }
        return content.toString();
    }

    private static PrivateKey getPrivateKeyFromPKCS8(String algorithm, InputStream ins) throws Exception {
        if (ins == null || StringUtils.isEmpty(algorithm)) {
            return null;
        }
        KeyFactory keyFactory = KeyFactory.getInstance(algorithm);
        byte[] encodedKey = StreamUtil.readText(ins).getBytes();
        encodedKey = Base64.decodeBase64(encodedKey);
        return keyFactory.generatePrivate(new PKCS8EncodedKeySpec(encodedKey));
    }
}
