package com.js.pay.utils;

import java.io.File;
import com.jcraft.jsch.Channel;
import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSch;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpException;
import lombok.extern.slf4j.Slf4j;

import com.google.common.base.Preconditions;
import com.jcraft.jsch.*;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

/**
 * @author liuh
 * @version V1.0 2019/05/24
 */
@Slf4j
public class SFTPUtil {
    private Session session;
    private ChannelSftp channelSftp;

    /**
     * 通过ssh连接远程主机，并开启sftp通道
     * @throws JSchException
     */
    public SFTPUtil(String host, int port, String userName, String password) throws JSchException {
        connect(host, port, userName, password);
    }

    private void connect(String host, int port, String userName, String password) throws JSchException {
        JSch jSch = new JSch();
        //采用指定的端口连接服务器
        session = jSch.getSession(userName,host,port);
        //如果服务器连接不上，则抛出异常
        Preconditions.checkNotNull(session, "SSH session is null.");
        //设置登陆主机的密码
        session.setPassword(password);
        //设置第一次登陆的时候提示，可选值：(ask | yes | no)
        session.setConfig("StrictHostKeyChecking", "no");
        //设置登陆超时时间
        session.connect(30000);
        channelSftp = (ChannelSftp) session.openChannel("sftp");
        channelSftp.connect(10000);
    }

    /**
     * 将单个本地文件上传至指定远程路径
     * @param localPath 本地路径
     * @param remotePath 指定远程路径
     * @throws SftpException
     */
    public void putSingle(String localPath, String remotePath,String remoteFileName) throws SftpException {
        Preconditions.checkNotNull(localPath, "本地路径不能为空！");
        File localFile = new File(localPath);
        Preconditions.checkArgument(localFile.exists() && !localFile.isDirectory(), "本地文件不存在或者本地文件是个文件夹");

        Preconditions.checkNotNull(remotePath, "远程路径不能为空！");

        // 远程路径中含有未被创建的目录时先创建目录
        createDir(remotePath);
        channelSftp.put(localPath, remoteFileName);
    }

    /**
     * 将单个本地文件上传至指定远程路径
     *
     * @param inputStream 输入流
     * @param remotePath 指定远程路径
     * @throws SftpException
     */
    public void putSingle(InputStream inputStream, String remotePath,String remoteFileName) throws SftpException {
        Preconditions.checkNotNull(inputStream, "输入流空！");

        Preconditions.checkNotNull(remotePath, "远程路径不能为空！");

        // 远程路径中含有未被创建的目录时先创建目录
        createDir(remotePath);
        channelSftp.put(inputStream, remoteFileName);
    }

    /**
     * 获取远程文件的输入流
     * @param remotePath 远程文件路径
     * @return 输入流
     * @throws SftpException
     */
    public InputStream get(String remotePath) throws SftpException {
        Preconditions.checkNotNull(remotePath, "远程路径不能为空");
        return channelSftp.get(remotePath);
    }

    /**
     * 创建目录
     */
    public boolean createDir(String remotePath){
        try{
            if (isDirExist(remotePath)){
                this.channelSftp.cd(remotePath);
                return true;
            }
            String pathArry[] = remotePath.split("/");
            StringBuffer filePath = new StringBuffer("/");
            for (String path : pathArry){
                if (path.equals("")){
                    continue;
                }
                filePath.append(path + "/");
                if (isDirExist(filePath.toString())){
                    channelSftp.cd(filePath.toString());
                }else{
                    // 建立目录
                    channelSftp.mkdir(filePath.toString());
                    // 进入并设置为当前目录
                    channelSftp.cd(filePath.toString());
                }

            }
            this.channelSftp.cd(remotePath);
            return true;
        }catch (SftpException e){
            e.printStackTrace();
        }
        return false;
    }

    /**
     * 判断目录是否存在
     * @param directory
     * @return
     */
    public boolean isDirExist(String directory){
        boolean isDirExistFlag = false;
        try{
            SftpATTRS sftpATTRS = channelSftp.lstat(directory);
            isDirExistFlag = true;
            return sftpATTRS.isDir();
        }catch (Exception e){
            if (e.getMessage().toLowerCase().equals("no such file")){
                isDirExistFlag = false;
            }
        }
        return isDirExistFlag;
    }


    /**
     * 获取指定远程目录下的文件名称集合, 如果远程路径是一个文件返回本身，如果远程目录不存在会抛出异常
     *
     * @param remotePath 远程目录
     * @return
     * @throws SftpException
     */
    public List<String> listFiles(String remotePath) throws SftpException {
        Preconditions.checkNotNull(remotePath, "远程路径不能为空");
        Vector<ChannelSftp.LsEntry> files = channelSftp.ls(remotePath);
        Iterator<ChannelSftp.LsEntry> iterator = files.iterator();
        List<String> fileNames = new ArrayList<>(files.size());
        while(iterator.hasNext()){
            ChannelSftp.LsEntry entry = iterator.next();
            String fileName = entry.getFilename();
            if((".").equals(fileName) || ("..").equals(fileName)){
                continue;
            }
            fileNames.add(fileName);
        }
        return fileNames;
    }

    /**
     * 删除指定路径的文件, 指定文件不能是目录
     * @param filePath 文件路径
     * @throws SftpException 没有权限删除或者指定文件是个目录
     */
    public void rm(String filePath) throws SftpException {
        Preconditions.checkNotNull(filePath, "指定路径不能为空");
        channelSftp.rm(filePath);
    }

    /**
     * 关闭连接
     */
    public void releaseConnection(){
        if(channelSftp != null){
            channelSftp.disconnect();
        }
        if(session != null){
            session.disconnect();
        }
    }
}
