package com.js.pay.service.impl;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.js.api.jspay.service.*;
import com.js.common.constant.CommonConstant;
import com.js.common.constant.Constant;
import com.js.common.enums.ResultEnum;
import com.js.common.enums.SiteEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.common.enums.StorePlateformEnum;
import com.js.common.model.req.*;
import com.js.common.model.vo.*;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.ResultUtil;
import com.js.dal.dao.mapper.KycStoreMapper;
import com.js.dal.dao.mapper.KycSunrateStoreMapper;
import com.js.dal.dao.model.KycStore;
import com.js.dal.dao.model.KycSunrateStore;
import com.js.dal.dao.service.impl.IBaseServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.config.annotation.Service;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.*;

/**
 * @Description:申请境外虚拟户-店铺记录具体实现
 * @Author: liuh
 * @Create: 2019-05-21
 **/
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
@Slf4j
public class ApiKycStoreServiceImpl extends IBaseServiceImpl<KycStore> implements ApiKycStoreService {
    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    ApiKycSunrateService apiKycSunrateService;
    @Autowired
    ApiKycSunrateStoreService apiKycSunrateStoreService;
    @Autowired
    ApiSunrateTradePriceService apiSunrateTradePriceService;
    @Autowired
    KycSunrateStoreMapper kycSunrateStoreMapper;
    @Autowired
    ApiJsKycChargeService apiJsKycChargeService;

    @Override
    public ResponseMessage addStore(KycStoreReq kycStoreReq, KycNaturalVO kycNaturalVO) {
        log.info("申请虚拟户-添加店铺-开始：{}", JSON.toJSONString(kycStoreReq));
        if (!kycStoreReq.getStoreSituation()) {
            KycStore kycStoreQuery = new KycStore();
            kycStoreQuery.setKycNaturalId(kycNaturalVO.getId());
            kycStoreQuery.setStoreSituation(kycStoreReq.getStoreSituation());
            List<KycStore> kycStoreList = iBaseMapper.select(kycStoreQuery);
            if (CollectionUtils.isNotEmpty(kycStoreList) && kycStoreList.size() > CommonConstant.NO_STORE_MAX_APPLY_COUNT) {
                log.info("暂无店铺申请数量已经超出限制，不予再次申请");
                return ResultUtil.error(ResultEnum.ALREADY_TO_MAX_APPLY_COUNT);
            }
        }
        if (kycStoreReq.getStoreSituation()) {
            if (checkAuthToken(kycStoreReq.getSellId(), kycStoreReq.getMwsAuthToken(), kycStoreReq.getStoreTheSite().name(), null)) {
                return ResultUtil.error(ResultEnum.STORE_TOKEN_DUPLICATE);
            }
        }
        KycStore kycStore = new KycStore();
        BeanUtils.copyProperties(kycStoreReq, kycStore);
        kycStore.setId(IdUtil.simpleUUID());
        kycStore.setStorePlateformCode(kycStoreReq.getStorePlateform().name());
        kycStore.setStorePlateformName(kycStoreReq.getStorePlateform().getDesc());
        kycStore.setStoreTheSiteCode(kycStoreReq.getStoreTheSite().name());
        //授权状态如果为已有店铺则变更为：已经授权但没有确认第一笔回款，否则改为 未授权状态
        kycStore.setAuthStatus(kycStoreReq.getStoreSituation() ? StoreAuthStatusEnum.AUTH.ordinal() : StoreAuthStatusEnum.UN_AUTH.ordinal());
        kycStore.setEnable(true);
        kycStore.setStoreTheSiteName(kycStoreReq.getStoreTheSite().getSiteName());
        kycStore.setKycNaturalId(kycNaturalVO.getId());
        kycStore.setCreateId(kycNaturalVO.getId());
        kycStore.setCreateName(kycNaturalVO.getCertificationName());
        super.insertSelective(kycStore);
        log.info("申请虚拟户-添加店铺-结束");
        return ResultUtil.success(kycStore.getId());
    }

    /**
     * 校检店铺 token 是否重复
     *
     * 同一大区下。店铺之间的授权 token 不允许重复
     * @author liutianyu
     * @return true 重复，false 不重复
     */
    private Boolean checkAuthToken(String sellId, String  mwsAuthToken, String storeTheSiteCode, String id) {
        Example example = new Example(KycStore.class);
        Example.Criteria criteria = example.createCriteria();
        if (ObjectUtil.isNotEmpty(id)) {
            criteria.andNotEqualTo("id", id);
        }
        criteria.andEqualTo("delFlag", false);
        criteria.andEqualTo("sellId", sellId);
        criteria.andEqualTo("mwsAuthToken", mwsAuthToken);
        criteria.andEqualTo("storeTheSiteCode", storeTheSiteCode);
        List<KycStore> kycStores = iBaseMapper.selectByExample(example);
        if (kycStores.size() > 0) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public ResponseMessage storeList(KycShowStoreReq kycShowStoreReq, KycNaturalVO kycNaturalVO) {
        log.info("js店铺列表查询开始 -> 当前登录用户ID：{},入参：{}",kycNaturalVO.getId(),JSON.toJSONString(kycShowStoreReq));
        kycShowStoreReq.setKycNaturalId(kycNaturalVO.getId());
        kycShowStoreReq.setDelFlag(false);
        Page page = PageHelper.startPage(kycShowStoreReq.getStartPage(),kycShowStoreReq.getPageSize());
        List<KycStoreVO> kycStoreVOList = kycStoreMapper.queryStoreList(kycShowStoreReq);
        PageInfo<KycStoreVO> pageInfo = page.toPageInfo();
        log.info("js店铺列表查询结束");
        return ResultUtil.success(pageInfo);
    }

    @Override
    public ResponseMessage queryStoreListByCurr(KycShowStoreReq kycShowStoreReq){
        log.info("js店铺列表查询开始 -> 当前登录用户ID：{},入参：{}",kycShowStoreReq.getKycNaturalId(),JSON.toJSONString(kycShowStoreReq));
        kycShowStoreReq.setKycNaturalId(kycShowStoreReq.getKycNaturalId());
        kycShowStoreReq.setDelFlag(false);
        Page page = PageHelper.startPage(kycShowStoreReq.getStartPage(),kycShowStoreReq.getPageSize());
        List<KycStoreVO> kycStoreVOList = kycStoreMapper.queryStoreListByCurr(kycShowStoreReq);
        PageInfo<KycStoreVO> pageInfo = page.toPageInfo();
        log.info("js店铺列表查询结束");
        return ResultUtil.success(pageInfo);
    }

    @Override
    public ResponseMessage storeListNoPage(KycStoreReq kycStoreReq,KycNaturalVO kycNaturalVO) {
        log.info("js店铺列表查询开始 -> 当前登录用户ID：{},查询条件:{}",kycNaturalVO.getId(),kycStoreReq);
        KycShowStoreReq kycShowStoreReq = new KycShowStoreReq();
        if(null != kycStoreReq.getStorePlateform()){
            kycShowStoreReq.setStorePlateform(kycStoreReq.getStorePlateform());
        }
        if(null != kycStoreReq.getStoreTheSite()){
            kycShowStoreReq.setStoreTheSite(kycStoreReq.getStoreTheSite());
        }
        kycShowStoreReq.setKycNaturalId(kycNaturalVO.getId());
        kycShowStoreReq.setDelFlag(false);
        List<KycStoreVO> kycStoreVOList = kycStoreMapper.queryStoreList(kycShowStoreReq);
        log.info("js店铺列表查询结束");
        return ResultUtil.success(kycStoreVOList);
    }

    @Override
    public ResponseMessage findApplyCountNum(String browserIP,KycNaturalVO kycNaturalVO) {
        log.info("获取当前用户站点申请账户数量开始 -> 当前登录用户ID：{}",kycNaturalVO.getId());
        List<KycShowApplyVO> mapList = kycStoreMapper.findApplyCountNum(kycNaturalVO.getId());
        if(CollectionUtils.isEmpty(mapList)){
            log.info("未查找到当前用户申请的店铺信息");
            return ResultUtil.success(ResultEnum.SUCCESS);
        }
        String orgCode = apiSunrateTradePriceService.findOrgCode(kycNaturalVO.getId(),kycNaturalVO.getRegisterType());
        if(StringUtils.isBlank(orgCode)){
            log.info("获取当前用户orgCode不存在:{}",orgCode);
            return ResultUtil.success(ResultEnum.SUCCESS);
        }
        JsKycChargeReq jsKycChargeRateReq = new JsKycChargeReq();
        jsKycChargeRateReq.setKycNaturalId(kycNaturalVO.getId());
        jsKycChargeRateReq.setChargeType("RATE");
        JsKycChargeVO jsKycChargeRateVO = apiJsKycChargeService.findKycChargeInfo(jsKycChargeRateReq);

        List<JsPaySunrateTradeQueryPriceVo> queryPriceList = new ArrayList<>();
        mapList.stream().forEach(kycShowApplyVO -> {
            SiteEnum storeTheSite = kycShowApplyVO.getSiteName();
            //汇差标准
            BigDecimal rate = null == jsKycChargeRateVO ? storeTheSite.getJsTradeRateValue() : jsKycChargeRateVO.getChargeScale();
            log.info("客户币种{}，公共汇差：{}，执行汇差：{}",storeTheSite.getCurrency(),storeTheSite.getJsTradeRateValue(),rate);
            kycShowApplyVO.setAccountTotalAmt(null == kycShowApplyVO.getAccountTotalAmt() ? new BigDecimal(0) : kycShowApplyVO.getAccountTotalAmt());
            JsPaySunrateTradeQueryPriceVo jsPaySunrateTradeQueryPriceVo = xhQueryPrice(kycShowApplyVO, browserIP, orgCode);
            if(null != kycShowApplyVO.getAccountTotalAmt()){
                kycShowApplyVO.setCurrency(kycShowApplyVO.getSiteName().getCurrency());
                kycShowApplyVO.setCurrencySymbol(kycShowApplyVO.getSiteName().getCurrencySymbol());
                kycShowApplyVO.setAccountCNYTotalAmt(new BigDecimal(0));
            }
            if(null != jsPaySunrateTradeQueryPriceVo){
                kycShowApplyVO.setQueryPriceFlag(true);
                BigDecimal queryRate = jsPaySunrateTradeQueryPriceVo.getQueryRate();
                BigDecimal exeRate = queryRate.subtract(rate).setScale(4,BigDecimal.ROUND_HALF_UP);
                log.info("客户询价汇率：{},执行汇率：{}",queryRate,exeRate);
                jsPaySunrateTradeQueryPriceVo.setQueryRate(exeRate);
                kycShowApplyVO.setAccountCNYTotalAmt(kycShowApplyVO.getAccountTotalAmt().multiply(exeRate).setScale(2,BigDecimal.ROUND_HALF_UP));
                log.info("换汇后总金额：{},扣除汇差后总金额：{}",kycShowApplyVO.getAccountTotalAmt().multiply(jsPaySunrateTradeQueryPriceVo.getQueryRate()).setScale(2,BigDecimal.ROUND_HALF_UP),kycShowApplyVO.getAccountCNYTotalAmt());

                BigDecimal buyAmt = jsPaySunrateTradeQueryPriceVo.getBuyAmt();
                BigDecimal exeAmt = jsPaySunrateTradeQueryPriceVo.getSellAmt().multiply(exeRate).setScale(2,BigDecimal.ROUND_HALF_UP);
                log.info("执行汇差前 ：{} 兑换 ：{}。执行汇差后 ：{} 兑换 ：{}。",jsPaySunrateTradeQueryPriceVo.getSellAmt(),buyAmt,jsPaySunrateTradeQueryPriceVo.getSellAmt(),exeAmt);
                jsPaySunrateTradeQueryPriceVo.setBuyAmt(exeAmt);
                queryPriceList.add(jsPaySunrateTradeQueryPriceVo);
            }else{
                kycShowApplyVO.setQueryPriceFlag(false);
            }
        });

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("acctList",mapList);
        jsonObject.put("priceList",queryPriceList);
        log.info("获取当前用户站点申请账户数量及汇率信息结束 -> 查询结果：{}", jsonObject);
        return ResultUtil.success(jsonObject);
    }

    @Override
    public KycStoreVO findKycStoreByPrimaryKey(String kycStoreId, KycNaturalVO kycNaturalVO) {
        KycStore kycStoreQuery = new KycStore();
        kycStoreQuery.setId(kycStoreId);
        kycStoreQuery.setKycNaturalId(kycNaturalVO.getId());
        KycStore kycStore = iBaseMapper.selectOne(kycStoreQuery);
        if(null == kycStore){
            return null;
        }
        log.info("查询到店铺信息;{}", JSON.toJSONString(kycStore));
        KycStoreVO kycStoreVO = new KycStoreVO();
        kycStoreVO.setStoreId(kycStore.getId());
        kycStoreVO.setStoreName(kycStore.getStoreName());
        kycStoreVO.setStoreUrl(kycStore.getStoreUrl());
        kycStoreVO.setStorePlateformCode(kycStore.getStorePlateformCode());
        kycStoreVO.setStorePlateformName(kycStore.getStorePlateformName());
        kycStoreVO.setStoreTheSiteCode(kycStore.getStoreTheSiteCode());
        kycStoreVO.setStoreTheSiteName(kycStore.getStoreTheSiteName());
        kycStoreVO.setCurrencySymbol(SiteEnum.valueOf(kycStore.getStoreTheSiteCode()).getCurrencySymbol());//币种符号
        kycStoreVO.setAuthStatus(kycStore.getAuthStatus());//店铺授权状态
        return kycStoreVO;
    }

    @Override
    public ResponseMessage editStore(KycStoreEditReq kycStoreEditReq, KycNaturalVO kycNaturalVO) {
        log.info("编辑店铺信息开始：{}", JSON.toJSONString(kycStoreEditReq));
        KycStore kycStoreQuery = new KycStore();
        kycStoreQuery.setId(kycStoreEditReq.getKycStoreId());
        kycStoreQuery.setKycNaturalId(kycNaturalVO.getId());
        KycStore kycStore = iBaseMapper.selectOne(kycStoreQuery);
        if(null == kycStore){
            log.info("编辑保存店铺信息未查询到店铺信息");
            return ResultUtil.error(ResultEnum.MESSAGE_NOT_EXIST);
        }
        if (checkAuthToken(kycStoreEditReq.getSellId(), kycStoreEditReq.getMwsAuthToken(),
                kycStoreEditReq.getStoreTheSite().name(), kycStoreEditReq.getKycStoreId())) {
            return ResultUtil.error(ResultEnum.STORE_TOKEN_DUPLICATE);
        }
        kycStore.setStoreSituation(true);
        kycStore.setStoreName(kycStoreEditReq.getStoreName());
        kycStore.setStoreUrl(kycStoreEditReq.getStoreUrl());
        kycStore.setSellId(kycStoreEditReq.getSellId());
        kycStore.setMwsAuthToken(kycStoreEditReq.getMwsAuthToken());
        kycStore.setAuthStatus(StoreAuthStatusEnum.AUTH.ordinal());
        kycStore.setUpdateId(kycNaturalVO.getId());
        kycStore.setUpdateName(kycNaturalVO.getCertificationName());
        int updateCount = iBaseMapper.updateByPrimaryKeySelective(kycStore);
        log.info("编辑保存店铺信息结束:{}",updateCount);
        return ResultUtil.success(ResultEnum.SUCCESS);
    }

    @Override
    public ResponseMessage showWithdrawCharge(KycNaturalVO kycNaturalVO) {
        List<Map> mapList = kycStoreMapper.showWithdrawCharge(kycNaturalVO.getId());
        DecimalFormat df = new DecimalFormat("0.00%");
        List<Map<String,Object>> resultMapList = new ArrayList<>();
        String oldStorePlateformCode = "";
        Map<String,Object> resultMap = new HashMap<>();
        for(int i = 0 ; i < mapList.size(); i ++){
            Map<String,Object> map = mapList.get(i);
            String selStorePlateformCode = map.get("storePlateformCode").toString();
            StorePlateformEnum storePlateformEnum = StorePlateformEnum.valueOf(selStorePlateformCode);
            SiteEnum siteEnum = SiteEnum.valueOf(map.get("storeTheSiteCode").toString());
            String chargeFormat = null == map.get("chargeScale") ? df.format(siteEnum.getJsTradeServiceChargeValue()) : df.format(map.get("chargeScale"));
            if(selStorePlateformCode.equals(oldStorePlateformCode)){
                resultMap.put("storeTheSiteList",resultMap.get("storeTheSiteList") + " " + siteEnum.getSiteName() + ":" + (null == map.get("chargeScale") ? df.format(siteEnum.getJsTradeServiceChargeValue()) : df.format(new BigDecimal(((Number)map.get("chargeScale")).doubleValue()))));
            }else{
                oldStorePlateformCode = selStorePlateformCode;
                if(i > 0){
                    resultMapList.add(resultMap);
                }
                resultMap = new HashMap<>();
                resultMap.put("storePlateformCode",selStorePlateformCode);
                resultMap.put("storePlateformDesc",storePlateformEnum.getDesc());
                resultMap.put("storeTheSiteList",siteEnum.getSiteName() + ":" + chargeFormat);
            }
            if(i == mapList.size() - 1){
                resultMapList.add(resultMap);
            }
        }
        return ResultUtil.success(resultMapList);
    }

    @Override
    public ResponseMessage findApplySite(KycNaturalVO kycNaturalVO) {
        log.info("获取当前用户已申请站点信息 -> 当前登录用户ID：{}",kycNaturalVO.getId());
        String orgCode = apiSunrateTradePriceService.findOrgCode(kycNaturalVO.getId(),kycNaturalVO.getRegisterType());
        if(StringUtils.isBlank(orgCode)){
            log.info("获取当前用户orgCode不存在:{}",orgCode);
            return ResultUtil.success(ResultEnum.SUCCESS);
        }
        List<KycShowApplyVO> mapList = kycStoreMapper.findApplyCountNum(kycNaturalVO.getId());
        if(CollectionUtils.isEmpty(mapList)){
            log.info("未查找到当前用户申请的店铺信息");
            return ResultUtil.success(ResultEnum.SUCCESS);
        }
        mapList.stream().forEach(kycShowApplyVO -> {
            SiteEnum storeTheSite = kycShowApplyVO.getSiteName();
            StorePlateformEnum storePlateform = kycShowApplyVO.getStorePlateform();
            kycShowApplyVO.setPlateformAndSite((null == storePlateform ? "" : storePlateform.getDesc()) + (null == storeTheSite ? "" : storeTheSite.getSiteName()));
        });
        log.info("获取当前用户已申请站点信息结束 -> 查询结果：{}", mapList);
        return ResultUtil.success(mapList);
    }

    private JsPaySunrateTradeQueryPriceVo xhQueryPrice(KycShowApplyVO kycShowApplyVO, String browserIP, String orgCode){
        SunrateTradePriceReq sunrateTradePriceReq = new SunrateTradePriceReq();
        sunrateTradePriceReq.setOrgCode(orgCode);
        sunrateTradePriceReq.setBuyCur(CommonConstant.TRADE_CURRENCY_CNH);//离岸人民币币种
        sunrateTradePriceReq.setSellCur(kycShowApplyVO.getSiteName().getCurrency());
        sunrateTradePriceReq.setTxnMode(CommonConstant.TXN_MODE_SELL);
        sunrateTradePriceReq.setTxnAmt(new BigDecimal(1));
        sunrateTradePriceReq.setBrowserIP(browserIP);
        ResponseMessage responseMessage = apiSunrateTradePriceService.xhTradeQueryPrice(sunrateTradePriceReq);
        log.info("寻汇-询价结束-返回结果:{},描述信息:{}",responseMessage.isSuccess(),responseMessage.getMsg());
        if(responseMessage.isSuccess()){
            JSONObject jsonObject = JSONObject.parseObject(JSON.toJSONString(responseMessage.getData()));
            return jsonObject.toJavaObject(JsPaySunrateTradeQueryPriceVo.class);
        }
        log.info("询价结果未成功，返回空值");
        return null;
    }
}
