package com.js.loan.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONObject;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.js.api.jsloan.service.ApiStatisticsInfoService;
import com.js.common.constant.Constant;
import com.js.common.enums.IdCardAreaCodeEnum;
import com.js.common.enums.ResultEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.common.model.req.JsLoanStatisticsInfoReq;
import com.js.common.model.vo.JsLoanStatisticsInfoKycVO;
import com.js.common.model.vo.JsLoanStatisticsInfoStoreVO;
import com.js.common.model.vo.JsLoanStatisticsInfoVO;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.ResultUtil;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.ibatis.session.RowBounds;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class ApiStatisticsInfoServiceImpl implements ApiStatisticsInfoService {

    @Autowired
    KycStoreMapper kycStoreMapper;
//    @Autowired
//    KycNaturalMapper kycNaturalMapper;
    @Autowired
    KycCertiHKCompanyMapper kycCertiHKCompanyMapper;
    @Autowired
    KycCertiHKPersonalMapper kycCertiHKPersonalMapper;
    @Autowired
    KycCertiMainLandCompanyMapper kycCertiMainLandCompanyMapper;
    @Autowired
    KycCertiMainLandPersonalMapper kycCertiMainLandPersonalMapper;
    @Autowired
    PlatformStatisticsByKycMapper platformStatisticsByKycMapper;
    @Autowired
    PlatformStatisticsByStoreMapper platformStatisticsByStoreMapper;
    @Autowired
    PlatformAmazonRiskQuotaMapper platformAmazonRiskQuotaMapper;
    @Autowired
    JsLoanRiskInfoMapper jsLoanRiskInfoMapper;

    @Override
    public ResponseMessage getBaseInfo(String kycNaturalId) {
        // 因为业务原因，目前大陆个人、或有大陆个人身份的人才会有信息。
        // 获取用户注册信息，用户信息直接从准入那里获取。
        Example example = new Example(JsLoanRiskInfo.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andEqualTo("delFlag", false);
        JsLoanRiskInfo jsLoanRiskInfo = jsLoanRiskInfoMapper.selectOneByExample(example);
        if (ObjectUtil.isNull(jsLoanRiskInfo)) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        }

        // 借款人的身份证号
        String idNo = jsLoanRiskInfo.getBorrowerIdcard();
        long age = calAge(idNo);
        // 1.已婚 2.未婚 3.离异 4.再婚
        String marriageStatus = jsLoanRiskInfo.getMarriageStatus();
        IdCardAreaCodeEnum area = IdCardAreaCodeEnum.getArea(idNo.substring(0, 4));
        // 是否是本地
        Boolean isLocal;
        String idProvince;
        String idCity;
        if (ObjectUtil.isNotNull(area)) {
            idProvince = area.getProvince();
            idCity = area.getCity();
        } else {
            idProvince = null;
            idCity = null;
        }
        String bankProvince = jsLoanRiskInfo.getCorpBankProvinceName();
        String bankCity = jsLoanRiskInfo.getCorpBankCityName();
        if (bankProvince.contains(idProvince)) {
            if (bankCity.contains(idCity)) {
                isLocal = true;
            } else {
                // 如果城市匹配失败了，尝试去掉一些尾缀再次匹配。
                if (idCity.length() > 2) {
                    if (idCity.endsWith("州")) {
                        idCity = idCity.substring(0, idCity.length() - 1);
                        if (bankCity.contains(idCity)) {
                            isLocal = true;
                        } else {
                            isLocal = false;
                        }
                    } else {
                        isLocal = false;
                    }
                } else {
                    isLocal = false;
                }
            }
        } else {
            isLocal = false;
        }
        Map<String ,Object> result = new HashMap<>();
        result.put("js-1001", age);
        result.put("js-1002", marriageStatus);
        result.put("js-1003", isLocal);
        result.put("js-1004", idProvince);
        result.put("js-1007", idCity);
        return ResultUtil.success(result, ResultEnum.QUERY_SUCCESS);
    }

    @Override
    public ResponseMessage getPlatformStatisticsInfo(String kycNaturalId) {
        KycStore searchCondition = new KycStore();
        searchCondition.setKycNaturalId(kycNaturalId);
        List<KycStore> kycStores = kycStoreMapper.select(searchCondition);
        JsLoanStatisticsInfoVO result = new JsLoanStatisticsInfoVO();
        if (kycStores.size() > 0) {
            BigDecimal storeNum = new BigDecimal(kycStores.size());
            BigDecimal cancelAuthNum = new BigDecimal(kycStores.stream()
                    .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.CANCEL_AUTH.ordinal())
                    .count());
            // 取消授权店铺占比
            BigDecimal cancelAuthRatio = cancelAuthNum.divide(storeNum, 4, RoundingMode.HALF_UP);
            //关闭店铺占比 TODO 暂时提供不了。
//            BigDecimal shutdownRatio;

            BigDecimal authAndAuthWithAccount = new BigDecimal(kycStores.stream()
                            .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.AUTH.ordinal()
                                    || e.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal())
                            .count());
            BigDecimal authAndAuthWithAccountRatio;
            BigDecimal authWithAccountNum;
            if (authAndAuthWithAccount.intValue() == 0) {
                authAndAuthWithAccountRatio = new BigDecimal(0);
                authWithAccountNum = new BigDecimal(0);
            } else {
                authWithAccountNum = new BigDecimal(kycStores.stream()
                        .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal())
                        .count());
                authAndAuthWithAccountRatio = authWithAccountNum.divide(authAndAuthWithAccount, 4, RoundingMode.HALF_UP);
            }
            result.setCancelAuthRatio(cancelAuthRatio);
            result.setCancelAuthNum(cancelAuthNum);
//            result.setShutdownRatio(shutdownRatio);
            result.setStoreNum(storeNum);
            result.setAuthAndAuthWithAccount(authAndAuthWithAccount);
            result.setAuthWithAccountNum(authWithAccountNum);
            result.setAuthAndAuthWithAccountRatio(authAndAuthWithAccountRatio);
        }

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("storeNum", result.getStoreNum().intValue());
        jsonObject.put("unAuthStoreNum", result.getStoreNum().subtract(result.getAuthAndAuthWithAccount()).intValue());
        jsonObject.put("authStoreNum", result.getAuthAndAuthWithAccount().intValue());
        jsonObject.put("unAuthWithAccountStoreNum", result.getStoreNum().subtract(result.getAuthWithAccountNum()).intValue());
        jsonObject.put("authWithAccountStoreNum", result.getAuthWithAccountNum().intValue());
        //jsonObject.put("js-4001", result.getAuthAndAuthWithAccount());
        //jsonObject.put("js-4006", result.getAuthAndAuthWithAccountRatio());
        //jsonObject.put("cancelAuthRatio", result.getCancelAuthRatio());
        //jsonObject.put("shutdownRatio", result.getShutdownRatio());
        //jsonObject.put("cancelAuthNum", result.getCancelAuthNum());

        return ResultUtil.success(jsonObject, ResultEnum.QUERY_SUCCESS);
    }

    @Override
    public ResponseMessage getAmazonRiskQuota(String kycNaturalId) {
        Example example = new Example(PlatformAmazonRiskQuota.class);
        example.setOrderByClause("statistics_time desc");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        List<PlatformAmazonRiskQuota> platformAmazonRiskQuotas = platformAmazonRiskQuotaMapper.selectByExampleAndRowBounds(example, new RowBounds(0, 1));
        if (platformAmazonRiskQuotas.size() > 0) {
            PlatformAmazonRiskQuota platformAmazonRiskQuota = platformAmazonRiskQuotas.get(0);
            JSONObject jsonObject = new JSONObject();
            jsonObject.put("js-4001", platformAmazonRiskQuota.getAuthAndAuthWithAccountNum());
            jsonObject.put("js-4002", platformAmazonRiskQuota.getAmazonUsSaleRatio());
            jsonObject.put("js-4003", platformAmazonRiskQuota.getMonthlySale());
            jsonObject.put("js-4004", platformAmazonRiskQuota.getRefundChainRate());
            jsonObject.put("js-4005", platformAmazonRiskQuota.getAvailableProductNumChainRate());
            jsonObject.put("js-4006", platformAmazonRiskQuota.getAuthWithAccountAndAuthRatio());
            jsonObject.put("js-4007", platformAmazonRiskQuota.getSalesTopTenRatio());
            jsonObject.put("js-4008", platformAmazonRiskQuota.getSalesTopThreeRatio());
            jsonObject.put("js-4009", platformAmazonRiskQuota.getMonthlyIncome());
            jsonObject.put("js-4010", platformAmazonRiskQuota.getBindTime());
            jsonObject.put("js-4011", platformAmazonRiskQuota.getIncomeChainRate());
            jsonObject.put("js-4012", platformAmazonRiskQuota.getIncomeThreeRatio());
            jsonObject.put("js-4013", platformAmazonRiskQuota.getIncomeSixRatio());
            jsonObject.put("js-4014", platformAmazonRiskQuota.getIncomeGrowthRateOnYear());
            jsonObject.put("js-4015", true); // TODO 待定 by liuty
            jsonObject.put("js-4016", platformAmazonRiskQuota.getAverageIncome());
            return ResultUtil.success(jsonObject, ResultEnum.QUERY_SUCCESS);
        } else {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        }
    }

    @Override
    public ResponseMessage getStoreList(JsLoanStatisticsInfoReq jsLoanStatisticsInfoReq) {
        PageHelper.startPage(jsLoanStatisticsInfoReq);
        List<JsLoanStatisticsInfoStoreVO> storeList = platformStatisticsByStoreMapper.getStoreList(jsLoanStatisticsInfoReq);
        convertUnit(storeList);
        PageInfo<JsLoanStatisticsInfoStoreVO> pageInfo = new PageInfo<>(storeList);
        return ResultUtil.success(pageInfo, ResultEnum.QUERY_SUCCESS);
    }

    /**
     * 把总月均销售额、笔均入账金额、月均退款额 单位换算成万元
     */
    private void convertUnit(List<JsLoanStatisticsInfoStoreVO> storeList) {
        BigDecimal unit = new BigDecimal("10000");
        for (JsLoanStatisticsInfoStoreVO vo : storeList) {
            vo.setMonthlySale(vo.getMonthlySale().divide(unit, 4, RoundingMode.HALF_UP));
            vo.setAverageIncome(vo.getAverageIncome().divide(unit, 4, RoundingMode.HALF_UP));
            vo.setMonthlyRefundAmount(vo.getMonthlyRefundAmount().divide(unit, 4, RoundingMode.HALF_UP));
        }
    }

    @Override
    public ResponseMessage getStore(String storeId) {
        JsLoanStatisticsInfoReq jsLoanStatisticsInfoReq = new JsLoanStatisticsInfoReq();
        jsLoanStatisticsInfoReq.setStoreId(storeId);
        List<JsLoanStatisticsInfoStoreVO> storeList = platformStatisticsByStoreMapper.getStoreList(jsLoanStatisticsInfoReq);
        if (storeList.size() == 1) {
            convertUnit(storeList);
            return ResultUtil.success(storeList.get(0), ResultEnum.QUERY_SUCCESS);
        } else if (storeList.size() == 0) {
            return ResultUtil.success(ResultEnum.QUERY_SUCCESS);
        } else {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        }
    }

    @Override
    public ResponseMessage getKycHis(String kycNaturalId) {
        ZonedDateTime now = ZonedDateTime.now();
        int dayOfMonth = now.getDayOfMonth();
        ZonedDateTime start;
        if (dayOfMonth >= 20){
            // 计算当月和前11个月的数据，一共12个数据点。
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(11);
        } else {
            // 计算当月和前12个月的数据，一共13个数据点
            start = now.withDayOfMonth(1).withHour(0).withMinute(0).withSecond(0).withNano(0).minusMonths(12);
        }
        Example example = new Example(PlatformStatisticsByKyc.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andGreaterThanOrEqualTo("statisticsTime", Date.from(start.toInstant()));
        criteria.andLessThanOrEqualTo("statisticsTime", Date.from(now.toInstant()));

        // 按月分组，取每月中最后的一天指标（当月以当天为准）
        List<PlatformStatisticsByKyc> platformStatisticsByKycs = platformStatisticsByKycMapper.selectByExample(example);
        Map<String, PlatformStatisticsByKyc> map = platformStatisticsByKycs.stream()
                .collect(Collectors.groupingBy(e -> DateUtil.format(e.getStatisticsTime(), "yyyy-MM"),
                        Collectors.collectingAndThen(Collectors.maxBy(Comparator.comparing(PlatformStatisticsByKyc::getStatisticsTime)),
                                Optional::get)
                ));
        List<JsLoanStatisticsInfoKycVO> result = new ArrayList<>();
        // 转换数据结构
        for (Map.Entry<String, PlatformStatisticsByKyc> entry : map.entrySet()) {
            JsLoanStatisticsInfoKycVO jsLoanStatisticsInfoKycVO = new JsLoanStatisticsInfoKycVO();
            BeanUtil.copyProperties(entry.getValue(), jsLoanStatisticsInfoKycVO);
            jsLoanStatisticsInfoKycVO.setMonth(entry.getKey());
            result.add(jsLoanStatisticsInfoKycVO);
        }
        // 按时间排序。
        result = result.stream()
                .sorted(Comparator.comparing(e -> DateUtil.parse(e.getMonth(), "yyyy-MM")))
                .collect(Collectors.toList());
        return ResultUtil.success(result, ResultEnum.QUERY_SUCCESS);
    }

    /**
     * 计算身份证有效期
     */
    private long getIdEffectiveTime(Date startTime, Date endDate) {
        LocalDateTime effectiveStartTime = LocalDateTime.ofInstant(startTime.toInstant(), ZoneId.systemDefault());
        LocalDateTime effectiveEndTime = LocalDateTime.ofInstant(endDate.toInstant(), ZoneId.systemDefault());
        return effectiveStartTime.until(effectiveEndTime, ChronoUnit.MONTHS);
    }

    /**
     * 通过身份证计算年龄
     */
    private long calAge(String idNo) {
        String birth = idNo.substring(6, 14);
        LocalDate birthDay = LocalDate.parse(birth, DateTimeFormatter.BASIC_ISO_DATE);
        return birthDay.until(LocalDate.now(), ChronoUnit.YEARS);
    }

    /**
     * 通过身份计算性别(偶数女，奇数男)
     */
    private int calSex(String idNo) {
        return Integer.parseInt(idNo.substring(16, 17)) % 2;
    }

}
