package com.js.loan.service.qichacha.impl;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.crypto.SecureUtil;
import cn.hutool.http.HttpRequest;
import cn.hutool.http.HttpResponse;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.base.IBaseMapper;
import com.js.api.jsloan.service.qichacha.QiChaChaEnterpriseInfoService;
import com.js.common.JsException.LogicException;
import com.js.common.constant.Constant;
import com.js.common.enums.ResultEnum;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.ResultUtil;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.*;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.BiConsumer;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class QiChaChaEnterpriseInfoServiceImpl implements QiChaChaEnterpriseInfoService {

    @Value("${qichacha.appKey}")
    private String appKey;

    @Value("${qichacha.secretKey}")
    private String secretKey;

    private final String FULL_DETAILS_BY_NAME_URL = "http://api.qichacha.com/ECIV4/GetFullDetailsByName";
    private final String HISTORYT_ECI_URL = "http://api.qichacha.com/History/GetHistorytEci";
    private final String HISTORYT_SHARE_HOLDER_URL = "http://api.qichacha.com/History/GetHistorytShareHolder";
    private final String INVESTMENT_LIST_URL = "http://api.qichacha.com/ECIInvestment/GetInvestmentList";

    @Autowired
    QiChaChaEnterpriseInfoMapper qiChaChaEnterpriseInfoMapper;
    @Autowired
    QichachaFullDetailsMapper qichachaFullDetailsMapper;
    @Autowired
    QichachaHistorytEciMapper qichachaHistorytEciMapper;
    @Autowired
    QichachaHistorytShareHolderMapper qichachaHistorytShareHolderMapper;
    @Autowired
    QichachaInvestmentListMapper qichachaInvestmentListMapper;
    @Autowired
    JsLoanRiskInfoMapper jsLoanRiskInfoMapper;

    @Override
    public ResponseMessage getEnterpriseInfo(String kycNaturalId) {
        if (ObjectUtil.isEmpty(kycNaturalId)) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        }
        JsLoanRiskInfo jsLoanRiskInfo = new JsLoanRiskInfo();
        jsLoanRiskInfo.setKycNaturalId(kycNaturalId);
        jsLoanRiskInfo.setDelFlag(false);
        jsLoanRiskInfo = jsLoanRiskInfoMapper.selectOne(jsLoanRiskInfo);
        if (ObjectUtil.isNull(jsLoanRiskInfo) || ObjectUtil.isEmpty(jsLoanRiskInfo.getGuaranteeCorp())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        }
        QichachaFullDetails fullDetails = getFullDetailsByName(kycNaturalId, jsLoanRiskInfo.getGuaranteeCorp());
        QichachaHistorytEci qichachaHistorytEci = getQichachaHistorytEci(kycNaturalId, jsLoanRiskInfo.getGuaranteeCorp());
        QichachaHistorytShareHolder historytShareHolder = getHistorytShareHolder(kycNaturalId, jsLoanRiskInfo.getGuaranteeCorp());

        JSONObject fullDetailsJson = JSONUtil.parseObj(fullDetails.getResponseBodyData());
        JSONObject qichachaHistorytEciJson = JSONUtil.parseObj(qichachaHistorytEci.getResponseBodyData());
        JSONObject historytShareHolderJson = JSONUtil.parseObj(historytShareHolder.getResponseBodyData());
        if (!"200".equals(fullDetailsJson.getStr("Status"))) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        }

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("js-2001", calMonths(fullDetailsJson));
        jsonObject.put("js-2002", jsLoanRiskInfo.getCorpBankProvinceName().replace("市", "").replace("省", ""));
        jsonObject.put("js-2003", calOperIsPartner(fullDetailsJson));
        jsonObject.put("js-2004", calOperChange(qichachaHistorytEciJson));
        jsonObject.put("js-2005", calPartnerChange(historytShareHolderJson));
        jsonObject.put("js-2006", calHasExecution(kycNaturalId, fullDetailsJson));
        jsonObject.put("js-2007", calHasPenalty(kycNaturalId, fullDetailsJson));
        jsonObject.put("js-2008", calHasException(kycNaturalId, fullDetailsJson));
        jsonObject.put("js-2009", jsLoanRiskInfo.getCorpBankCityName().replace("市", "").replace("省", ""));
        return  ResultUtil.success(jsonObject, ResultEnum.QUERY_SUCCESS);
    }

    @Override
    public ResponseMessage getEnterprise25Partners(String kycNaturalId, String enterpriseName) {
        QichachaFullDetails fullDetails = getFullDetailsByName(kycNaturalId, enterpriseName);
        if (fullDetails == null) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        } else {
            JSONObject fullDetailsJson = JSONUtil.parseObj(fullDetails.getResponseBodyData());
            String status = fullDetailsJson.getStr("Status");
            if ("200".equals(status)) {
                JSONObject result = fullDetailsJson.getJSONObject("Result");
                JSONArray partners = result.getJSONArray("Partners");
                JSONArray objects = new JSONArray();
                for (int i = 0; i < partners.size(); i++) {
                    JSONObject partner = partners.getJSONObject(i);
                    // 只计算大于等于 25% 的人(名字结尾不是公司)
                    String stockName = partner.getStr("StockName");
                    if (!stockName.endsWith("公司")) {
                        String stockPercent = partner.getStr("StockPercent").replace("%", "");
                        if (new BigDecimal(stockPercent).compareTo(new BigDecimal(25)) >= 0) {
                            JSONObject jsonObject = new JSONObject();
                            jsonObject.put("stockName", partner.getStr("StockName"));
                            jsonObject.put("stockPercent", partner.getStr("StockPercent"));
                            objects.add(jsonObject);
                        }
                    }
                }
                return ResultUtil.success(objects, ResultEnum.QUERY_SUCCESS);
            } else {
                return ResultUtil.error(ResultEnum.QUERY_ERROR);
            }
        }
    }

    @Override
    public ResponseMessage getEnterprisePartners2(String kycNaturalId, String enterpriseName) {
        List<Map<String, Object>> list = getEnterprisePartners(kycNaturalId, enterpriseName, new BigDecimal("0.25"), new BigDecimal("1"), -1);
        return ResultUtil.success(list, ResultEnum.QUERY_SUCCESS);
    }

    @Override
    public ResponseMessage checkEnterpriseAndCreditCode(String kycNaturalId, String enterpriseName, String creditCode) {
        QichachaFullDetails fullDetails = getFullDetailsByName(kycNaturalId, enterpriseName);
        if (ObjectUtil.isEmpty(fullDetails)) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        } else {
            JSONObject fullDetailsJson = JSONUtil.parseObj(fullDetails.getResponseBodyData());
            String status = fullDetailsJson.getStr("Status");
            if ("200".equals(status)) {
                String code = fullDetailsJson.getJSONObject("Result").getStr("CreditCode");
                if (creditCode.equals(code)) {
                    return ResultUtil.success(true, ResultEnum.QUERY_SUCCESS);
                } else {
                    return ResultUtil.success(false, ResultEnum.QUERY_SUCCESS);
                }
            } else {
                return ResultUtil.error(ResultEnum.QUERY_ERROR);
            }
        }
    }

    @Override
    public ResponseMessage getHasException(String kycNaturalId, String enterpriseName) {
        QichachaFullDetails fullDetails = getFullDetailsByName(kycNaturalId, enterpriseName);
        if (ObjectUtil.isEmpty(fullDetails)) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR);
        } else {
            JSONObject fullDetailsJson = JSONUtil.parseObj(fullDetails.getResponseBodyData());
            String status = fullDetailsJson.getStr("Status");
            if ("200".equals(status)) {
                JSONObject result = fullDetailsJson.getJSONObject("Result");
                JSONArray items = result.getJSONArray("Exceptions");
                boolean hasException = calHasItemNearlyYear(items, "AddDate");
                return ResultUtil.success(hasException, ResultEnum.QUERY_SUCCESS);
            } else {
                return ResultUtil.error(ResultEnum.QUERY_ERROR);
            }
        }
    }

    /**
     * 穿透查询企业股东
     * @param kycNaturalId 用户 id
     * @param enterpriseName 企业名称列表
     * @param percentSave 获取占股多少的股东以小数表示
     * @param percent 当前公司持有目标公司的股份。调用时传 1
     * @param deep 穿透层数。 -1 代表一直查询到结束
     * @return resultList [{stockName: 股东名, stockPercent: 持股比例}]
     */
    private List<Map<String, Object>> getEnterprisePartners(String kycNaturalId, String enterpriseName,
                                                            BigDecimal percentSave, BigDecimal percent, int deep) {
        if (ObjectUtil.isEmpty(kycNaturalId) || ObjectUtil.isEmpty(enterpriseName)) {
            throw LogicException.le(ResultEnum.QICHACHA_ATTACH_ENTERPRISE_PARTNER_VALIDATE_FAILURE);
        }
        if (BigDecimal.ONE.compareTo(percentSave) < 0 || BigDecimal.ONE.compareTo(percent) < 0) {
            throw LogicException.le(ResultEnum.QICHACHA_ATTACH_ENTERPRISE_PARTNER_VALIDATE_FAILURE2);
        }
        List<Map<String, Object>> resultList = new ArrayList<>();
        if (deep < -1) {
            log.error("穿透查询企业股东深度错误");
            throw LogicException.le(ResultEnum.QICHACHA_ATTACH_ENTERPRISE_PARTNER_FAILURE);
        } else if (deep == 0) {
            return resultList;
        }
        QichachaFullDetails fullDetails = getFullDetailsByName(kycNaturalId, enterpriseName);
        if (fullDetails == null) {
            return resultList;
        } else {
            JSONObject fullDetailsJson = JSONUtil.parseObj(fullDetails.getResponseBodyData());
            String status = fullDetailsJson.getStr("Status");
            if ("200".equals(status)) {
                JSONObject result = fullDetailsJson.getJSONObject("Result");
                JSONArray partners = result.getJSONArray("Partners");
                for (int i = 0; i < partners.size(); i++) {
                    JSONObject partner = partners.getJSONObject(i);
                    // 只计算大于等于 percentSave 的人(名字结尾不是公司)
                    String stockName = partner.getStr("StockName");
                    if (!stockName.endsWith("公司")) {
                        String stockPercent = partner.getStr("StockPercent").replace("%", "");
                        BigDecimal newPercent = new BigDecimal(stockPercent).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP).multiply(percent);
                        if (newPercent.compareTo(percentSave) >= 0) {
                            Map<String, Object> map = new HashMap<>();
                            map.put("stockName", partner.getStr("StockName"));
                            map.put("stockPercent", newPercent.multiply(new BigDecimal("100")));
                            resultList.add(map);
                        }
                    } else {
                        // 持股人是公司
                        String stockPercent = partner.getStr("StockPercent").replace("%", "");
                        BigDecimal newPercent = new BigDecimal(stockPercent).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP).multiply(percent);
                        if (newPercent.compareTo(percentSave) >= 0) {
                            if (deep != -1) {
                                deep--;
                            }
                            List<Map<String, Object>> list = getEnterprisePartners(kycNaturalId, stockName, percentSave,
                                    new BigDecimal(stockPercent).divide(new BigDecimal("100"), 4, RoundingMode.HALF_UP), deep);
                            resultList.addAll(list);
                        }
                    }
                }
                return resultList;
            } else {
                return resultList;
            }
        }
    }

    private boolean calHasException(String kycNaturalId, JSONObject fullDetailsJson) {
        return calHasItemTemplate(kycNaturalId, fullDetailsJson, "Exceptions", "AddDate");
    }

    private boolean calHasPenalty(String kycNaturalId, JSONObject fullDetailsJson) {
        return calHasItemTemplate(kycNaturalId, fullDetailsJson, "Penalty", "PublicDate");
    }

    private boolean calHasExecution(String kycNaturalId, JSONObject fullDetailsJson) {
        return calHasItemTemplate(kycNaturalId, fullDetailsJson, "ZhiXingItems", "Liandate");
    }

    /**
     * 计算所指明细项是否有一年内的明细（包含关联企业）
     * @param kycNaturalId 用户 id
     * @param fullDetailsJson 该用户主体的工商信息
     * @param selectKey 需要计算明细项的 key
     * @param determineKey 判定时间所需要的字段 key
     * @return
     */
    private boolean calHasItemTemplate(String kycNaturalId, JSONObject fullDetailsJson, String selectKey, String determineKey) {
        JSONObject result = fullDetailsJson.getJSONObject("Result");
        String name = result.getStr("Name");
        JSONArray items = result.getJSONArray(selectKey);
        if (calHasItemNearlyYear(items, determineKey)) {
            // 如果主企业已经存在问题则直接返回。
            return true;
        }
        // 查找全部关联企业的名字
        List<String> nameList = calAttachEnterprise(kycNaturalId, Arrays.asList(name), 2, null);
        // 获取全部关联企业的信息。
        for (String n : nameList) {
            QichachaFullDetails fullDetails = getFullDetailsByName(kycNaturalId, n);
            JSONObject jsonObject = JSONUtil.parseObj(fullDetails.getResponseBodyData());
            JSONArray array = jsonObject.getJSONObject("Result").getJSONArray(selectKey);
            if (calHasItemNearlyYear(array, determineKey)) {
                // 关联企业经存在问题则直接返回。
                return true;
            }
        }
        return false;
    }

    /**
     * 计算该企业的关联的企业
     *
     * @param kycNaturalId 用户 kycNaturalId
     * @param nameList 公司名称列表
     * @param deep 查找深度。如果为 -1 则会一直查找到底。
     */
    private List<String> calAttachEnterprise(String kycNaturalId, List<String> nameList, int deep, List<String> resultList) {
        if (resultList == null) {
            resultList = new ArrayList<>();
        }
        if (deep < -1) {
            log.error("关联企业查找深度错误");
            throw LogicException.le(ResultEnum.QICHACHA_ATTACH_ENTERPRISE_FAILURE);
        }
        if (deep == 0) {
            return resultList;
        }
        List<String> newNameList = new ArrayList<>();
        for (String name : nameList) {
            QichachaInvestmentList investmentList = getInvestmentList(kycNaturalId, name);
            if (ObjectUtil.isNotNull(investmentList)) {
                if ("201".equals(JSONUtil.parseObj(investmentList.getResponseBodyData()).getStr("Status"))) {
                    return resultList;
                }
                JSONArray array = JSONUtil.parseObj(investmentList.getResponseBodyData()).getJSONArray("Result");
                for (int i = 0; i < array.size(); i++) {
                    String name1 = array.getJSONObject(i).getStr("Name");
                    if (!resultList.contains(name1)) {
                        resultList.add(name1);
                        newNameList.add(name1);
                    }
                }
            }
        }
        if (newNameList.size() == 0) {
            return resultList;
        } else {
            if (deep == -1) {
                return calAttachEnterprise(kycNaturalId, newNameList, deep, resultList);
            } else {
                deep--;
                return calAttachEnterprise(kycNaturalId, newNameList, deep, resultList);
            }
        }
    }

    private long calMonths(JSONObject fullDetailsJson) {
        JSONObject result = fullDetailsJson.getJSONObject("Result");
        Date startDate = result.getDate("StartDate");
        ZonedDateTime now = ZonedDateTime.now();
        ZonedDateTime time = ZonedDateTime.ofInstant(startDate.toInstant(), ZoneId.systemDefault());
        return time.until(now, ChronoUnit.MONTHS);
    }

    /**
     * 企查查企业对外投资查询
     *
     * @param kycNaturalId 用户的 kyc id
     * @param name 要查询的公司名称
     */
    private QichachaInvestmentList getInvestmentList(String kycNaturalId, String name) {
        try {
            Map<String, Object> parameter = new HashMap<>();
            parameter.put("searchKey", name);
            return sendRequestTemplate(kycNaturalId, name, parameter, INVESTMENT_LIST_URL, QichachaInvestmentList.class,
                    qichachaInvestmentListMapper, (map, entity) -> {
                        entity.setResponseBodyData(map.get("response"));
                        entity.setKycNaturalId(map.get("kycNaturalId"));
                        entity.setName(map.get("name"));
                    });
        } catch (Exception e) {
            e.printStackTrace();
            log.error("企查查企业对外投资查询失败：{}", e.getMessage());
            return null;
        }
    }

    /**
     * 企查查历史股东查询
     *
     * @param kycNaturalId 用户的 kyc id
     * @param name 要查询的公司名称
     */
    private QichachaHistorytShareHolder getHistorytShareHolder(String kycNaturalId, String name) {
        try {
            Map<String, Object> parameter = new HashMap<>();
            parameter.put("keyWord", name);
            return sendRequestTemplate(kycNaturalId, name, parameter, HISTORYT_SHARE_HOLDER_URL, QichachaHistorytShareHolder.class,
                    qichachaHistorytShareHolderMapper, (map, entity) -> {
                entity.setResponseBodyData(map.get("response"));
                entity.setKycNaturalId(map.get("kycNaturalId"));
                entity.setName(map.get("name"));
            });
        } catch (Exception e) {
            e.printStackTrace();
            log.error("企查查历史股东查询失败：{}", e.getMessage());
            return null;
        }
    }

    /**
     * 企查查历史工商企业查询
     *
     * @param kycNaturalId 用户的 kyc id
     * @param name 要查询的公司名称
     */
    private QichachaHistorytEci getQichachaHistorytEci(String kycNaturalId, String name) {
        try {
            Map<String, Object> parameter = new HashMap<>();
            parameter.put("keyWord", name);
            return sendRequestTemplate(kycNaturalId, name, parameter, HISTORYT_ECI_URL, QichachaHistorytEci.class,
                    qichachaHistorytEciMapper, (map, entity) -> {
                entity.setResponseBodyData(map.get("response"));
                entity.setKycNaturalId(map.get("kycNaturalId"));
                entity.setName(map.get("name"));
            });
        } catch (Exception e) {
            e.printStackTrace();
            log.error("企查查历史工商企业查询失败：{}", e.getMessage());
            return null;
        }
    }

    /**
     * 企查查工商企业查询
     *
     * @param kycNaturalId 用户的 kyc id
     * @param name 要查询的公司名称
     */
    private QichachaFullDetails getFullDetailsByName(String kycNaturalId, String name) {
        try {
            Map<String, Object> parameter = new HashMap<>();
            parameter.put("keyword", name);
            return sendRequestTemplate(kycNaturalId, name, parameter, FULL_DETAILS_BY_NAME_URL, QichachaFullDetails.class,
                    qichachaFullDetailsMapper, (map, entity) -> {
                entity.setResponseBodyData(map.get("response"));
                entity.setKycNaturalId(map.get("kycNaturalId"));
                entity.setName(map.get("name"));
            });
        } catch (Exception e) {
            e.printStackTrace();
            log.error("请求企查查工商信息出错");
            return null;
        }
    }

    /**
     * 企查查请求发送模板
     *
     * 请求结果会被缓存下来。如果7天之内再次请求。将使用本地数据而不会调用第三方接口
     * @param kycNaturalId 用户 kycId
     * @param name 公司名
     * @param parameter 调用接口时需要传递的参数
     * @param url 调用接口需要使用的地址
     * @param clazz 需要实体化存储的类的
     * @param mapper 操作数据库需要使用的 mapper
     * @param setProperties 函数式接口，使用该接口实现对不同类型个性化的赋值。
     * @return
     * @throws IllegalAccessException
     * @throws InstantiationException
     */
    private <T extends IBaseModel> T sendRequestTemplate(String kycNaturalId, String name, Map<String, Object> parameter,
                                                            String url, Class<T> clazz, IBaseMapper<T> mapper,
                                                         BiConsumer<Map<String, String>, T> setProperties)
            throws IllegalAccessException, InstantiationException {
        //检查数据库中数据是否符合使用条件
        Example example = new Example(clazz);
        Example.Criteria criteria = example.createCriteria();
        // TODO 这里要不要加上 kycId 的限制呢？
        //criteria.andEqualTo("kycNaturalId", kycNaturalId);
        criteria.andEqualTo("name", name);
        T old = mapper.selectOneByExample(example);
        if (ObjectUtil.isNotNull(old)) {
            ZonedDateTime updateDts = ZonedDateTime.ofInstant(old.getUpdateDts().toInstant(), ZoneId.systemDefault());
            ZonedDateTime now = ZonedDateTime.now();
            long days = updateDts.until(now, ChronoUnit.DAYS);
            if (days <= 7) {
                return old;
            }
        }
        // 数据不存在或已经过期则重新调用企查查接康口更新数据
        HttpResponse response = sendRequestToQichacha(url, parameter);
        log.info("result: {}", response.body());
        if (response.getStatus() == 200) {
            HashMap<String, String> map = new HashMap<>();
            map.put("kycNaturalId", kycNaturalId);
            map.put("response", response.body());
            map.put("name", name);
            String status = JSONUtil.parseObj(response.body()).getStr("Status");
            if ("200".equals(status) || "201".equals(status)) {
                if (ObjectUtil.isNotNull(old)) {
                    old.setUpdateId("sys");
                    old.setUpdateName("sys");
                    old.setUpdateDts(new Date());
                    setProperties.accept(map, old);
                    log.info("class: {}, 数据更新成功", clazz);
                    mapper.updateByPrimaryKeySelective(old);
                    return old;
                } else {
                    T newEntity  = clazz.newInstance();
                    newEntity.setId(IdUtil.simpleUUID());
                    newEntity.setCreateId("sys");
                    newEntity.setCreateName("sys");
                    newEntity.setCreateDts(new Date());
                    setProperties.accept(map, newEntity);
                    log.info("class: {}, 数据插入成功", clazz);
                    mapper.insertSelective(newEntity);
                    return newEntity;
                }
            } else {
                log.error("请求信息出错: {}", response.body());
                throw LogicException.le(ResultEnum.QICHACHA_QEQUEST_FAILURE);
            }
        } else {
            log.error("请求信息出错: {}", response.body());
            throw LogicException.le(ResultEnum.QICHACHA_QEQUEST_FAILURE);
        }
    }

    private HttpResponse sendRequestToQichacha(String url, Map<String, Object> parameter) {
        parameter.put("key", appKey);
        long timespan = Instant.now().getEpochSecond();
        String token = calToken(timespan);
        return HttpRequest.get(url)
                    .header("Token", token)
                    .header("Timespan", new Long(timespan).toString())
                    .form(parameter)
                    .execute();
    }

    /**
     * 检查近期有无相关记录
     *
     * @param items 检查项目
     * @param key 使用的检查项
     * @return
     */
    private Boolean calHasItemNearlyYear(JSONArray items, String key) {
        if (ObjectUtil.isEmpty(items)) {
            return false;
        } else {
            LocalDateTime localDateTime = LocalDateTime.now().minusYears(1);
            for (int i = 0; i < items.size(); i++) {
                JSONObject item = items.getJSONObject(i);
                Date addDate = item.getDate(key);
                // 只取近一年的。
                if (addDate.getTime() > localDateTime.toInstant(ZoneOffset.of("+8")).toEpochMilli()) {
                    return true;
                }
            }
            return false;
        }
    }

    private Boolean calOperChange(JSONObject qichachaHistorytEciJson) {
        if ("201".equals(qichachaHistorytEciJson.getJSONObject("Result").getStr("Status"))) {
            return false;
        }
        JSONObject result = qichachaHistorytEciJson.getJSONObject("Result");
        JSONArray operList = result.getJSONArray("OperList");
        ZonedDateTime zonedDateTime = ZonedDateTime.now().minusYears(1);
        for (int i = 0; i < operList.size(); i++) {
            JSONObject jsonObject = operList.getJSONObject(i);
            Date changeDate = jsonObject.getDate("ChangeDate");
            if (zonedDateTime.toInstant().toEpochMilli() < changeDate.getTime()) {
                return true;
            }
        }
        return false;
    }

    private Boolean calPartnerChange(JSONObject historytShareHolderJson) {
        if ("201".equals(historytShareHolderJson.getJSONObject("Result").getStr("Status"))) {
            return false;
        }
        JSONObject result = historytShareHolderJson.getJSONObject("Result");
        JSONArray changeDateList = result.getJSONArray("ChangeDateList");
        ZonedDateTime zonedDateTime = ZonedDateTime.now().minusYears(1);
        if (changeDateList == null) {
            // 无历史信息代表未查到。
            return false;
        }
        for (int i = 0; i < changeDateList.size(); i++) {
            Date changeDate = changeDateList.getDate(i);
            if (zonedDateTime.toInstant().toEpochMilli() < changeDate.getTime()) {
                return true;
            }
        }
        return false;
    }

    private Boolean calOperIsPartner(JSONObject fullDetailsJson) {
        JSONObject result = fullDetailsJson.getJSONObject("Result");
        String operName = result.getStr("OperName");
        JSONArray partners = result.getJSONArray("Partners");
        for (int i = 0; i < partners.size(); i++) {
            JSONObject partner = partners.getJSONObject(i);
            String stockName = partner.getStr("StockName");
            if (operName.equals(stockName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 计算规则：
     * 验证加密值（key+Timespan+SecretKey组成的32位md5加密的大写字符串）
     *
     * @param timespan 精确到秒的Unix时间戳
     * @return
     */
    private String calToken (long timespan) {
        return SecureUtil.md5(appKey + timespan + secretKey).toUpperCase();
    }

}
