package com.js.loan.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.IdUtil;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.js.api.jsloan.service.ApiJsLoanBillFileService;
import com.js.common.constant.CommonConstant;
import com.js.common.constant.Constant;
import com.js.common.enums.ResultEnum;
import com.js.common.model.req.JsLoanBillFilePageReq;
import com.js.common.model.req.JsLoanBillFileReq;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.ResultUtil;
import com.js.dal.dao.mapper.JsLoanApplyInfoMapper;
import com.js.dal.dao.mapper.JsLoanBillFileMapper;
import com.js.dal.dao.model.JsLoanApplyInfo;
import com.js.dal.dao.model.JsLoanBillFile;
import com.js.loan.utils.DateUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import java.util.*;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class ApiJsLoanBillFileServiceImpl implements ApiJsLoanBillFileService {

    @Autowired
    private JsLoanBillFileMapper jsLoanBillFileMapper;

    @Autowired
    private JsLoanApplyInfoMapper jsLoanApplyInfoMapper;

    /**
     * 对账申请查询
     * @param jsLoanBillFilePageReq
     * @return
     */
    @Override
    public ResponseMessage applyQuery(JsLoanBillFilePageReq jsLoanBillFilePageReq) {
        log.info("对账申请查询 kyc:{}",jsLoanBillFilePageReq.getKycNaturalId());
        Example example=new Example(JsLoanBillFile.class);
        example.setOrderByClause("apply_date DESC");
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId",jsLoanBillFilePageReq.getKycNaturalId());
        PageHelper.startPage(jsLoanBillFilePageReq);
        List<JsLoanBillFile> jsLoanBillFileList = jsLoanBillFileMapper.selectByExample(example);
        PageInfo<JsLoanBillFile> jsLoanBillFileVo = new PageInfo<>(jsLoanBillFileList);
        return ResultUtil.success(jsLoanBillFileVo);
    }

    /**
     * 对账申请
     * @param jsLoanBillFileReq
     * @return
     */
    @Override
    public ResponseMessage applyOperation(JsLoanBillFileReq jsLoanBillFileReq) {

        log.info("申请对账文件开始 req:{}",jsLoanBillFileReq);
        JsLoanBillFile jsLoanBillFileSel = new JsLoanBillFile();
        jsLoanBillFileSel.setKycNaturalId(jsLoanBillFileReq.getKycNaturalId());
        jsLoanBillFileSel.setBillStartDate(DateUtils.StrToDate(jsLoanBillFileReq.getBillStartDate()));
        jsLoanBillFileSel.setBillEndDate(DateUtils.StrToDate(jsLoanBillFileReq.getBillEndDate()));
        List<JsLoanBillFile> jsLoanBillFileList  = jsLoanBillFileMapper.select(jsLoanBillFileSel);
        if(jsLoanBillFileList.size() > 0){
            log.info("已经申请下载过相同时间段的账单，请至【可下载账单】查看下载 kyc:{}",jsLoanBillFileReq.getKycNaturalId());
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"已经申请下载过相同时间段的账单，请至【可下载账单】查看下载");
        }
        //查询贷款记录
        Example example=new Example(JsLoanApplyInfo.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andBetween("applyDate",jsLoanBillFileReq.getBillStartDate(),DateUtils.getDateEndDayStr(jsLoanBillFileReq.getBillEndDate()));
        criteria.andEqualTo("kycNaturalId",jsLoanBillFileReq.getKycNaturalId());
        criteria.andEqualTo("delFlag",false);
        List<JsLoanApplyInfo> jsLoanApplyInfoList = jsLoanApplyInfoMapper.selectByExample(example);
        if(jsLoanApplyInfoList.size() < 1){
            log.info("您申请的对账文件再该时间段未查询到交易数据，请核实!");
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"您申请的对账文件再该时间段未查询到交易数据，请核实!");
        }
        JsLoanBillFile jsLoanBillFileIns = new JsLoanBillFile();
        jsLoanBillFileIns.setKycNaturalId(jsLoanBillFileReq.getKycNaturalId());
        jsLoanBillFileIns.setApplyDate(new Date());
        jsLoanBillFileIns.setBillStartDate(DateUtils.StrToDate(jsLoanBillFileReq.getBillStartDate()));
        jsLoanBillFileIns.setBillEndDate(DateUtils.StrToDate(jsLoanBillFileReq.getBillEndDate()));
        jsLoanBillFileIns.setApplyStatus(CommonConstant.LOAN_BILL_FILE_APPLY_STATUS_INIT);//初始化
        jsLoanBillFileIns.setBillType("010");//账单类型 010贷款账单
        jsLoanBillFileIns.setLoanBillNo(IdUtil.simpleUUID());
        jsLoanBillFileIns.setId(IdUtil.simpleUUID());
        int count = jsLoanBillFileMapper.insertSelective(jsLoanBillFileIns);
        if(count < 1){
            log.info("贷款申请插入数据库失败 kyc:{}",jsLoanBillFileReq.getKycNaturalId());
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"贷款申请失败");
        }
        return ResultUtil.success(ResultEnum.SUCCESS.getCode(),"对账文件申请成功");
    }

    /**
     * 对账下载
     * @param loanBillNo
     * @return
     */
    @Override
    public ResponseMessage download(String kycNaturalId,String loanBillNo) {

        log.info("查询导出数据 ...kyc:{},loanBillNo:{}",kycNaturalId,loanBillNo);
        //查询申请记录
        JsLoanBillFile jsLoanBillFile = new JsLoanBillFile();
        jsLoanBillFile.setKycNaturalId(kycNaturalId);
        jsLoanBillFile.setLoanBillNo(loanBillNo);
        jsLoanBillFile = jsLoanBillFileMapper.selectOne(jsLoanBillFile);
        if(null == jsLoanBillFile){
             log.info("查询导出数据失败未查询到申请记录");
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"查询导出数据失败未查询到申请记录");
        }
        if(!jsLoanBillFile.getApplyStatus().equals(CommonConstant.LOAN_BILL_FILE_APPLY_STATUS_SUCCESS)){
            log.info("申请的记录未审核通过，不可以下载");
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"申请的记录未审核通过，不可以下载");
        }
        //查询贷款记录
        Example example=new Example(JsLoanApplyInfo.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId",kycNaturalId);
        criteria.andBetween("applyDate",jsLoanBillFile.getBillStartDate(),DateUtils.getDateEndDay(jsLoanBillFile.getBillEndDate()));
        criteria.andEqualTo("delFlag",false);
        PageHelper.startPage(1,5000);
        List<JsLoanApplyInfo> jsLoanApplyInfoList = jsLoanApplyInfoMapper.selectByExample(example);

        if(jsLoanApplyInfoList.size() < 1){
            log.info("您申请的时间段没有相关借款数据，请核实");
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"您申请的时间段没有相关借款数据，请核实");
        }
        ArrayList<Map<String, Object>> rows = CollUtil.newArrayList();
        for(JsLoanApplyInfo jsLoanApplyInfo : jsLoanApplyInfoList){
            Map<String, Object> row = new LinkedHashMap<>();
            row.put("贷款编号", jsLoanApplyInfo.getLoanNo());
            row.put("放款时间", "2019-09-09"); //放款时间
            row.put("放款金额", jsLoanApplyInfo.getRealAmt());
            if(jsLoanApplyInfo.getRepayType().equals("01")){//自动还款
                row.put("还款方式", "按日计息");
            }
            row.put("贷款期限", jsLoanApplyInfo.getLoanDays() + "天");
            row.put("日利率", jsLoanApplyInfo.getInterestRate() + "%");
            row.put("还款日", "2019-09-09");//还款日
            row.put("放款账户", jsLoanApplyInfo.getAcctNo());
            row.put("待还本金", "3000.00");//待还本金
            row.put("待还利息+费用", "30.00");//待还利息+费用
            row.put("已还金额", "3000.00");//已还金额
            row.put("已还本金", "3000.00");//已还本金
            row.put("已还利息+费用", "30.00");//已还利息+费用
            row.put("结清日期", "2019-09-09");//结清日期
            rows.add(row);
        }
        return ResultUtil.success(rows,ResultEnum.SUCCESS.getCode(),"下载对账文件成功");
    }
}
