package com.js.loan.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.json.JSONObject;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.js.api.jsloan.service.ApiStatisticsInfoService;
import com.js.common.constant.Constant;
import com.js.common.enums.RegisterType;
import com.js.common.enums.ResultEnum;
import com.js.common.enums.StoreAuthStatusEnum;
import com.js.common.model.req.JsLoanStatisticsInfoReq;
import com.js.common.model.vo.JsLoanStatisticsInfoKycVO;
import com.js.common.model.vo.JsLoanStatisticsInfoStoreVO;
import com.js.common.model.vo.JsLoanStatisticsInfoVO;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.ResultUtil;
import com.js.dal.dao.mapper.*;
import com.js.dal.dao.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.springframework.beans.factory.annotation.Autowired;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Slf4j
@Service(
        protocol = {"rest", "dubbo"},
        version = Constant.DUBBO_VERSION,
        application = "${dubbo.application.id}",
        registry = "${dubbo.registry.id}"
)
public class ApiStatisticsInfoServiceImpl implements ApiStatisticsInfoService {

    @Autowired
    KycNaturalMapper kycNaturalMapper;
    @Autowired
    KycStoreMapper kycStoreMapper;
    @Autowired
    KycCertiHKCompanyMapper kycCertiHKCompanyMapper;
    @Autowired
    KycCertiHKPersonalMapper kycCertiHKPersonalMapper;
    @Autowired
    KycCertiMainLandCompanyMapper kycCertiMainLandCompanyMapper;
    @Autowired
    KycCertiMainLandPersonalMapper kycCertiMainLandPersonalMapper;
    @Autowired
    PlatformStatisticsByKycMapper platformStatisticsByKycMapper;
    @Autowired
    PlatformStatisticsByStoreMapper platformStatisticsByStoreMapper;


    @Override
    public ResponseMessage getBaseInfo(String kycNaturalId) {
        // 获取用户注册信息
        KycNatural kycNatural = kycNaturalMapper.selectByPrimaryKey(kycNaturalId);
        JSONObject result = new JSONObject();
        result.put("js-1001", false);
        result.put("js-1002", 0);
        result.put("js-1003", 22);
        result.put("js-1004", "未婚");
        result.put("js-1006", true);

        if (ObjectUtil.isEmpty(kycNatural)) {
            return ResultUtil.success(result, ResultEnum.QUERY_SUCCESS);
        } else {
            // 判断具体的注册类型。大陆企业\个人 香港企业\个人
            String registerType = kycNatural.getRegisterType();
            if (RegisterType.MAINLAND_PERSONAL.name().equals(registerType)) {
                // 中国大陆-个人
                KycCertiMainLandPersonal searchCondition = new KycCertiMainLandPersonal();
                searchCondition.setKycNaturalId(kycNatural.getId());
                List<KycCertiMainLandPersonal> kycCertiMainLandPersonals = kycCertiMainLandPersonalMapper.select(searchCondition);
                KycCertiMainLandPersonal kycCertiMainLandPersonal;
                // 校验数据
                if (kycCertiMainLandPersonals.size() != 1) {
                    return ResultUtil.error("数据错误认证信匹配失败", ResultEnum.QUERY_ERROR);
                } else {
                    kycCertiMainLandPersonal = kycCertiMainLandPersonals.get(0);
                }
                String idNo = kycCertiMainLandPersonal.getIdNo();
                if (ObjectUtil.isEmpty(idNo) || idNo.length() < 18) {
                    return ResultUtil.error("kyc 身份证号不正确", ResultEnum.QUERY_ERROR);
                }
                // 身份证有效期是否超过半年。
                long until = getIdEffectiveTime(kycCertiMainLandPersonal.getEffectiveStartTime(), kycCertiMainLandPersonal.getEffectiveEndTime());
                result.put("1001", until > 6);
                // 计算性别
                result.put("1002", calSex(idNo));
                // 计算年龄(周岁)
                result.put("js-1003", calAge(idNo));
                // TODO 计算身份证所在地区
            } else if (RegisterType.MAINLAND_COMPANY.name().equals(registerType)) {
                // 中国大陆-企业
                KycCertiMainLandCompany searchConditon = new KycCertiMainLandCompany();
                searchConditon.setKycNaturalId(kycNatural.getId());
                List<KycCertiMainLandCompany> kycCertiMainLandCompanies = kycCertiMainLandCompanyMapper.select(searchConditon);
                // 校验数据
                KycCertiMainLandCompany kycCertiMainLandCompanie;
                if (kycCertiMainLandCompanies.size() != 1) {
                    return ResultUtil.error("数据错误认证信匹配失败", ResultEnum.QUERY_ERROR);
                } else {
                    kycCertiMainLandCompanie = kycCertiMainLandCompanies.get(0);
                }
                // TODO 企业相关的先不做。表还没有。
            } else if (RegisterType.HK_PERSONAL.name().equals(registerType)) {
                // 中国香港-个人
                KycCertiHKPersonal searchCondition = new KycCertiHKPersonal();
                searchCondition.setKycNaturalId(kycNatural.getId());
                List<KycCertiHKPersonal> kycCertiHKPersonals = kycCertiHKPersonalMapper.select(searchCondition);
            } else if (RegisterType.MAINLAND_COMPANY.name().equals(registerType)) {
                // 中国香港-企业
            } else if (RegisterType.NONE_TYPE.name().equals(registerType)) {
                return ResultUtil.error("kyc 尚未实名认证", ResultEnum.QUERY_ERROR);
            } else {
                log.error("用户注册类型错误：registerType:{}, id:{}", registerType, kycNatural.getId());
                return ResultUtil.error(ResultEnum.QUERY_ERROR);
            }

        }

        return ResultUtil.success(result, ResultEnum.QUERY_SUCCESS);
    }

    @Override
    public ResponseMessage getPlatformKycInfo(JsLoanStatisticsInfoReq jsLoanStatisticsInfoReq) {
        if (ObjectUtil.isEmpty(jsLoanStatisticsInfoReq.getKycNaturalId())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR.getCode(), "用户 id 不许为空");
        }
        if (ObjectUtil.isEmpty(jsLoanStatisticsInfoReq.getStatisticsTime())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR.getCode(), "统计时间不许为空");
        }
        Example example = new Example(PlatformStatisticsByKyc.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("kycNaturalId", jsLoanStatisticsInfoReq.getKycNaturalId());
        Date statisticsTime = jsLoanStatisticsInfoReq.getStatisticsTime();
        ZonedDateTime time = ZonedDateTime.ofInstant(statisticsTime.toInstant(), ZoneId.systemDefault());
        statisticsTime = Date.from(ZonedDateTime.
                of(time.getYear(), time.getMonthValue(), 1, 0,0 ,0 ,0,ZoneId.of("UTC")).toInstant());
        criteria.andEqualTo("statisticsTime", statisticsTime);

        List<PlatformStatisticsByKyc> platformStatisticsByKycs = platformStatisticsByKycMapper.selectByExample(example);
        if (platformStatisticsByKycs.size() == 1) {
            JsLoanStatisticsInfoKycVO jsLoanStatisticsInfoKycVO = new JsLoanStatisticsInfoKycVO();
            BeanUtil.copyProperties(platformStatisticsByKycs.get(0), jsLoanStatisticsInfoKycVO);

            JSONObject jsonObject = new JSONObject();
            jsonObject.put("js-4002", jsLoanStatisticsInfoKycVO.getHalfYearSaleRatio());
            jsonObject.put("js-4003", jsLoanStatisticsInfoKycVO.getMonthlySale());
            jsonObject.put("js-4009", jsLoanStatisticsInfoKycVO.getMonthlyIncome());
            jsonObject.put("js-4011", jsLoanStatisticsInfoKycVO.getIncomeChainRate());
            jsonObject.put("js-4012", jsLoanStatisticsInfoKycVO.getIncomeThreeRatio());
            jsonObject.put("js-4013", jsLoanStatisticsInfoKycVO.getIncomeSixRatio());
            jsonObject.put("js-4014", jsLoanStatisticsInfoKycVO.getIncomeGrowthRateOnYear());
            if (ObjectUtil.isNotEmpty(jsLoanStatisticsInfoKycVO.getStatisticsTime())) {
                jsonObject.put("js-statisticsTime", DateUtil.format(jsLoanStatisticsInfoKycVO.getStatisticsTime(), "yyyy-MM-dd HH:mm:ss"));
            }
            jsonObject.put("js-kycNaturalId", jsLoanStatisticsInfoKycVO.getKycNaturalId());
            return ResultUtil.success(jsonObject, ResultEnum.QUERY_SUCCESS);
        } else if (platformStatisticsByKycs.size() == 0) {
            return ResultUtil.success("用户统计信息不在", ResultEnum.QUERY_SUCCESS);
        } else {
            return ResultUtil.error("用户统计信息数据错误", ResultEnum.QUERY_ERROR);
        }
    }

    @Override
    public ResponseMessage getPlatformStoreInfoList(JsLoanStatisticsInfoReq jsLoanStatisticsInfoReq) {
        if (ObjectUtil.isEmpty(jsLoanStatisticsInfoReq.getKycNaturalId())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR.getCode(), "用户 id 不许为空");
        }
        if (ObjectUtil.isEmpty(jsLoanStatisticsInfoReq.getStatisticsTime())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR.getCode(), "统计时间不许为空");
        }
        PageHelper.startPage(jsLoanStatisticsInfoReq);
        List<JsLoanStatisticsInfoStoreVO> result = platformStatisticsByStoreMapper.getPlatformStoreInfoList(jsLoanStatisticsInfoReq);
        List<JSONObject> jsonObjectList = new ArrayList<>();
        for (JsLoanStatisticsInfoStoreVO jsLoanStatisticsInfoStoreVO : result) {
            JSONObject jsonObject = ConvertStoreVOToJSONObject(jsLoanStatisticsInfoStoreVO);
            jsonObjectList.add(jsonObject);
        }
        PageInfo<JsLoanStatisticsInfoStoreVO> pageInfo = new PageInfo<>(result);
        JSONObject pages = new JSONObject();
        pages.put("pageNum", pageInfo.getPageNum());
        pages.put("pageSize", pageInfo.getPageSize());
        pages.put("total", pageInfo.getTotal());
        pages.put("list", jsonObjectList);
        return ResultUtil.success(pages, ResultEnum.QUERY_SUCCESS);
    }

    @Override
    public ResponseMessage getPlatformStoreInfo(JsLoanStatisticsInfoReq jsLoanStatisticsInfoReq) {
        if (ObjectUtil.isEmpty(jsLoanStatisticsInfoReq.getStroeId())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR.getCode(), "店铺 id 不许为空");
        }
        if (ObjectUtil.isEmpty(jsLoanStatisticsInfoReq.getStatisticsTime())) {
            return ResultUtil.error(ResultEnum.QUERY_ERROR.getCode(), "统计时间不许为空");
        }
        List<JsLoanStatisticsInfoStoreVO> result = platformStatisticsByStoreMapper.getPlatformStoreInfoList(jsLoanStatisticsInfoReq);
        if (result.size() == 1) {
            JsLoanStatisticsInfoStoreVO jsLoanStatisticsInfoStoreVO = result.get(0);
            JSONObject jsonObject = ConvertStoreVOToJSONObject(jsLoanStatisticsInfoStoreVO);
            return ResultUtil.success(jsonObject, ResultEnum.QUERY_SUCCESS);
        } else if (result.size() == 0) {
            return ResultUtil.success("无符合条件的店铺信息", ResultEnum.QUERY_SUCCESS);
        } else {
            return ResultUtil.error("店铺信息条目异常。", ResultEnum.QUERY_ERROR);
        }
    }

    private JSONObject ConvertStoreVOToJSONObject(JsLoanStatisticsInfoStoreVO jsLoanStatisticsInfoStoreVO) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("js-4003", jsLoanStatisticsInfoStoreVO.getMonthlySale());
        jsonObject.put("js-4004", jsLoanStatisticsInfoStoreVO.getRefundChainRate());
        jsonObject.put("js-4005", jsLoanStatisticsInfoStoreVO.getAvailableProductNumChainRate());
        jsonObject.put("js-4007", jsLoanStatisticsInfoStoreVO.getSalesTopTenRatio());
        jsonObject.put("js-4008", jsLoanStatisticsInfoStoreVO.getSalesTopThreeRatio());
        jsonObject.put("js-4009", jsLoanStatisticsInfoStoreVO.getMonthlyIncome());
        jsonObject.put("js-4010", jsLoanStatisticsInfoStoreVO.getBindTime());
        jsonObject.put("js-4011", jsLoanStatisticsInfoStoreVO.getIncomeChainRate());
        jsonObject.put("js-4012", jsLoanStatisticsInfoStoreVO.getIncomeThreeRatio());
        jsonObject.put("js-4013", jsLoanStatisticsInfoStoreVO.getIncomeSixRatio());
        jsonObject.put("js-4014", jsLoanStatisticsInfoStoreVO.getIncomeGrowthRateOnYear());
        if (ObjectUtil.isNotEmpty(jsLoanStatisticsInfoStoreVO.getStatisticsTime())) {
            jsonObject.put("js-statisticsTime", DateUtil.format(jsLoanStatisticsInfoStoreVO.getStatisticsTime(), "yyyy-MM-dd HH:mm:ss"));
        }
        jsonObject.put("js-averageIncome", jsLoanStatisticsInfoStoreVO.getAverageIncome());
        jsonObject.put("js-storeId", jsLoanStatisticsInfoStoreVO.getStoreId());
        jsonObject.put("js-storeName", jsLoanStatisticsInfoStoreVO.getStoreName());
        return jsonObject;
    }

    @Override
    public ResponseMessage getPlatformStatisticsInfo(String kycNaturalId) {
        KycStore searchCondition = new KycStore();
        searchCondition.setKycNaturalId(kycNaturalId);
        List<KycStore> kycStores = kycStoreMapper.select(searchCondition);
        JsLoanStatisticsInfoVO result = new JsLoanStatisticsInfoVO();
        if (kycStores.size() > 0) {
            BigDecimal storeNum = new BigDecimal(kycStores.size());
            BigDecimal cancelAuthTotal = new BigDecimal(kycStores.stream()
                    .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.CANCEL_AUTH.ordinal())
                    .count());
            // 取消授权店铺占比
            BigDecimal cancelAuthRatio = cancelAuthTotal.divide(storeNum, 4, RoundingMode.HALF_UP);
            //关闭店铺占比 TODO 暂时提供不了。
//            BigDecimal shutdownRatio;

            BigDecimal authAndAuthWithAccount = new BigDecimal(kycStores.stream()
                            .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.AUTH.ordinal()
                                    || e.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal())
                            .count());
            BigDecimal authAndAuthWithAccountRatio;
            if (authAndAuthWithAccount.intValue() == 0) {
                authAndAuthWithAccountRatio = new BigDecimal(0);
            } else {
                BigDecimal authWithAccountTotal = new BigDecimal(kycStores.stream()
                        .filter(e -> e.getAuthStatus() == StoreAuthStatusEnum.AUTH_WITH_ACCOUNT.ordinal())
                        .count());
                authAndAuthWithAccountRatio = authWithAccountTotal.divide(authAndAuthWithAccount, 4, RoundingMode.HALF_UP);
            }
            result.setCancelAuthRatio(cancelAuthRatio);
//            result.setShutdownRatio(shutdownRatio);
            result.setStoreNum(storeNum);
            result.setAuthAndAuthWithAccount(authAndAuthWithAccount);
            result.setAuthAndAuthWithAccountRatio(authAndAuthWithAccountRatio);
        }

        JSONObject jsonObject = new JSONObject();
        jsonObject.put("js-4001", result.getAuthAndAuthWithAccount());
        jsonObject.put("js-4006", result.getAuthAndAuthWithAccountRatio());
        jsonObject.put("js-cancelAuthRatio", result.getCancelAuthRatio());
        jsonObject.put("js-shutdownRatio", result.getShutdownRatio());
        jsonObject.put("js-storeNum", result.getStoreNum());

        return ResultUtil.success(jsonObject, ResultEnum.QUERY_SUCCESS);
    }

    /**
     * 计算身份证有效期
     */
    private long getIdEffectiveTime(Date startTime, Date endDate) {
        LocalDateTime effectiveStartTime = LocalDateTime.ofInstant(startTime.toInstant(), ZoneId.systemDefault());
        LocalDateTime effectiveEndTime = LocalDateTime.ofInstant(endDate.toInstant(), ZoneId.systemDefault());
        return effectiveStartTime.until(effectiveEndTime, ChronoUnit.MONTHS);
    }

    /**
     * 通过身份证计算年龄
     */
    private long calAge(String idNo) {
        String birth = idNo.substring(6, 14);
        LocalDate birthDay = LocalDate.parse(birth, DateTimeFormatter.BASIC_ISO_DATE);
        return birthDay.until(LocalDate.now(), ChronoUnit.YEARS);
    }

    /**
     * 通过身份计算性别(偶数女，奇数男)
     */
    private int calSex(String idNo) {
        return Integer.parseInt(idNo.substring(16, 17)) % 2;
    }

}
