package com.js.web.service.impl;

import cn.hutool.core.lang.Validator;
import cn.hutool.core.util.CharsetUtil;
import cn.hutool.core.util.HexUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.crypto.asymmetric.KeyType;
import cn.hutool.crypto.asymmetric.RSA;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Maps;
import com.js.api.jspay.service.ApiKycNaturalService;
import com.js.common.JsException.LogicException;
import com.js.common.constant.Constant;
import com.js.common.constant.RegexConstant;
import com.js.common.enums.CertiStatusEnum;
import com.js.common.enums.EmailRemindTypeEnum;
import com.js.common.enums.ResultEnum;
import com.js.common.model.req.*;
import com.js.common.model.vo.KycNaturalVO;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.JjwtUtil;
import com.js.common.util.RedisUtil;
import com.js.common.util.ResultUtil;
import com.js.web.service.IMailService;
import com.js.web.service.JsKycNaturalService;
import com.js.web.validate.code.ValidateCodeException;
import com.js.web.validate.code.ValidateCodeProcessorHolder;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.dubbo.config.annotation.Reference;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Map;
import java.util.Optional;

/**
 * @ClassName: JsUserServiceImpl
 * @Description: 用户服务实现
 * @Author: yjzhao
 * @Create: 2019-05-16  09:16
 **/
@Slf4j
@Service
public class JsKycNaturalServiceImpl implements JsKycNaturalService {

    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    IMailService iMailService;
    @Autowired
    private ValidateCodeProcessorHolder validateCodeProcessorHolder;

    @Reference(
            version = Constant.DUBBO_VERSION,
            application = "${dubbo.application.id}",
            registry = "${dubbo.registry.id}"
    )
    private ApiKycNaturalService apiKycNaturalService;

    @Override
    public ResponseMessage verifyEmailExists(String email) {
        Integer count = apiKycNaturalService.verifyEmailExists(email);
        if (count > 0) {
            return ResultUtil.error(ResultEnum.EMAIL_EXISTS_ERROR.getCode(), ResultEnum.EMAIL_EXISTS_ERROR.getMsg());
        }
        return ResultUtil.success(ResultEnum.SUCCESS.getCode(), ResultEnum.SUCCESS.getMsg());
    }

    @Override
    public ResponseMessage verifyMobileExists(String phoneNo) {
        Integer count = apiKycNaturalService.verifyMobileExists(phoneNo);
        if (count > 0) {
            return ResultUtil.error(ResultEnum.MOBILE_EXISTS_ERROR.getCode(), ResultEnum.MOBILE_EXISTS_ERROR.getMsg());
        }
        return ResultUtil.success(ResultEnum.SUCCESS.getCode(), ResultEnum.SUCCESS.getMsg());
    }

    @Override
    public KycNaturalVO findKycNaturalVOByPhoneNo(String phoneNo) {
        return apiKycNaturalService.findKycNaturalVOByPhoneNo(phoneNo);
    }

    @Override
    public KycNaturalVO findKycNaturalVOByEmail(String email) {
        return apiKycNaturalService.findKycNaturalVOByEmail(email);
    }

    @Override
    public ResponseMessage register(RegiserKycNaturalReq regiserKycNaturalReq, HttpServletRequest request, HttpServletResponse response) {
        log.info("注册信息入参：{}", JSON.toJSONString(regiserKycNaturalReq));
        //校验密码是否正确
        String password = regiserKycNaturalReq.getPassword();
        String phoneNo = regiserKycNaturalReq.getPhoneNo();
        if(!regiserKycNaturalReq.getPassword().equals(regiserKycNaturalReq.getConfirmPassword())){
            return ResultUtil.error(ResultEnum.PWD_REGEX_ERROR_TWO);
        }
        //校验验证码是否正确
        try {
            ValidateCodeReq validateCodeReq = regiserKycNaturalReqConvertToValidateCodeReq(regiserKycNaturalReq);
            validateCodeProcessorHolder.findValidateCodeProcessor("sms").validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),ResultEnum.VALIDATE_CODE_ERROT.getMsg());
        }

        //校验手机号是否已注册
        ResponseMessage verifyMobileExistsResp = verifyMobileExists(phoneNo);
        if (!verifyMobileExistsResp.isSuccess()) {
            return ResultUtil.error(ResultEnum.REGISTE_ERROR.getCode(), ResultEnum.REGISTE_ERROR.getMsg() + ", 手机号已被注册.");

        }

        //保存用户注册信息
        BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
        String pwd = encoder.encode(password);
        RegiserKycNaturalReq saveRegiserKycNaturalReq = new RegiserKycNaturalReq();
        saveRegiserKycNaturalReq.setEmail(regiserKycNaturalReq.getEmail());
        saveRegiserKycNaturalReq.setPassword(pwd);
        saveRegiserKycNaturalReq.setPhoneNo(phoneNo);
        saveRegiserKycNaturalReq.setInviteCode(regiserKycNaturalReq.getInviteCode());
        ResponseMessage registerResp = apiKycNaturalService.register(saveRegiserKycNaturalReq);
        if (!registerResp.isSuccess()) {
            return registerResp;
        }

//        //注册完成发送邮箱激活邮件
//        ValidateCodeVO validateCodeVO = new ValidateCodeVO();
//        try {
//            ValidateCodeReq validateCodeReq = new ValidateCodeReq();
//            validateCodeReq.setEmailAdd(email);
//            Optional<ValidateCodeVO> validateCodeVOOp = validateCodeProcessorHolder.findValidateCodeProcessor("email").create(validateCodeReq, response);
//            if (validateCodeVOOp.isPresent()) {
//                validateCodeVO = validateCodeVOOp.get();
//            }
//        } catch (Exception e) {
//            log.error("发送邮箱激活邮件异常", e);
//        }

        //注册直接登陆
        KycNaturalVO kycNaturalVO = apiKycNaturalService.findKycNaturalVOByPhoneNo(phoneNo);
        if (kycNaturalVO == null) {
            return ResultUtil.error(ResultEnum.REGISTE_ERROR.getCode(), ResultEnum.REGISTE_ERROR.getMsg());
        }

        JSONObject loginSuccJsonVO = getLoginSuccJsonVO(kycNaturalVO);
//        if (StringUtils.isNotBlank(validateCodeVO.getMarkStr())) {
//            loginSuccJsonVO.put("emailMarkStr", validateCodeVO.getMarkStr());
//        }
        return ResultUtil.success(loginSuccJsonVO, ResultEnum.REGISTE_SUCCESS);

    }

    /**
     * @Author yjzhao
     * @Description 登陆成功后获取登陆后需要展示VO
     * @Date 2019/5/28 14:11
     * @Param []
     * @return com.js.common.model.vo.common.ResponseMessage
     **/
    private JSONObject getLoginSuccJsonVO(KycNaturalVO kycNaturalVO) {
        //redis中token标识
        String id = kycNaturalVO.getId();
        Map<String, Object> claimsMap = Maps.newHashMap();
        claimsMap.put(Constant.TOKEN_KEY_NAME, id);
        JSONObject tokenJson = JjwtUtil.generateToken(claimsMap);

        //将用户信息放入redis中
        String kycNaturalVOStr =  JSON.toJSONString(kycNaturalVO);
        redisUtil.setex(Constant.REDIS_JS_WEB_USER + id, Constant.LOGIN_TIMEOUT, kycNaturalVOStr);
        //返回token
        JSONObject jsonObject = new JSONObject();
        jsonObject.putAll(tokenJson);
        jsonObject.put("kycNaturalVO", kycNaturalVOStr);
        return jsonObject;
    }

    @Override
    public ResponseMessage getPwdKey(String phoneNo, HttpServletRequest request, HttpServletResponse response) {
        RSA rsa = new RSA();
        //私钥存入redis中
        String privateKey = rsa.getPrivateKeyBase64();
        //公钥回传给前端
        String publicKey = rsa.getPublicKeyBase64();
        log.info("privateKey:{}, publicKey:{}", privateKey, publicKey);

        redisUtil.setex(Constant.REDIS_JS_WEB_PWD_KEY + phoneNo, Constant.REDIS_JS_WEB_PWD_KEY_TIMEOUT, privateKey);
        Map<String, String> rtnMap = Maps.newHashMap();
        rtnMap.put("pwdKey", publicKey);
        return ResultUtil.success(rtnMap, ResultEnum.GET_PWD_KEY_SUCCESS.getCode(), ResultEnum.GET_PWD_KEY_SUCCESS.getMsg());
    }

    @Override
    public ResponseMessage loginByPwd(LoginKycNaturalReq loginKycNaturalReq, HttpServletRequest request, HttpServletResponse response) {
        String password = loginKycNaturalReq.getPassword();
        String phoneNo = loginKycNaturalReq.getPhoneNo();

        //2. 校验密码并根据登陆账号获取用户
        //3. 如果校验成功将用户信息存入redis中
        ResponseMessage kycNaturalVOResp = apiKycNaturalService.findKycNaturalVOForLogin(phoneNo, password);
        if (!kycNaturalVOResp.isSuccess()) {
            return kycNaturalVOResp;
        }
        KycNaturalVO kycNaturalVO = (KycNaturalVO) kycNaturalVOResp.getData();

        JSONObject loginSuccJsonVO = getLoginSuccJsonVO(kycNaturalVO);

        //校验是否已邮箱认证，如果没有则发送邮箱验证码    去掉登录成功后邮箱校验发送激活码问题
//        if (!kycNaturalVO.getEmailActiveFlag()) {
//            //注册完成发送邮箱激活邮件
//            ValidateCodeVO validateCodeVO = new ValidateCodeVO();
//            try {
//                ValidateCodeReq validateCodeReq = new ValidateCodeReq();
//                validateCodeReq.setEmailAdd(kycNaturalVO.getEmail());
//                Optional<ValidateCodeVO> validateCodeVOOp = validateCodeProcessorHolder.findValidateCodeProcessor("email").create(validateCodeReq, response);
//                if (validateCodeVOOp.isPresent()) {
//                    validateCodeVO = validateCodeVOOp.get();
//                }
//                if (StringUtils.isNotBlank(validateCodeVO.getMarkStr())) {
//                    loginSuccJsonVO.put("emailMarkStr", validateCodeVO.getMarkStr());
//                }
//            } catch (Exception e) {
//                e.printStackTrace();
//            }
//        }
        return ResultUtil.success(loginSuccJsonVO, ResultEnum.LOGIN_SUCCESS);
    }


    @Override
    public ResponseMessage logOut(HttpServletRequest request, HttpServletResponse response) {
        Optional<String> kycId = JjwtUtil.getTokenValue(Constant.TOKEN_KEY_NAME, request);
        if (!kycId.isPresent()) {
            return ResultUtil.success(ResultEnum.LOGOUT_SUCCESS);
        }
        redisUtil.del(Constant.REDIS_JS_WEB_USER + kycId);
        return ResultUtil.success(ResultEnum.LOGOUT_SUCCESS);
    }

    @Override
    public Optional<KycNaturalVO> findKycNaturalVOByWebToken(HttpServletRequest request, HttpServletResponse response) {
        Optional<String> userIdOp = JjwtUtil.getTokenValue(Constant.TOKEN_KEY_NAME, request);
        if (!userIdOp.isPresent()) {
            log.error("未获取到当前登录用户信息，登录已失效");
            throw LogicException.le(ResultEnum.USER_TOKEN_FAILURE);
        }
        String userId = userIdOp.get();
        String userJson = redisUtil.get(Constant.REDIS_JS_WEB_USER + userId);
        if (StrUtil.isBlank(userJson)) {
            log.error("token正确，redis未获取到当前用户信息");
            throw LogicException.le(ResultEnum.USER_TOKEN_FAILURE);
        }
        log.info("redis获取当前登录用户成功");
        try {
            KycNaturalVO kycNaturalVO = JSON.parseObject(userJson, new TypeReference<KycNaturalVO>(){});
            if (kycNaturalVO != null) {
                return Optional.of(kycNaturalVO);
            }
        } catch (Exception e) {
            log.error("获取用户信息异常", e);
        }
        throw LogicException.le(ResultEnum.SYSTEM_ERROR);
    }

    @Override
    public ResponseMessage changePwd(ChPwdReq chPwdReq, HttpServletRequest request, HttpServletResponse response) {
        Optional<KycNaturalVO> kycNaturalVOOp = findKycNaturalVOByWebToken(request, response);
        if (!kycNaturalVOOp.isPresent()) {
            return ResultUtil.error(ResultEnum.CHANGE_PWD_ERROR);
        }
        if(chPwdReq.getNewPwd().equals(chPwdReq.getOldPwd())){
            log.error("新旧密码相同");
            return ResultUtil.error(ResultEnum.PWD_NEW_EQUELS_OLD);
        }
        KycNaturalVO kycNaturalVO = kycNaturalVOOp.get();

        //校验原始密码
        ResponseMessage kycNaturalVOResp = apiKycNaturalService.findKycNaturalVOForLogin(kycNaturalVO.getPhoneNo(), chPwdReq.getOldPwd());
        if (!kycNaturalVOResp.isSuccess()) {
            return ResultUtil.error(ResultEnum.CHANGE_PWD_ERROR.getCode(), ResultEnum.CHANGE_PWD_ERROR.getMsg() + ", 原始密码错误");
        }
        //更改新密码
        BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
        String pwd = encoder.encode(chPwdReq.getNewPwd());
        ChPwdReq saveChPwdReq = new ChPwdReq();
        saveChPwdReq.setKycNaturalId(kycNaturalVO.getId());
        saveChPwdReq.setNewPwd(pwd);
        return apiKycNaturalService.changePwd(saveChPwdReq);
    }

    @Override
    public ResponseMessage changePwd(ChPwdReq chPwdReq) {
        log.info("修改密码入参：{}",chPwdReq);
        if(StringUtils.isBlank(chPwdReq.getPhoneNo()) || StringUtils.isBlank(chPwdReq.getNewPwd()) || StringUtils.isBlank(chPwdReq.getConfirmPwd())){
            log.error("因入参不全,返回错误信息,更改密码失败");
            return ResultUtil.error(ResultEnum.CHANGE_PWD_ERROR);
        }
        if(!chPwdReq.getNewPwd().equals(chPwdReq.getConfirmPwd())){
            log.error("两次密码不一致");
            return ResultUtil.error(ResultEnum.PWD_REGEX_ERROR_TWO);
        }
        KycNaturalVO kycNaturalVOByPhoneNo = apiKycNaturalService.findKycNaturalVOByPhoneNo(chPwdReq.getPhoneNo());
        if(null == kycNaturalVOByPhoneNo || StringUtils.isBlank(kycNaturalVOByPhoneNo.getId())){
            log.error("根据手机号查询用户信息不存在,返回更改密码失败phone:{}",chPwdReq.getPhoneNo());
            return ResultUtil.error(ResultEnum.CHANGE_PWD_ERROR);
        }
        //更改新密码
        BCryptPasswordEncoder encoder = new BCryptPasswordEncoder();
        String pwd = encoder.encode(chPwdReq.getNewPwd());
        ChPwdReq saveChPwdReq = new ChPwdReq();
        saveChPwdReq.setKycNaturalId(kycNaturalVOByPhoneNo.getId());
        saveChPwdReq.setNewPwd(pwd);
        return apiKycNaturalService.changePwd(saveChPwdReq);
    }

    @Override
    public ResponseMessage forgetPwd4Image(ForgetPwdReq forgetPwd, HttpServletRequest request, HttpServletResponse response) {
        String imageCode = forgetPwd.getImageCode();
        if (StrUtil.isEmpty(imageCode)) {
            return ResultUtil.success(ResultEnum.VALIDATE_CODE_EXISTS_ERROR);
        }
        String markStr = forgetPwd.getMarkStr();
        if (StrUtil.isEmpty(markStr)) {
            return ResultUtil.success(ResultEnum.NO_MUST_PARAM_ERROR);
        }
        String phoneNo = forgetPwd.getPhoneNo();
        if (StrUtil.isEmpty(phoneNo)) {
            return ResultUtil.success(ResultEnum.NO_MUST_PARAM_ERROR);
        }
        //校验验证码是否正确
        try {
            ValidateCodeReq validateCodeReq = new ValidateCodeReq();
            validateCodeReq.setMarkStr(markStr);
            validateCodeReq.setImageCode(imageCode);
            validateCodeProcessorHolder.findValidateCodeProcessor("image").validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),ResultEnum.VALIDATE_CODE_ERROT.getMsg());
        }
        ResponseMessage phoneNoExistsResp = phoneNoExists(phoneNo);
        if (!phoneNoExistsResp.isSuccess()) {
            return phoneNoExistsResp;
        }

        return ResultUtil.success(ResultEnum.SUCCESS);
    }

    @Override
    public ResponseMessage forgetPwd4SMS(ForgetPwdReq forgetPwd, HttpServletRequest request, HttpServletResponse response) {
        String smsCode = forgetPwd.getSmsCode();
        if (StrUtil.isEmpty(smsCode)) {
            return ResultUtil.success(ResultEnum.VALIDATE_CODE_EXISTS_ERROR);
        }
        String markStr = forgetPwd.getMarkStr();
        if (StrUtil.isEmpty(markStr)) {
            return ResultUtil.success(ResultEnum.NO_MUST_PARAM_ERROR);
        }
        String phoneNo = forgetPwd.getPhoneNo();
        if (StrUtil.isEmpty(phoneNo)) {
            return ResultUtil.success(ResultEnum.NO_MUST_PARAM_ERROR);
        }
        //校验验证码是否正确
        try {
            ValidateCodeReq validateCodeReq = new ValidateCodeReq();
            validateCodeReq.setMarkStr(markStr);
            validateCodeReq.setSmsCode(smsCode);
            validateCodeProcessorHolder.findValidateCodeProcessor("sms").validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),ResultEnum.VALIDATE_CODE_ERROT.getMsg());
        }

        ResponseMessage phoneNoExistsResp = phoneNoExists(phoneNo);
        if (!phoneNoExistsResp.isSuccess()) {
            return phoneNoExistsResp;
        }

        //跑登陆，返回token
        KycNaturalVO kycNaturalVO = apiKycNaturalService.findKycNaturalVOByPhoneNo(phoneNo);
        if (kycNaturalVO == null) {
            return ResultUtil.success(ResultEnum.USER_NOT_EXIST);
        }

        JSONObject loginSuccJsonVO = getLoginSuccJsonVO(kycNaturalVO);

        return ResultUtil.success(loginSuccJsonVO, ResultEnum.SUCCESS);
    }

    @Override
    public ResponseMessage emailValidate(ValidateCodeReq validateCodeReq, HttpServletRequest request, HttpServletResponse response) {
        if(StringUtils.isBlank(validateCodeReq.getEmailAdd())){
            return ResultUtil.error(ResultEnum.NO_MUST_PARAM_ERROR);
        }
        if (!Validator.isMactchRegex(RegexConstant.EMAIL_REGEX, validateCodeReq.getEmailAdd())) {
            return ResultUtil.error(ResultEnum.EMAIL_REGEX_ERROR);
        }
        //校验邮箱是否已注册
        String email = validateCodeReq.getEmailAdd();
        ResponseMessage verifyEmailExistsResp = verifyEmailExists(email);
        if (!verifyEmailExistsResp.isSuccess()) {
            return ResultUtil.error(ResultEnum.REGISTE_ERROR.getCode(), ResultEnum.REGISTE_ERROR.getMsg() + ", 邮箱已被注册.");

        }
        //校验验证码
        try {
            validateCodeProcessorHolder.findValidateCodeProcessor("email").validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT);
        }

        //更改邮箱验证标识
        Optional<KycNaturalVO> kycNaturalVOOp = findKycNaturalVOByWebToken(request, response);
        if (!kycNaturalVOOp.isPresent()) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT);
        }
        KycNaturalReq kycNaturalReq = new KycNaturalReq();
        kycNaturalReq.setId(kycNaturalVOOp.get().getId());
        kycNaturalReq.setEmailActiveFlag(true);
        kycNaturalReq.setEmail(validateCodeReq.getEmailAdd());
        apiKycNaturalService.updateKycNatural(kycNaturalReq);

        return ResultUtil.success(ResultEnum.VALIDATE_CODE_SUCC);
    }

    @Override
    public KycNaturalVO findKycNaturalVO(KycNaturalVO kycNaturalVO) {
        KycNaturalVO kycNaturalVOReturn = apiKycNaturalService.findKycNaturalVOByPhoneNo(kycNaturalVO.getPhoneNo());
        return kycNaturalVOReturn;
    }

    private ResponseMessage phoneNoExists(String phoneNo) {
        //校验用户是否存在
        if(Validator.isMobile(phoneNo)){
            Integer count = apiKycNaturalService.verifyMobileExists(phoneNo);
            if (count <= 0) {
                return ResultUtil.error(ResultEnum.USER_NOT_EXIST);
            }
        } else if (Validator.isEmail(phoneNo)){
            Integer count = apiKycNaturalService.verifyEmailExists(phoneNo);
            if (count <= 0) {
                return ResultUtil.error(ResultEnum.USER_NOT_EXIST);
            }
        } else {
            //暂不支持其他类型登陆
            return ResultUtil.error(ResultEnum.ERROR.getCode(), ResultEnum.ERROR.getMsg());
        }
        return ResultUtil.success(ResultEnum.SUCCESS);
    }

    /**
     * @Author yjzhao
     * @Description 解密密码
     * @Date 2019/5/19 14:49
     * @Param [phoneNo, pwd]
     * @return java.util.Optional<java.lang.String>
     **/
    private Optional<String> decryptPwd(String phoneNo, String pwd) {

        //从redis中获取pwdkey对应私钥
        String pwdRsaPrivateKey = redisUtil.get(Constant.REDIS_JS_WEB_PWD_KEY + phoneNo);
        if (StrUtil.isEmpty(pwdRsaPrivateKey)) {
            return Optional.empty();
        }
        //解密
        try {
            RSA rsa = new RSA(pwdRsaPrivateKey, null);
            byte[] pwdDecodeHex = HexUtil.decodeHex(pwd);
            byte[] pwdByte = rsa.decrypt(pwdDecodeHex, KeyType.PrivateKey);
            return Optional.of(StrUtil.str(pwdByte, CharsetUtil.CHARSET_UTF_8));
        } catch (Exception e) {
            return Optional.empty();
        }
    }

    private ValidateCodeReq regiserKycNaturalReqConvertToValidateCodeReq (RegiserKycNaturalReq regiserKycNaturalReq) {
        ValidateCodeReq validateCodeReq = new ValidateCodeReq();
        validateCodeReq.setMarkStr(regiserKycNaturalReq.getMarkStr());
        validateCodeReq.setPhoneNo(regiserKycNaturalReq.getPhoneNo());
        validateCodeReq.setSmsCode(regiserKycNaturalReq.getSmsCode());
        return validateCodeReq;
    }

    @Override
    public ResponseMessage changeMoblie(KycSecurityMobileReq kycSecurityMobileReq, KycNaturalVO kycNaturalVO,HttpServletRequest request,  HttpServletResponse response) {
        if(kycNaturalVO.getPhoneNo().equals(kycSecurityMobileReq.getNewMobile())){
            log.error("新手机号与原手机号相同！");
            return ResultUtil.error(ResultEnum.PHONE_NEW_EQUELS_OLD);
        }
        //校验手机验证码
        try {
            ValidateCodeReq validateCodeReq = new ValidateCodeReq();
            validateCodeReq.setPhoneNo(kycSecurityMobileReq.getNewMobile());
            validateCodeReq.setMarkStr(kycSecurityMobileReq.getMarkStr());
            validateCodeReq.setSmsCode(kycSecurityMobileReq.getSmsCode());
            validateCodeProcessorHolder.findValidateCodeProcessor("sms").validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            log.error("手机验证码验证失败");
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT);
        }
        //更改手机号
        KycNaturalReq kycNaturalReq = new KycNaturalReq();
        kycNaturalReq.setId(kycNaturalVO.getId());
        kycNaturalReq.setPhoneNo(kycSecurityMobileReq.getNewMobile());
        apiKycNaturalService.updateKycNatural(kycNaturalReq);

        Optional<String> kycId = JjwtUtil.getTokenValue(Constant.TOKEN_KEY_NAME, request);
        if (!kycId.isPresent()) {
            return ResultUtil.success(ResultEnum.LOGOUT_SUCCESS);
        }
        redisUtil.del(Constant.REDIS_JS_WEB_USER + kycId);
        return ResultUtil.success(ResultEnum.LOGOUT_SUCCESS);
    }

    @Override
    public ResponseMessage changeEmail(KycSecurityEmailReq kycSecurityEmailReq,KycNaturalVO kycNaturalVO, HttpServletResponse response) {
        if(kycNaturalVO.getEmail().equals(kycSecurityEmailReq.getNewEmail())){
            log.error("新手机号与原手机号相同！");
            return ResultUtil.error(ResultEnum.PHONE_NEW_EQUELS_OLD);
        }
        //校验验证码
        try {
            ValidateCodeReq validateCodeReq = new ValidateCodeReq();
            validateCodeReq.setEmailAdd(kycSecurityEmailReq.getNewEmail());
            validateCodeReq.setMarkStr(kycSecurityEmailReq.getMarkStr());
            validateCodeReq.setEmailCode(kycSecurityEmailReq.getEmailCode());
            validateCodeProcessorHolder.findValidateCodeProcessor("email").validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            log.error("邮箱验证码验证失败");
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT);
        }

        //更改邮箱
        KycNaturalReq kycNaturalReq = new KycNaturalReq();
        kycNaturalReq.setId(kycNaturalVO.getId());
        kycNaturalReq.setEmail(kycSecurityEmailReq.getNewEmail());
        apiKycNaturalService.updateKycNatural(kycNaturalReq);

        return ResultUtil.success(ResultEnum.VALIDATE_CODE_SUCC);
    }

    @Override
    public ResponseMessage updateKycNaturalInfo(KycNaturalVO kycNaturalVO) {
        if(!CertiStatusEnum.KYC_INFO_SUBMIT.equals(kycNaturalVO.getCertificationStatus()) && !CertiStatusEnum.CONTACTS_AUDIT_FAILURE.equals(kycNaturalVO.getCertificationStatus())){
            log.error("当前用户状态异常");
            return ResultUtil.error(ResultEnum.ERROR.getCode(),"当前用户状态异常");
        }
        KycNaturalReq kycNaturalReq = new KycNaturalReq();
        kycNaturalReq.setId(kycNaturalVO.getId());
        kycNaturalReq.setCertificationStatus(CertiStatusEnum.SUBMIT_FIRST_TRIAL);
        int updateCount = apiKycNaturalService.updateKycNatural(kycNaturalReq);
        log.info("更新用户表信息条数：{}",updateCount);
        String phoneNo = kycNaturalVO.getPhoneNo();
        String phoneStr = phoneNo.substring(3,7);
        String str = phoneNo.replace(phoneStr,"****");
        String emailContent = "亲，有新用户注册啦，请尽快审核。用户名：" + kycNaturalVO.getCertificationName() + ",手机号：" + str;
        iMailService.sendRemindEmail("用户审核提醒",emailContent , EmailRemindTypeEnum.REMIND_EMAIL);
        return ResultUtil.success(ResultEnum.SUCCESS);
    }
}
