package com.js.web.validate.code.impl;

import com.js.common.enums.ResultEnum;
import com.js.common.model.req.ValidateCodeReq;
import com.js.common.model.vo.ValidateCodeVO;
import com.js.web.validate.code.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import javax.servlet.http.HttpServletResponse;
import java.util.Map;
import java.util.Optional;

/**
 * 验证码抽象类
 */
@Slf4j
public abstract class AbstractValidateCodeProcessor<C extends ValidateCode> implements ValidateCodeProcessor {

    /**
     * 收集系统中所有的 {@link ValidateCodeGenerator} 接口的实现。
     */
    @Autowired
    private Map<String, ValidateCodeGenerator> validateCodeGenerators;

    @Autowired
    private ValidateCodeRepository validateCodeRepository;

    @Override
    public Optional<ValidateCodeVO> create(ValidateCodeReq validateCodeReq, HttpServletResponse response) throws Exception {
        C validateCode = generate(validateCodeReq);
        save(validateCode);
        return send(validateCodeReq, validateCode, response);

    }

    /**
     * 根据Holder返回得CodeProcessor类型来获取验证码的类型
     *
     * @return
     */
    private ValidateCodeType getValidateCodeType() {
        String type = StringUtils.substringBefore(getClass().getSimpleName(), "CodeProcessor");
        return ValidateCodeType.valueOf(type.toUpperCase());
    }

    /**
     * 生成验证码
     *
     * @param validateCodeReq
     * @return
     */
    private C generate(ValidateCodeReq validateCodeReq) {
        String type = getValidateCodeType().toString().toLowerCase();
        String generatorName = type + ValidateCodeGenerator.class.getSimpleName();
        ValidateCodeGenerator validateCodeGenerator = validateCodeGenerators.get(generatorName);
        if (validateCodeGenerator == null) {
            throw new ValidateCodeException("验证码生成器" + generatorName + "不存在");
        }
        return (C) validateCodeGenerator.generate(validateCodeReq);
    }

    /**
     * 保存验证码
     */
    private void save(C validateCode){
        ValidateCode code = new ValidateCode(validateCode.getCode(), validateCode.getMarkStr(), validateCode.getExpireIn());
        validateCodeRepository.save(code, getValidateCodeType());
    }

    /**
     * 发送校验码，由子类实现
     *
     * @param validateCodeReq
     * @param validateCode
     * @throws Exception
     */
    protected abstract Optional<ValidateCodeVO> send(ValidateCodeReq validateCodeReq, C validateCode, HttpServletResponse response) throws Exception;

    @SuppressWarnings("unchecked")
    @Override
    public void validate(ValidateCodeReq validateCodeReq, HttpServletResponse response) {

        ValidateCodeType codeType = getValidateCodeType();

        C codeInRedis = (C) validateCodeRepository.get(validateCodeReq, codeType);

        String codeInRequest = validateCodeReq.getImageCode();
        try {
            if (ValidateCodeType.IMAGE == codeType) {
                codeInRequest = validateCodeReq.getImageCode();
            } else if (ValidateCodeType.SMS == codeType) {
                codeInRequest = validateCodeReq.getSmsCode();
            } else if (ValidateCodeType.EMAIL == codeType) {
                codeInRequest = validateCodeReq.getEmailCode();
            }
        } catch (Exception ex) {
            log.error("获取验证码的值异常",ex);
            throw new ValidateCodeException(ResultEnum.VALIDATE_CODE_ERROT.getCode(),"获取验证码的值失败");
        }

        if (StringUtils.isBlank(codeInRequest)) {
            log.error(codeType + "验证码的值不能为空");
            throw new ValidateCodeException(ResultEnum.VALIDATE_CODE_ERROT.getCode(),"验证码不能为空");
        }

        if (codeInRedis == null || StringUtils.isEmpty(codeInRedis.getCode())) {
            log.error(codeType + "验证码已失效");
            throw new ValidateCodeException(ResultEnum.VALIDATE_CODE_ERROT.getCode(),"验证码已失效");
        }

        if (!StringUtils.equals(codeInRedis.getCode(), codeInRequest)) {
            log.error(codeType + "验证码不匹配");
            throw new ValidateCodeException(ResultEnum.VALIDATE_CODE_ERROT.getCode(),"验证码不匹配");
        }

        validateCodeRepository.remove(validateCodeReq, codeType);

    }
}
