package com.js.web.controller.web;

import com.js.common.constant.Constant;
import com.js.common.enums.ResultEnum;
import com.js.common.model.req.ValidateCodeReq;
import com.js.common.model.vo.ValidateCodeVO;
import com.js.common.model.vo.common.ResponseMessage;
import com.js.common.util.RedisUtil;
import com.js.common.util.ResultUtil;
import com.js.web.validate.code.ValidateCodeException;
import com.js.web.validate.code.ValidateCodeProcessor;
import com.js.web.validate.code.ValidateCodeProcessorHolder;
import com.js.web.validate.code.ValidateCodeType;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.context.request.ServletWebRequest;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Optional;

/**
 * @Author yjzhao
 * @Description 建筑mall 验证码 controller
 * @Date 2019/4/8 15:15
 * @Param
 * @return
 **/
@RestController
@RequestMapping(value = Constant.ApiPath.V1,
        produces = {MediaType.APPLICATION_JSON_UTF8_VALUE})
public class ValidateCodeController {

    @Autowired
    private ValidateCodeProcessorHolder validateCodeProcessorHolder;
    @Autowired
    RedisUtil redisUtil;
    @Value("${jsweb.errormaxcount.smsmaxcount}")
    private Integer maxCount;

    /**
     * 创建验证码，根据验证码类型不同，调用不同的 {@link ValidateCodeProcessor}接口实现
     *
     * @param request code_timestamp:时间戳。若是手机验证码,还需参数 -> mobile: 手机号
     * @param response
     * @param type [image, sms]
     * @throws Exception
     */
    @PostMapping(value = Constant.ApiLoginAuth.ANON + "/code/{type}",
            consumes = {MediaType.APPLICATION_JSON_UTF8_VALUE})
    public ResponseMessage createCode(HttpServletRequest request,
                           HttpServletResponse response,
                           @PathVariable String type,
                           @RequestBody ValidateCodeReq validateCodeReq) throws Exception {
        String redisKey = "";
        int errorCountInteger = 0;
        if(ValidateCodeType.SMS.toString().toLowerCase().equals(type)){
            if(StringUtils.isBlank(validateCodeReq.getPhoneNo())){
                return ResultUtil.error(ResultEnum.NO_MUST_PARAM_ERROR.getCode(),ResultEnum.NO_MUST_PARAM_ERROR.getMsg() + "-手机号不能为空");
            }
            redisKey = Constant.REDIS_JS_USER_SMS + validateCodeReq.getPhoneNo();
            String errorCountString = redisUtil.get(redisKey);
            errorCountInteger = StringUtils.isBlank(errorCountString) ? 0 : Integer.valueOf(errorCountString);
            if(errorCountInteger >= maxCount){
                //如果当前手机号发送超过最大次数且最大次数为取消时，提示信息
                return ResultUtil.error(ResultEnum.SEND_VALIDATE_CODE_ERROR.getCode(), ResultEnum.SEND_VALIDATE_CODE_ERROR.getMsg() + ", 已超过限制次数，请于24小时后重试.");
            }
        }
        Optional<ValidateCodeVO> validateCodeVO = validateCodeProcessorHolder.findValidateCodeProcessor(type).create(validateCodeReq, response);
        if (validateCodeVO.isPresent()) {
            if(ValidateCodeType.SMS.toString().toLowerCase().equals(type)){
                errorCountInteger = errorCountInteger + 1;
                redisUtil.setex(redisKey,Constant.LOGIN_TIMEOUT * 24,String.valueOf(errorCountInteger));
            }
            return ResultUtil.success(validateCodeVO.get(), ResultEnum.SEND_VALIDATE_CODE_SUCCESS);
        }
        return ResultUtil.error(ResultEnum.SEND_VALIDATE_CODE_ERROR);
    }

    /**
     * 校验验证码，根据验证码类型不同，调用不同的 {@link ValidateCodeProcessor}接口实现
     *
     * @param request code_timestamp:时间戳
     * @param response
     * @param type [image, sms]
     * @throws Exception
     */
    @PostMapping(value = Constant.ApiLoginAuth.ANON + "/validateCode/{type}",
            consumes = {MediaType.APPLICATION_JSON_UTF8_VALUE})
    public ResponseMessage validateCode(HttpServletRequest request,
                                        HttpServletResponse response,
                                        @PathVariable String type,
                                        @RequestBody ValidateCodeReq validateCodeReq) throws Exception {

        try {
            validateCodeProcessorHolder.findValidateCodeProcessor(type).validate(validateCodeReq, response);
        } catch (ValidateCodeException exception) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),exception.getMessage());
        } catch (Exception e) {
            return ResultUtil.error(ResultEnum.VALIDATE_CODE_ERROT.getCode(),ResultEnum.VALIDATE_CODE_ERROT.getMsg());
        }

        return ResultUtil.success(ResultEnum.VALIDATE_CODE_SUCC.getCode(),ResultEnum.VALIDATE_CODE_SUCC.getMsg());
    }


}
