package com.js.common.util;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;

import java.util.Collections;

/**
 * 基于 redis 的分布式锁。不可重入。非阻塞
 * 在 redis 下为单节点时没有问题。主从会可能出现锁丢失。
 * 该实现的缺点在于如果 redis 节点故障，会导致丢掉全部的锁。
 * 参考: 关键字 redlock：
 * https://www.w3cschool.cn/redis/redis-yj3f2p0c.html
 * https://wudashan.cn/2017/10/23/Redis-Distributed-Lock-Implement/
 * https://redis.io/topics/distlock
 * http://www.shenyanchao.cn/blog/2018/12/28/argument-on-Redis/
 * http://zhangtielei.com/posts/blog-redlock-reasoning.html
 * http://zhangtielei.com/posts/blog-redlock-reasoning-part2.html
 *
 * @author liutianyu
 */
@Slf4j
@Component
public class RedisDistributedLock {

    @Autowired
    private JedisPool jedisPool;

    private static final String LOCK_SUCCESS = "OK";
    private static final String SET_IF_NOT_EXIST = "NX";
    private static final String SET_WITH_EXPIRE_TIME = "PX";
    private static final Long RELEASE_SUCCESS = 1L;

    /**
     * 尝试获取分布式锁
     * @param lockKey 锁名
     * @param requestId 请求标识。解锁时需要相同才能解锁成功。
     * @param expireTime 超期时间单位毫秒。根据业务情况的不同，需要设置合理超时时间，防止锁提前过期。而业务操作仍未完成。
     * @return 是否获取成功
     */
    public boolean tryGetDistributedLock(String lockKey, String requestId, int expireTime) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            String result = jedis.set(lockKey, requestId, SET_IF_NOT_EXIST, SET_WITH_EXPIRE_TIME, expireTime);
            if (LOCK_SUCCESS.equals(result)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            log.error(e.getMessage());
            return false;
        } finally {
            RedisUtil.returnResource(jedisPool, jedis);
        }
    }

    /**
     * 释放分布式锁
     * @param lockKey 锁
     * @param requestId 请求标识
     * @return 是否释放成功
     */
    public boolean releaseDistributedLock(String lockKey, String requestId) {
        Jedis jedis = null;
        try {
            jedis = jedisPool.getResource();
            String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";
            Object result = jedis.eval(script, Collections.singletonList(lockKey), Collections.singletonList(requestId));
            if (RELEASE_SUCCESS.equals(result)) {
                return true;
            }
            return false;
        } catch (Exception e) {
            log.error(e.getMessage());
            return false;
        } finally {
            RedisUtil.returnResource(jedisPool, jedis);
        }
    }
}
