package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dto.AppDto;
import cn.quantgroup.big.stms.sys.model.App;
import cn.quantgroup.big.stms.sys.service.AppService;
import cn.quantgroup.big.stms.sys.service.PasswordService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @Author meng.xin@quantgroup.cn
 * @Date 2020-01-06 16:13
 * @Version 1.0
 */
@RestController
@RequestMapping("/app")
public class AppController extends BaseController {
    private static final int SECRET_LENGTH = 16;// 秘钥长度
    @Autowired
    private AppService service;
    @Autowired
    private PasswordService passwordService;

    @RequestMapping(value = "/get/{id}", method = RequestMethod.POST)
    public Result get(@PathVariable Integer id) {
        return Result.success(service.findById(id));
    }

    @RequestMapping(value = "/listall", method = RequestMethod.POST)
    public Result listAll() {
        List<App> list = service.findAll(getSort());
        List<App> validList = list.stream().filter(t -> t.getStatus() == BasicDataStatus.VALID).collect(Collectors.toList());
        return Result.success(validList);
    }

    @RequestMapping(value = "/list/{page}/{size}", method = RequestMethod.POST)
    public Result list(@RequestBody App app, @PathVariable int page, @PathVariable int size) {
        PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
        Example<App> example = Example.of(app, getExampleMatcher());
        return Result.success(service.findAll(example, pageRequest));
    }

    @RequestMapping(value = "/addnew", method = RequestMethod.POST)
    public Result addNew(@Valid @RequestBody AppDto appDto) {
        App app = service.findByCode(appDto.getCode());
        if (null != app) {
            throw new BizException("应用编码已存在", ResultCode.PARAM_ERROR);
        }
        app = new App();
        BeanUtils.copyProperties(appDto, app);
        // 设置秘钥
        app.setAppsecret(getSecret(SECRET_LENGTH));
        service.save(app);
        return Result.success();
    }

    @RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
    public Result delete(@PathVariable Integer id) {
        App app = service.findById(id);
        if (null == app) {
            throw new BizException("应用不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.SAVE.equals(app.getStatus())) {
            throw new BizException("新增状态才能删除", ResultCode.PARAM_ERROR);
        }
        service.delete(app);
        return Result.success();
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result update(@Valid @RequestBody AppDto appDto) {
        App app = service.findById(appDto.getId());
        if (null == app) {
            throw new BizException("应用不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.SAVE.equals(app.getStatus())) {
            throw new BizException("新增状态才能修改", ResultCode.PARAM_ERROR);
        }
        // 校验应用编码是否修改
        if (!app.getCode().equals(appDto.getCode())) {
            App temp = service.findByCode(appDto.getCode());
            if (null != temp) {
                throw new BizException("应用编码已存在", ResultCode.PARAM_ERROR);
            }
        }
        BeanUtils.copyProperties(appDto, app);
        service.save(app);
        return Result.success();
    }

    @RequestMapping(value = "/submit/{id}", method = RequestMethod.POST)
    public Result submit(@PathVariable Integer id) {
        App app = service.findById(id);
        if (null == app) {
            throw new BizException("应用不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.SAVE.equals(app.getStatus())) {
            throw new BizException("新增状态才能提交", ResultCode.PARAM_ERROR);
        }
        service.submit(app);
        return Result.success();
    }

    @RequestMapping(value = "/enabled/{id}", method = RequestMethod.POST)
    public Result enabled(@PathVariable Integer id) {
        App app = service.findById(id);
        if (null == app) {
            throw new BizException("应用不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.INVALID.equals(app.getStatus())) {
            throw new BizException("无效状态才能启用", ResultCode.PARAM_ERROR);
        }
        service.enabled(app);
        return Result.success();
    }

    @RequestMapping(value = "/disabled/{id}", method = RequestMethod.POST)
    public Result disabled(@PathVariable Integer id) {
        App app = service.findById(id);
        if (null == app) {
            throw new BizException("应用不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.VALID.equals(app.getStatus())) {
            throw new BizException("有效状态才能禁用", ResultCode.PARAM_ERROR);
        }
        service.disabled(app);
        return Result.success();
    }

    @RequestMapping(value = "/resetsecret/{id}", method = RequestMethod.POST)
    public Result resetSecret(@PathVariable Integer id) {
        App app = service.findById(id);
        if (null == app) {
            throw new BizException("应用不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.VALID.equals(app.getStatus())) {
            throw new BizException("有效状态才能更新秘钥", ResultCode.PARAM_ERROR);
        }
        // 重置秘钥
        app.setAppsecret(getSecret(SECRET_LENGTH));
        service.save(app);
        return Result.success();
    }

    /**
     * 获取秘钥
     *
     * @return
     */
    private String getSecret(int length) {
        return passwordService.getPassword(length);
    }
}
