package cn.quantgroup.big.stms.sys.controller.v2;

import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dao.OrganizationDao;
import cn.quantgroup.big.stms.sys.dto.OrganizationDto;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import cn.quantgroup.big.stms.sys.service.v2.IOrgSerivice;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Example;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Objects;

@RestController
@RequestMapping(value = "/v2/organization", produces = {"application/json"})
@Slf4j
public class OrganizationV2Controller {

  private final OrganizationService organizationService;

  private final IOrgSerivice iOrgSerivice;

  private final OrganizationDao organizationDao;
  @Value("${default.org.id}")
  private String defaultOrgId;

  public OrganizationV2Controller(OrganizationService organizationService,
      IOrgSerivice iOrgSerivice, OrganizationDao organizationDao) {
    this.organizationService = organizationService;
    this.iOrgSerivice = iOrgSerivice;
    this.organizationDao = organizationDao;
  }

  /**
   * 商户创建组织接口
   *
   * @param organizationDto 组织信息
   * @return Result
   */
  @PostMapping(value = "/add")
  public Result addNew(@Valid @RequestBody OrganizationDto organizationDto) {
    log.info("新建组织信息：{}",organizationDto);
    Organization organization = organizationService.findByCode(organizationDto.getCode());
    if (Objects.nonNull(organization)) {
      if(StringUtils.isNotEmpty(organizationDto.getSupplierCode())&&organizationDto.getSupplierCode().equals(organization.getSupplierCode())){
        return Result.success(organization);
      }
      throw new BizException("组织编码已存在_"+organization.getId(), ResultCode.PARAM_ERROR);
    }

    //供应链需要做supplierCode和name重复判断
    if(StringUtils.isNotEmpty(organizationDto.getName())&&StringUtils.isNotEmpty(organizationDto.getSupplierCode())){
       Organization query = new Organization();
       query.setSupplierCode(organizationDto.getSupplierCode());
      query.setName(organizationDto.getName());
       Example<Organization> example = Example.of(query);
       List<Organization> organizationList = organizationService.findAll(example);
       if(CollectionUtils.isNotEmpty(organizationList)){
         return Result.success(organizationList.get(0));
       }
    }


    organization = new Organization();
    BeanUtils.copyProperties(organizationDto, organization);

    /*
      如果是创建商户，挂在固定的羊小咩下面；
      如果不能，父节点的父节点必须是羊小咩。
     */
    if (Objects.isNull(organizationDto.getParent())) {
      Organization parent = new Organization();
      parent.setId(defaultOrgId);
      organization.setParent(parent);
    } else {
      if (StringUtils.isNotEmpty(organizationDto.getParent().getId())) {
        Organization parent = organizationService.findById(organizationDto.getParent().getId());
        if (parent==null || !defaultOrgId.equals(parent.getParent().getId())) {
          throw new BizException("商户不能创建第三级组织", ResultCode.PARAM_ERROR);
        }
      }
    }
    organization.setStatus(BasicDataStatus.VALID);
    return Result.success(organizationService.saveAndSubmit(organization));
  }

  @PostMapping(value = "/delete")
  public Result delete(@RequestParam("id") String id) {
    /*
      羊小咩顶级组织结构不支持删除
     */
    if (defaultOrgId.equals(id)) {
      throw new BizException(ResultCode.UN_DELETE_ORG);
    }
    Organization organization = organizationService.findById(id);
    /*
      供应商级别不支持删除，只能停用
     */
    if (defaultOrgId.equals(organization.getParent().getId())) {
      throw new BizException(ResultCode.UN_DELETE_ORG);
    }
    organizationService.deleteById(id);
    return Result.success();
  }

  /**
   * 冻结组织
   *
   * @param id 供应链id
   * @return 是否成功
   * TODO: 需要判断权限
   */
  @PostMapping(value = "/frozen")
  public Result frozen(@RequestParam("id") String id) {
    /*
      羊小咩顶级组织结构不支持冻结
     */
    if (defaultOrgId.equals(id)) {
      throw new BizException(ResultCode.UN_FROZEN_ORG);
    }

    return Result.success(iOrgSerivice.frozen(id));
  }

  /**
   * 冻结组织
   *
   * @param id 供应链id
   * @return 是否成功
   * TODO: 需要判断权限
   */
  @PostMapping(value = "/unfrozen")
  public Result unFrozen(@RequestParam("id") String id) {
    /*
      羊小咩顶级组织结构不支持冻结
     */
    if (defaultOrgId.equals(id)) {
      throw new BizException(ResultCode.UN_FROZEN_ORG);
    }

    return Result.success(iOrgSerivice.unFrozen(id));
  }

  /**
   * 组织更新
   *
   * @param organizationDto 只能更新组织name和fullName，description
   * @return Result
   */
  @RequestMapping(value = "/update", method = RequestMethod.POST)
  public Result update(@Valid OrganizationDto organizationDto) {
    Organization organization = organizationDao.findOne(organizationDto.getId());
    if (null == organization) {
      throw new BizException("组织机构不存在", ResultCode.PARAM_ERROR);
    }
    organization.setName(organizationDto.getName());
    organization.setDescription(organizationDto.getDescription());
    organization.setFullName(organizationDto.getFullName());
    organizationDao.save(organization);
    return Result.success();
  }

}
