package cn.quantgroup.big.stms.common.service.impl;

import cn.quantgroup.big.stms.common.dao.TreeRepository;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.model.BaseTreeVO;
import cn.quantgroup.big.stms.common.model.TreeNode;
import cn.quantgroup.big.stms.common.service.TreeService;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.jpa.domain.Specification;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;


@SuppressWarnings({"rawtypes", "unchecked"})
public abstract class BaseTreeServiceImpl<T extends BaseTreeVO<T,ID>, ID extends Serializable> extends BaseBasicDataServiceImpl<T, ID> implements TreeService<T, ID> {
	private static Logger logger = LoggerFactory.getLogger(BaseTreeServiceImpl.class);
	public static final String LEVEL_CODE_DELIMITER = "-";

    @Override
    protected void setAddNew(T entity) {
        //树节点相关属性的设置
        if (isRootNode(entity)) {
            //根节点
			entity.setParent(null);
//            try {
//                PropertyUtils.setSimpleProperty(entity, "parent", null);
//            } catch (Exception e) {
//            	logger.error(e.getMessage(), e);
//            }
            int count = getMaxLevelCodeOfChildren(null) + 1;
            entity.setLevelCode(StringUtils.leftPad("" + count, getlevelCodeLength(), "0"));
            entity.setDegree(0);//根结点设置为0
            entity.setIsLeaf(true);
        } else {
            T parent = findById(entity.getParent().getId());
            entity.setParent(parent);
            
//            try {
//                PropertyUtils.setSimpleProperty(entity, "parent", parent);
//            } catch (Exception e) {
//            	logger.error(e.getMessage(), e);
//            }
            
            long count = getMaxLevelCodeOfChildren(parent.getId()) + 1;
            String levelCode = parent.getLevelCode() + LEVEL_CODE_DELIMITER + StringUtils.leftPad("" + count, getlevelCodeLength(), "0");
            entity.setLevelCode(levelCode);
            entity.setDegree(parent.getDegree() + 1);
            entity.setIsLeaf(true);

            if (parent.getIsLeaf()) {
                parent.setIsLeaf(false);
                save(parent);
            }
        }
        super.setAddNew(entity);
    }

    /**
     * 是否根结点
     *
     * @param entity
     * @return
     */
    private boolean isRootNode(T entity) {
        boolean isRoot = true;
        if (null != entity.getParent()
                && null != entity.getParent().getId()
                && StringUtils.isNotEmpty(entity.getParent().getId().toString())) {
            isRoot = false;
        }
        return isRoot;
    }

    protected void setUpdate(T entity) {
        super.setUpdate(entity);

        if (isRootNode(entity)) {
			entity.setParent(null);
//            try {
//                PropertyUtils.setSimpleProperty(entity, "parent", null);
//            } catch (Exception e) {
//            	logger.error(e.getMessage(), e);
//            }
        }

    }

    @Override
    protected void afterDelete(T entity) {
        super.afterDelete(entity);
        if (null != entity.getParent()) {
            T parent = entity.getParent();
            long count = getChildrenSize(parent.getId());
            if (0 == count) {
                parent.setIsLeaf(true);
                save(parent);
            }
        }
    }


    /**
     * <p>date: 2017年6月29日 下午4:41:16</p>
     * <p>description:  获取当前节点到顶层节点的路径</p>
     *
     * @param
     * @return
     * @author rong.yang/阳荣
     */
    public List<T> getPath(ID id) {
        LinkedList<T> aa = new LinkedList<T>();
        T obj = getOne(id);
        aa.addFirst(obj);
        while (null != obj.getParent() && null != obj.getParent().getId()) {
            obj = getOne(obj.getParent().getId());
            aa.addFirst(obj);
        }

        return aa;
    }

    public List<T> getChildren(String parentId) {

        return findAll(new Specification<T>() {
            @Override
            public Predicate toPredicate(Root<T> root, CriteriaQuery<?> query, CriteriaBuilder cb) {
                Predicate predicate = null;
                if (StringUtils.isEmpty(parentId)) {
                    predicate = cb.isNull(root.get("parent").get("id"));
                } else {
                    predicate = cb.equal(root.get("parent").get("id"), parentId);
                }

                return predicate;
            }
        }, new Sort(Direction.ASC, "sortNo"));
    }

    /**
     * 如是 parentId 为null 或空，则返回根节点个数
     *
     * @param parentId
     * @return
     */
    public long getChildrenSize(ID parentId) {
        return count(new Specification<T>() {
            @Override
            public Predicate toPredicate(Root<T> root, CriteriaQuery<?> query, CriteriaBuilder cb) {

                Predicate predicate = null;
                if (null == parentId || StringUtils.isBlank(parentId.toString())) {
                    predicate = cb.isNull(root.get("parent").get("id"));
                } else {
                    predicate = cb.equal(root.get("parent").get("id"), parentId);
                }

                return predicate;
            }
        });
    }

    private int getMaxLevelCodeOfChildren(ID parentId) {
        int maxLevelCodeOfChildren = 0;//默认为零

        String maxLevelCode = null;
        if (null == parentId || StringUtils.isEmpty(parentId.toString())) {
            maxLevelCode = ((TreeRepository) getRepository()).getMaxLevelCodeOfTopNode();
        } else {
            maxLevelCode = ((TreeRepository) getRepository()).getMaxLevelCodeOfChildren(parentId);
        }

        if (StringUtils.isNotEmpty(maxLevelCode)) {
            int index = maxLevelCode.lastIndexOf(LEVEL_CODE_DELIMITER);
            if (-1 != index) {
                maxLevelCode = maxLevelCode.substring(index + 1);
            }

            maxLevelCodeOfChildren = Integer.valueOf(maxLevelCode);
        }

        return maxLevelCodeOfChildren;
    }


    protected void beforeDelete(T entity) {
        long count = getChildrenSize(entity.getId());
        if (count > 0) {
            throw new BizException("该记录还有下级数据，不能删除！");
        }
        super.beforeDelete(entity);
    }

    //获取层次码的长度
    protected int getlevelCodeLength() {
        return 4;
    }

    //TODO 待优化，非递归方法实现
    @Override
    public List<TreeNode<T>> getTree(List<T> list) {
        List<TreeNode<T>> treeNodeList = Lists.newArrayList();
        for (T t : list) {
            if (t.getParent() == null || !list.contains(t.getParent()) ) {
                TreeNode treeNode = new TreeNode();
                treeNode.setNode(t);
                treeNode.setChildNodes(getChild(t, list));
                treeNodeList.add(treeNode);
            }
        }
        return treeNodeList;
    }

    private List<TreeNode<T>> getChild(T t, List<T> list) {
        List<TreeNode<T>> treeNodeList = Lists.newArrayList();
        for (T tt : list) {
            if (tt.getParent() == t) {
                TreeNode<T> treeNode = new TreeNode<>();
                treeNodeList.add(treeNode);
                treeNode.setNode(tt);
                treeNode.setChildNodes(getChild(tt, list));
            }
        }
        return treeNodeList;
    }

}
