package cn.quantgroup.big.stms.sys.service.impl;

import cn.quantgroup.big.stms.common.constants.StmsConstants;
import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.model.TreeNode;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.PermissionService;
import cn.quantgroup.big.stms.common.service.impl.BaseBasicDataServiceImpl;
import cn.quantgroup.big.stms.sys.dao.OrganizationDao;
import cn.quantgroup.big.stms.sys.dao.UserDao;
import cn.quantgroup.big.stms.sys.dao.UserExternalDao;
import cn.quantgroup.big.stms.sys.dao.UserRoleDao;
import cn.quantgroup.big.stms.sys.dao.V2.RoleV2Dao;
import cn.quantgroup.big.stms.sys.dto.*;
import cn.quantgroup.big.stms.sys.model.*;
import cn.quantgroup.big.stms.sys.service.*;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.*;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.Path;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class UserServiceImpl extends BaseBasicDataServiceImpl<User, String> implements UserService {

  @Autowired
  private UserDao userDao;
  @Autowired
  private UserExternalDao userExternalDao;
  @Autowired
  private UserRoleService userRoleService;
  @Autowired
  private RoleResourceService roleResourceService;
  @Autowired
  private ResourceService resourceService;
  @Autowired
  private RoleService roleService;
  @Autowired
  private AppService appService;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private PermissionService permissionService;

  @Autowired
  private UserService userService;

  @Autowired
  private RoleV2Dao roleV2Dao;

  @Autowired
  private EntityManager entityManager;

  @Value("${keystone-operation.https}")
  private String KEYSTONE_OPERATION_HTTPS;

  @Autowired
  private OrganizationDao organizationDao;

  @Autowired
  private ConfigUtils configUtils;

  @Autowired
  private UserRoleDao userRoleDao;

  /**
   * 添加供应商用户权限
   */
  private static final String ADD_SUPPLIER_USER = "/channel/supplier/addSupplierUser";

  @Override
  public UserDao getRepository() {
    return userDao;
  }

  @Override
  public User getByAccount(String account) {
    return userDao.getByAccount(account);
  }

  @Override
  public User getByEmail(String email) {
    return userDao.getByEmail(email);
  }

  @Override
  public User getByMobile(String mobile) {
    return userDao.getByMobile(mobile);
  }

  /**
   * 通过组织id或者suppliercode查询用户列表
   *
   * @param userV2Dto
   * @param pageable
   * @return
   */
  @Override
  public Page<UserDto> findAllByOrganizationDto(UserV2Dto userV2Dto, Pageable pageable) {
    Page<User> userPage;
    Page<UserDto> result;
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();

    if (StringUtils.isNotEmpty(userV2Dto.getOrgId())) {
      userPage = userDao.findAllByOrganizationId(userV2Dto.getOrgId(), pageable);
    } else {
      List<String> orgs = new ArrayList<>();
      orgs.add(onlineUser.getOrgId());
      orgs.addAll(
          organizationDao.findAllByParentId(onlineUser.getOrgId()).stream().map(Organization::getId)
              .collect(Collectors.toList()));
      userPage = userDao.findAllByOrganizationIdIn(orgs, pageable);

    }
    if (StringUtils.isNotEmpty(userV2Dto.getSupplierCode())) {
      userPage = userDao.findAllBySupplierCode(userV2Dto.getSupplierCode(), pageable);
    }
    List<UserDto> users = userPage.getContent().stream()
        .filter(i -> !i.getId().equals(onlineUser.getId())).map(i -> {
          UserDto item = new UserDto();
          item.setId(i.getId());
          item.setAccount(i.getAccount());
          item.setEmail(i.getEmail());
          item.setCode(i.getCode());
          item.setMobile(i.getMobile());
          item.setName(i.getName());
          item.setCreateTime(
              LocalDateTime.ofInstant(i.getCreateTime().toInstant(), ZoneId.systemDefault()));
          item.setStatus(i.getStatus());
          //后续再进行优化，当前jpa返回调整麻烦
          item.setRoles(roleV2Dao.findAllByUserId(i.getId()).stream().map(j -> {
            RoleV2Dto item2 = new RoleV2Dto();
            item2.setId(j.getId());
            item2.setName(j.getName());
            item2.setCode(j.getCode());
            return item2;
          }).collect(Collectors.toList()));
          item.setLastLogin(configUtils.getLastLoginByUserId(i.getId()));

          return item;
        }).collect(Collectors.toList());

    result = new PageImpl<>(users, pageable, userPage.getTotalElements());
    return result;
  }

  @Override
  public Page<User> findAllByOrgChildren(User user, Pageable pageable) {
    // 获取user内的组织机构和子集的list
    List<Organization> orgArr = null;
    if (user.getOrganization() != null && !StringUtils.isBlank(user.getOrganization().getId())) {
      Organization org = organizationService.findById(user.getOrganization().getId());
      if (org != null) {
        orgArr = organizationService.findAll((root, query, cb) -> {
          return cb.like(root.get("levelCode"), org.getLevelCode() + "%");
        });
      }
    }
    if (orgArr != null && orgArr.size() > 1) {
      // list组织大于1个
      final List<Organization> orgs = orgArr;
      return this.findAll((root, query, cb) -> {
        Path<User> path = root.get("organization").get("id");
        CriteriaBuilder.In<Object> in = cb.in(path);
        for (Organization org : orgs) {
          in.value(org.getId());
        }
        return in;
      }, pageable);
    } else {
      // 只有一个或没有时，走原有查询逻辑
      ExampleMatcher matcher = ExampleMatcher.matching() // 构建对象
          .withStringMatcher(ExampleMatcher.StringMatcher.CONTAINING) // 改变默认字符串匹配方式：模糊查询
          .withIgnoreCase(true) // 改变默认大小写忽略方式：忽略大小写
          .withIgnoreNullValues();
      Example<User> example = Example.of(user, matcher);
      return this.findAll(example, pageable);
    }
  }

  @Override
  public boolean checkLogin(String account, String password) {
    User user = getByAccount(account);
    if (null == user) {
      return false;
    }
    password = DigestUtils.sha512Hex(password + user.getSalt());
    if (!user.getPassword().equals(password)) {
      return false;
    }
    return true;
  }

  @Override
  public List<TreeNode<ResourceLiteDto>> getResources(String appCode, Integer tenantId) {
    App app = appService.findByCode(appCode);
    if (null == app) {
      throw new BizException("应用编码有误", ResultCode.PARAM_ERROR);
    }

    // 所有资源
    List<Resource> list = Lists.newArrayList();

    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser.getId().equalsIgnoreCase(StmsConstants.ADMINISTRATOR_ID)) {
      // 超级管理员，显示所有管理功能菜单（需检验管理功能菜单的上级也应为管理菜单）
      Resource res = new Resource();
      res.setIsSuper(true);
//    		res.setApp(app);
      res.setStatus(BasicDataStatus.VALID);
      Example<Resource> res_example = Example.of(res, ExampleMatcher.matching());
      list = resourceService.findAll(res_example, getSortByCreateTimeAES());
    } else {
      List<Role> roleList;
      // 获取当前登录用户在应用中分配的角色
      User user = userService.findById(onlineUser.getId());
      if(user==null){
        roleList = new ArrayList<>();
      }else{
        roleList = getRoles(app.getId(),user);
      }

      // 获取角色分配的资源
      for (Role role : roleList) {
        // 判断角色是不是当前的租户
        if (Objects.nonNull(tenantId) && !Objects.equals(role.getTenantId(), tenantId)) {
          continue;
        }
        List<RoleResource> roleResources = getRoleResources(role);
        for (RoleResource roleResource : roleResources) {
          list.addAll(getList(roleResource.getResource()));
        }
      }
      list = deduplication(list);
      list.sort((o1, o2) -> {
        if (o1.getSortNo().compareTo(o2.getSortNo()) == 0) {
          return o1.getCreateTime().compareTo(o2.getCreateTime());
        } else {
          return o2.getSortNo().compareTo(o1.getSortNo());
        }
      });
    }

    // 将资源组织成树结构
    List<TreeNode<Resource>> resourceTree = resourceService.getTree(list);
    // 转化成一棵精简树
    List<TreeNode<ResourceLiteDto>> liteTree = getResourceLiteTree(resourceTree);

    return liteTree;
  }

  /**
   * 转换为[资源精简版]的树
   *
   * @param resourceTree
   * @return
   */
  private List<TreeNode<ResourceLiteDto>> getResourceLiteTree(
      List<TreeNode<Resource>> resourceTree) {
    // 资源精简版 树List
    List<TreeNode<ResourceLiteDto>> liteTree = Lists.newArrayList();

    for (TreeNode<Resource> resourceTreeNode : resourceTree) {
      // 树
      TreeNode<ResourceLiteDto> liteTreeNode = new TreeNode<>();

      // 节点
      Resource resourceNode = resourceTreeNode.getNode();
      ResourceLiteDto liteNode = new ResourceLiteDto();
      BeanUtils.copyProperties(resourceNode, liteNode);
      liteTreeNode.setNode(liteNode);

      // 孩子
      List<TreeNode<Resource>> resourceChildNodes = resourceTreeNode.getChildNodes();
      List<TreeNode<ResourceLiteDto>> liteChildNodes = getResourceLiteTree(resourceChildNodes);
      liteTreeNode.setChildNodes(liteChildNodes);

      liteTree.add(liteTreeNode);
    }

    return liteTree;
  }

  @Override
  public List<Resource> getResourcesList(Integer appId, User user) {
    List<Resource> list = Lists.newArrayList();
    // 获取当前登录用户在应用中分配的角色
    List<Role> roleList = getRoles(appId, user);
    // 获取角色分配的资源
    for (Role role : roleList) {
      List<RoleResource> roleResources = getRoleResources(role);
      for (RoleResource roleResource : roleResources) {
        list.addAll(getList(roleResource.getResource()));
      }
    }
    return list;
  }

  @Override
  public Map<Integer, List<Resource>> getResourcesTenantList(Integer appId, User user) {
    Map<Integer, List<Resource>> returnMap = new HashMap<>();
//        List<Resource> list = Lists.newArrayList();
    // 获取当前登录用户在应用中分配的角色
    List<Role> roleList = getRoles(appId, user);
    // 获取角色分配的资源
    for (Role role : roleList) {
      List<Resource> list = returnMap.get(role.getTenantId());
      if (CollectionUtils.isEmpty(list)) {
        list = Lists.newArrayList();
        returnMap.put(role.getTenantId(), list);
      }
      List<RoleResource> roleResources = getRoleResources(role);
      for (RoleResource roleResource : roleResources) {
        list.addAll(getList(roleResource.getResource()));
      }
    }
    return returnMap;
  }

  /**
   * 获取当前登录用户，指定应用中分配的角色
   *
   * @param appId
   * @return
   */
  private List<Role> getRoles(Integer appId) {
    return getRoles(appId, null);
  }

  /**
   * 获取指定用户，指定应用中分配的角色
   *
   * @param appId
   * @return
   */
  private List<Role> getRoles(Integer appId, User user) {
    // 获取用户所有分配角色
    List<UserRole> userRoles = getUserRoles(user);
    // 筛选出指定应用的角色
    List<Role> roleList = Lists.newArrayList();
    for (UserRole userRole : userRoles) {
      Role role = userRole.getRole();
      if (role.getApp().getId() == appId) {
        roleList.add(role);
      } else if (null == appId) {
        roleList.add(role);
      }
    }
    return roleList;
  }

  /**
   * 获取指定用户所有角色
   *
   * @return
   */
  private List<UserRole> getUserRoles(User user) {
    return getUserRoles(null, user);
  }

  /**
   * 获取指定用户所有角色, 可按应用过滤
   *
   * @return
   */
  @Override
  public List<UserRole> getUserRoles(Integer appId, User user) {
    if (null == user) {
      OnlineUser onlineUser = AppContextHolder.getOnlineUser();
      user = findById(onlineUser.getId());
    }
    UserRole exampleUr = new UserRole();
    exampleUr.setUser(user);
    Example<UserRole> example = Example.of(exampleUr);
    List<UserRole> userRoles = userRoleService.findAll(example);
    // 按应用过滤
    if (appId != null && appId != 0) {
      List<UserRole> roleList = Lists.newArrayList();
      for (UserRole userRole : userRoles) {
        if (userRole.getRole().getApp().getId() == appId) {
          roleList.add(userRole);
        }
      }
      return roleList;
    }
    return userRoles;
  }

  /**
   * 获取角色分配的资源
   *
   * @param role
   * @return
   */
  private List<RoleResource> getRoleResources(Role role) {
    RoleResource roleResource = new RoleResource();
    roleResource.setRole(role);
    Example<RoleResource> roleResourceExample = Example.of(roleResource);
    return roleResourceService.findAll(roleResourceExample);
  }

  /**
   * 获取资源以及所有上级资源
   *
   * @param resource
   * @return
   */
  private List<Resource> getList(Resource resource) {
    List<Resource> list = Lists.newArrayList();
    list.add(resource);
    Resource parent = resource.getParent();
    if (null != parent) {
      list.addAll(getList(parent));
    }
    return list;
  }

  /**
   * 去重
   *
   * @param list
   * @return
   */
  private List<Resource> deduplication(List<Resource> list) {
    // 根据id去重
    return list.stream().collect(Collectors.collectingAndThen(
        Collectors.toCollection(() -> new TreeSet<>(Comparator.comparing(Resource::getId))),
        ArrayList::new));
  }

  @Override
  @Transactional
  public void updateUserRole(UserRoleDto userRoleDto) {
    // 检查用户当前角色列表, 限定appid范围（可以为null）
    List<UserRole> cur_roles = getUserRoles(userRoleDto.getApp().getId(), userRoleDto.getUser());
    if (userRoleDto.getSelectedRoleIds().size() == 0) {
      // 清空用户所有角色
      userRoleService.deleteInBatch(cur_roles);
    } else {
      // 检查用户目标角色列表
      Iterable<Role> selected_roles = roleService.findAllById(userRoleDto.getSelectedRoleIds());
      Set<String> selected_roleIds = new HashSet<>();
      for (Role sr : selected_roles) {
//                if (sr.getApp().getId() != userRoleDto.getApp().getId()) {
//					throw new BizException("目标权限与操作应用ID不符合", ResultCode.PARAM_ERROR);
//                    continue;
//                }
        selected_roleIds.add(sr.getId());
      }
      // 对比集合，记录删除用户角色
      if (cur_roles.size() > 0) {
        List<UserRole> del_roles = new ArrayList<UserRole>();
        for (UserRole ur : cur_roles) {
          if (selected_roleIds.contains(ur.getRole().getId())) {
            selected_roleIds.remove(ur.getRole().getId());
          } else {
            del_roles.add(ur);
          }
        }
        userRoleService.deleteInBatch(del_roles);
      }
      // 对比后剩余选中角色selected_roleIds执行新增操作
      if (selected_roleIds.size() > 0) {
        List<UserRole> add_roles = new ArrayList<UserRole>();
        for (String roleId : selected_roleIds) {
          Role r = new Role();
          r.setId(roleId);
          UserRole newUr = new UserRole();
          newUr.setRole(r);
          newUr.setUser(userRoleDto.getUser());
          newUr.setCreator(AppContextHolder.getOnlineUser().getId());
          newUr.setCreateTime(new Date());
          add_roles.add(newUr);
        }
        userRoleService.saveAll(add_roles);
      }
    }
    // 清理用户权限缓存
    permissionService.clearPermission(userRoleDto.getUser().getId());
  }

  @Override
  public List<String> relevantUser(Role... roles) {
    List<String> list = new ArrayList<>();
    if (roles != null) {
      // 获取关联表
      List<UserRole> userRoles = null;
      if (roles.length == 1) {
        userRoles = userRoleService.findAllByRoleId(roles[0].getId());
        for (UserRole userRole : userRoles) {
          list.add(userRole.getUser().getId());
        }
      } else {
        userRoleService.findAll((root, query, cb) -> {
          Path<UserRole> path = root.get("role").get("id");
          CriteriaBuilder.In<Object> in = cb.in(path);
          for (Role role : roles) {
            in.value(role.getId());
          }
          return in;
        });
      }
      // 提取userId
      if (userRoles != null) {
        for (UserRole userRole : userRoles) {
          list.add(userRole.getUser().getId());
        }
      }
    }
    return list;
  }

  @Override
  public List<String> relevantUser(Resource... resources) {
    List<String> list = new ArrayList<>();
    if (resources != null) {
      List<RoleResource> roleResources = null;
      if (resources.length == 1) {
        RoleResource exampleRr = new RoleResource();
        exampleRr.setResource(resources[0]);
        Example<RoleResource> example = Example.of(exampleRr);
        roleResources = roleResourceService.findAll(example);
      } else {
        roleResources = roleResourceService.findAll((root, query, cb) -> {
          Path<RoleResource> path = root.get("resource").get("id");
          CriteriaBuilder.In<Object> in = cb.in(path);
          for (Resource resource : resources) {
            in.value(resource.getId());
          }
          return in;
        });
      }
      // 根据角色，提取userId
      if (roleResources != null) {
        List<Role> roles = new ArrayList<>();
        for (RoleResource roleResource : roleResources) {
          roles.add(roleResource.getRole());
        }
        list = relevantUser(roles.toArray(new Role[0]));
      }
    }
    return list;
  }

  @Override
  public Set<App> getApps(Integer tenantId) {
    Set<App> set = Sets.newHashSet();

    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser.getId().equalsIgnoreCase(StmsConstants.ADMINISTRATOR_ID)) {
      // 管理员只返回权限系统
      App app = appService.findByCode(StmsConstants.AUTH_APP_CODE);
      set.add(app);
    } else {
      // [普通用户]获取当前登录用户的所有角色
      User user = findById(onlineUser.getId());
      List<UserRole> userRoles = getUserRoles(user);
      // 获取用户所有分配的应用
      for (UserRole userRole : userRoles) {
        Role role = userRole.getRole();
        if (Objects.equals(tenantId, role.getTenantId()) && BasicDataStatus.VALID.equals(
            role.getStatus())) {
          App app = role.getApp();
          if (BasicDataStatus.VALID.equals(app.getStatus())) {
            set.add(app);
          }
        }
      }

      // 为app系统添加角色 比较low 也没想到好用的 感觉List<Map<App,List<Role>>>也够复杂
      for (App app : set) {
        List<String> roles = Lists.newArrayList();
        for (UserRole userRole : userRoles) {
          Role role = userRole.getRole();
          if (BasicDataStatus.VALID.equals(role.getStatus()) && Objects.equals(tenantId,
              role.getTenantId())) {
            if (app.equals(role.getApp())) {
              roles.add(role.getCode());
            }
          }
        }
        app.setRoles(roles);
      }
    }

    return set;
  }

  @Override
  @Transactional
  public void saveUser(User user, String accessToken) {
    this.save(user);
    // 如果是 设置羊小咩商品供应商(a6ed0680-849d-4821-bb8e-c3398ddb1b5a) 下添加用户，需要往供应链 传一份数据
    Organization org = organizationService.findById(user.getOrganization().getId());

    if (Objects.nonNull(org.getParent()) && (
        "a6ed0680-849d-4821-bb8e-c3398ddb1b5a".equals(org.getParent().getId())
            || "a6ed0680-849d-4821-bb8e-c3398ddb1b5a".equals(user.getOrganization().getId()))) {

      if (Objects.isNull(org.getSupplierCode())) {
        throw new BizException("此供应商不存在，请先添加供应商", ResultCode.SUPPLIER_NON_EXISTENT);
      }

      String path = KEYSTONE_OPERATION_HTTPS + ADD_SUPPLIER_USER;
      // 1 创建okhttp客户端对象
      OkHttpClient client = new OkHttpClient();
      RequestBody body = new FormBody.Builder().add("supplierCode", org.getSupplierCode())
          .add("userId", user.getId()).build();

      // 2 request 默认是get请求
      Request request = new Request.Builder().addHeader("access-token", accessToken).post(body)
          .url(path).build();
      // 3 进行请求操作
      JSONObject object = null;
      try {
        Response response = client.newCall(request).execute();
        // 4 判断是否请求成功
        if (response.isSuccessful()) {
          // 得到响应体中的身体,将其转成  string
          String string = response.body().string();
          log.info("[stms][UserServiceImpl][saveUser]调用供应链接口={},response={}", path, string);
          object = JSONObject.parseObject(string);
        } else {
          log.info("[stms][UserServiceImpl][saveUser]调用供应链接口={},response={}", path,
              JSONObject.toJSONString(response));
        }

      } catch (Exception e) {
        log.error("[stms][OrganizationServiceImpl][validateSupplierName]查询供应链商户存在不存在:",
            e);
      }
      if (Objects.nonNull(object)) {
        Object businessCode = object.get("businessCode");
        if ("3010".equals(businessCode.toString())) {
          throw new BizException("此供应商不存在，请先添加供应商", ResultCode.SUPPLIER_NON_EXISTENT);
        }
      } else {
        throw new BizException("此供应商不存在，请先添加供应商", ResultCode.SUPPLIER_NON_EXISTENT);
      }
    }
  }

  /**
   * 商户新增主账户 需要做与第三方账户关联
   *
   * @param user
   */
  @Override
  @Transactional
  public void addMasterAccount(User user, String externalId, List<String> roleIds) {
    userDao.save(user);
    /**
     * 更新第三方用户id,主账户和第一次的第三方登录用户共用同样的uuid
     */
    if (StringUtils.isNotEmpty(externalId)) {
      UserExternal userExternal = userExternalDao.findOne(externalId);
      if(userExternal!=null){
        userExternalDao.updateUserId(externalId, user.getId());
        if (StringUtils.isEmpty(userExternal.getSupplierCode()) && StringUtils.isNotEmpty(
            user.getSupplierCode())) {
          userExternalDao.bindSupplierCode(userExternal.getId(), user.getSupplierCode());
        }
      }
    }

    if (CollectionUtils.isNotEmpty(roleIds)) {
      roleIds.stream().distinct().forEach(roleId -> {
        UserRoleDto userRoleDto = new UserRoleDto();
        userRoleDto.setUser(user);
        Role role = roleService.findById(roleId);

        if (role == null) {
          throw new BizException(ResultCode.ROLE_NOT_EXISTENT);
        }
        userRoleDto.setApp(role.getApp());

        userRoleDto.setSelectedRoleIds(Collections.singletonList(roleId));
        userService.updateUserRole(userRoleDto);
      });
    }
  }

  @Override
  @Transactional
  public void addAccount(User user, List<String> roleIds) {
    userDao.save(user);
    if (CollectionUtils.isNotEmpty(roleIds)) {
      UserRoleDto userRoleDto = new UserRoleDto();
      userRoleDto.setUser(user);
      Role role = roleService.findById(roleIds.get(0));

      if (role == null) {
        throw new BizException(ResultCode.ROLE_NOT_EXISTENT);
      }
      userRoleDto.setApp(role.getApp());
      userRoleDto.setSelectedRoleIds(roleIds);
      userService.updateUserRole(userRoleDto);
    }
  }

  @Override
  public void updateAccount(User user, List<String> roleIds) {
    userDao.save(user);
    if (CollectionUtils.isNotEmpty(roleIds)) {
      UserRoleDto userRoleDto = new UserRoleDto();
      userRoleDto.setUser(user);
      Role role = roleService.findById(roleIds.get(0));

      if (role == null) {
        throw new BizException(ResultCode.ROLE_NOT_EXISTENT);
      }
      userRoleDto.setApp(role.getApp());
      userRoleDto.setSelectedRoleIds(roleIds);

      userService.updateUserRole(userRoleDto);

      // 清理用户权限缓存
      permissionService.clearPermission(userRoleDto.getUser().getId());

    } else {
      userRoleDao.deleteAllByUserId(user.getId());
    }
  }
}
