package cn.quantgroup.big.stms.sys.controller.v2;

import cn.quantgroup.big.stms.common.config.IAuthentication;
import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.enums.TenantEnum;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.PermissionService;
import cn.quantgroup.big.stms.common.utils.Constants;
import cn.quantgroup.big.stms.common.utils.IpAddrUtils;
import cn.quantgroup.big.stms.sys.dao.ConfigDao;
import cn.quantgroup.big.stms.sys.dao.UserExternalDao;
import cn.quantgroup.big.stms.sys.dto.TokenDto;
import cn.quantgroup.big.stms.sys.model.Config;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.model.UserExternal;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import cn.quantgroup.big.stms.sys.service.UserService;
import cn.quantgroup.big.stms.sys.service.remote.WechatRemoteService;
import cn.quantgroup.big.stms.sys.service.remote.model.CodeUnlimitVo;
import cn.quantgroup.big.stms.sys.service.v2.IUserService;
import cn.quantgroup.big.stms.sys.service.v2.strategy.AccountStrategy;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.Length;
import org.hibernate.validator.constraints.NotBlank;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.constraints.NotNull;
import java.util.List;
import java.util.Objects;
import java.util.UUID;

@RestController
@RequestMapping(value = "/v2/oauth", produces = {"application/json"})
@Slf4j
public class OauthV2Controller {

  private final AccountStrategy accountStrategy;
  private final IUserService iUserService;
  private final UserService userService;

  private final ConfigUtils configUtils;
  private OrganizationService organizationService;

  @Autowired
  private WechatRemoteService wechatRemoteService;

  @Autowired
  private ConfigDao configDao;

  @Autowired
  private UserExternalDao userExternalDao;

  @Autowired
  private PermissionService permissionService;
  @Autowired
  @Qualifier("LDAPAuthentication")
  private IAuthentication authentication;
  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  public OauthV2Controller(AccountStrategy accountStrategy, IUserService iUserService,
      UserService userService, ConfigUtils configUtils, OrganizationService organizationService) {
    this.accountStrategy = accountStrategy;
    this.iUserService = iUserService;
    this.userService = userService;
    this.configUtils = configUtils;
    this.organizationService = organizationService;
  }

  /**
   * 通过微信code进行登录
   *
   * @param code     微信登录返回的code
   * @param appId    第三方应用appId
   * @param tenantId 租户  id,默认值560761
   * @return Result
   * @see "https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/login/auth.code2Session.html"
   */
  @GetMapping(value = "/wechat/login")
  @ResponseBody
  public Result login(@RequestParam(value = "tenantId", defaultValue = "560761") Integer tenantId,
      @NotNull @RequestParam("appId") String appId, @NotNull @RequestParam("code") String code
  ) {
    try {
      return Result.success(accountStrategy.loginByCode(code, appId, tenantId));
    } catch (BizException ex) {
      return Result.failure(ex.getResultCode(), ex.getMessage());
    }
  }


  /**
   * 原有用户名密码登录验证码
   *
   * @param request
   * @throws Exception
   */
  @PostMapping("/login")
  @ResponseBody
  public Result login(@NotNull @RequestParam(name = "username") String username,
      @NotNull @RequestParam(name = "password") String password,
      @RequestParam(name = "verifycode", required = false) String verifycode,
      HttpServletRequest request) throws Exception {
    log.info("用户[{}]登录", username);
    if(!configUtils.checkLoginFailTimes(username)){
      return Result.failure(ResultCode.FAIL_LOGIN_MAX);
    }

    if (StringUtils.isNotBlank(verifycode)) {
      if (!username.equals(
          stringRedisTemplate.opsForValue().get(Constants.CAPTCHA_KEY + username))) {
        return Result.failure(ResultCode.INVALID_VERIFICATION_CODE);
      }
    }

    //判断用户是否存在
    User user = userService.getByAccount(username);
    if (null == user) {
      configUtils.addLoginFailTimes(username);
      return Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD);
    }

    if (!BasicDataStatus.VALID.getValue().equals(user.getStatus().getValue())) {
      return Result.failure(ResultCode.FORBIDDEN, "请联系管理员开通权限！");
    }

    if (!user.getIsEnable()) {
      return Result.failure(ResultCode.FORBIDDEN, "请联系管理员开通权限！");
    }

    //验证密码是否正确
    boolean isPass = userService.checkLogin(username, password);
    if (!isPass) {
      //登录验证
      isPass = authentication.authenticate(username, password);
      if (!isPass) {
        configUtils.addLoginFailTimes(username);
        return Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD);
      }
    }

    // 同一个用户，不同的系统，公用一个token
    // 根据用户id获取 用户token
    String accessToken = stringRedisTemplate.opsForValue().get(Constants.TOKEN_KEY + user.getId());

    if (StringUtils.isBlank(accessToken)) {
      accessToken = UUID.randomUUID().toString();
    }
    String refreshToken = UUID.randomUUID().toString();
    long now = System.currentTimeMillis();
    Long atExpire = now + Constants.ACCESS_TOKEN_EXPIRE * 1000L;
    Long rtExpire = now + Constants.REFRESH_TOKEN_EXPIRE * 1000L;

    String orgId = "";
    String orgName = "";
    String supplierCode = user.getSupplierCode();
    if (null != user.getOrganization()) {
      Organization current = organizationService.findById(user.getOrganization().getId());

      if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
        return Result.failure(ResultCode.ORGANIZATION_FROZEN, "请联系管理员开通权限！");
      }

      orgId = user.getOrganization().getId();
      orgName = user.getOrganization().getName();


      if(StringUtils.isEmpty(supplierCode) && StringUtils.isNotEmpty(user.getOrganization().getId())){
        Organization organization = organizationService.findById(user.getOrganization().getId());
        supplierCode = organization.getSupplierCode();
      }
    }

    String ip = IpAddrUtils.getIpAddr(request);



    OnlineUser onlineUser = new OnlineUser(user.getId(), user.getAccount(), user.getName(), orgId,
        orgName, ip, String.valueOf(user.getTenantId()), supplierCode);
    UserExternal userExternal = userExternalDao.findByUserId(user.getId());
    if (userExternal != null) {
      onlineUser.setBindStatus(true);
    }

    TokenDto tokenDto = new TokenDto();
    tokenDto.setAccessToken(accessToken);
    tokenDto.setAtExpire(atExpire.toString());
    tokenDto.setRefreshToken(refreshToken);
    tokenDto.setRtExpire(rtExpire.toString());
    configUtils.saveLoginToken(onlineUser, tokenDto);

    permissionService.refreshPermission(user.getId(), null);

    return Result.success(tokenDto);
  }

  @RequestMapping("/refreshtoken")
  public Result refreshtoken(@NotNull @RequestParam(name = "refreshtoken") String refreshToken) {
    try {
      return Result.success(accountStrategy.refreshToken(refreshToken));
    } catch (BizException ex) {
      return Result.failure(ex.getResultCode(), ex.getMessage());
    }
  }

  @PostMapping(value = "/currentuserinfo", produces = {"application/json"})
  public Result currentUserInfo() {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (StringUtils.isNotEmpty(onlineUser.getOrgId())) {
      Organization current = organizationService.findById(onlineUser.getOrgId());
      if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
        return Result.failure(ResultCode.ORGANIZATION_FROZEN, "请联系管理员开通权限！");
      }
    }
    if (StringUtils.isEmpty(onlineUser.getOrgId()) && StringUtils.isNotEmpty(
        onlineUser.getSupplierCode())) {
      List<Organization> current = organizationService.findBySupplierCode(
          onlineUser.getSupplierCode());
      if (current.stream().anyMatch(i -> BasicDataStatus.INVALID.equals(i.getStatus()))) {
        throw new BizException("请联系管理员开通权限！", ResultCode.ORGANIZATION_FROZEN);
      }
    }

    return Result.success(onlineUser);
  }

  /**
   * 微信绑定手机号码
   *
   * @param tenantId 租户id ，默认值560761
   * @param appId    微信小程序appid
   * @param code     绑定手机的code
   * @return Result
   */
  @PostMapping("/wechat/bindphone")
  public Result bindPhone(
      @RequestParam(value = "tenantId", defaultValue = "560761") Integer tenantId,
      @NotNull @RequestParam("appId") String appId, @NotNull @RequestParam("code") String code) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    boolean result = iUserService.bindPhone(onlineUser.getId(), tenantId, appId, code);
    if (result) {
      return Result.success(true);
    } else {
      return Result.failure(ResultCode.BIND_PHONE_ERROR);
    }
  }

  /**
   * 用户与供应商关联
   *
   * @param userId       用户id
   * @param supplierCode 供应商id
   * @return 成功true；失败false
   */
  @GetMapping(value = "/binding")
  public Result bind(@RequestParam("userId") String userId,
      @RequestParam("supplierCode") String supplierCode) {
    return Result.success(iUserService.bind(userId, supplierCode));
  }

  /**
   * 通过用户手机号码获取用户登录信息
   *
   * @param tenantId 租户id
   * @param appId    微信appid
   * @param phone    手机号码
   * @return Result
   */
  @GetMapping(value = "/userinfo")
  public Result userInfoByPhone(@RequestParam("tenantId") int tenantId,
      @RequestParam("appId") String appId, @RequestParam("phone") String phone) {
    try {
      log.info("userInfoByPhone:{},appId:{},tenantId:{}",phone,appId,tenantId);
      return Result.success(iUserService.getOnlineUserByPhone(tenantId, appId, phone));
    } catch (BizException ex) {
      log.info("userInfoByPhone:{}",ex.getMessage(),ex);
      return Result.failure(ex.getResultCode());
    }
  }


  @PostMapping(value = "/getBindCode", produces = MediaType.IMAGE_JPEG_VALUE)
  public @ResponseBody byte[] getWxaCodeUnlimit(@RequestParam("appId") String appId,
      @RequestParam("tenantId") Integer tenantId,
      @Length(max = 32) @RequestParam("scene") String scene,
      @RequestParam(value = "env", defaultValue = "release") String env,
      @RequestParam("page") String page) {
    Config config = configDao.findByTenantIdAndAppId(tenantId, appId);
    CodeUnlimitVo vo = new CodeUnlimitVo();
    vo.setPage(page);
    vo.setScene(scene);
    vo.setCheck_path(true);
    vo.setEnv_version(env);

    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    configUtils.saveBindingCode(scene, onlineUser, config);
    return wechatRemoteService.getWxaCodeUnlimit(configUtils.getExternalAccessToken(config.getId()),
        vo);
  }


  /**
   * 微信绑定扫码用户接口
   * @param code  微信登录产生的code
   * @param scene 请求getWxaCodeUnlimit给的场景码
   * @return 成功true；失败false
   */
  @GetMapping(value = "/bindingCode")
  public Result bindCode(@RequestParam("code") String code,
      @RequestParam("scene") String scene) {
    return Result.success(iUserService.bindCode(code, scene));
  }


  /**
   * 查询绑定结果扫码用户接口（前端用于判断是否绑定成功使用）
   *
   * @param scene 请求getWxaCodeUnlimit给的场景码
   * @return Result
   */
  @GetMapping(value = "/queryBindStatus")
  public Result queryBindStatus(
      @RequestParam("scene") String scene) {
    OnlineUser current = AppContextHolder.getOnlineUser();

    OnlineUser onlineUser = configUtils.getBindingScene(scene);
    if (onlineUser == null) {
      return Result.failure(ResultCode.UN_BIND_CODE);
    }
    //只能当前账户查询该账户的绑定情况
    if (!current.getId().equals(onlineUser.getId())) {
      return Result.failure(ResultCode.FORBIDDEN);
    }

    if (current.isBindStatus()) {
      return Result.success();
    } else {
      return Result.failure(ResultCode.UN_BIND_CODE);
    }
  }


  /**
   * 通过服务转发获取鉴权【给外部用的】 判断请求的URL是否有权限
   *
   * @param referer  请求URL
   * @param tenantId 租户id
   * @return
   */
  @GetMapping("/permission")
  @ResponseBody
  public Result checkPermission(@RequestHeader(value = "Referer") @NotBlank String referer,
      Integer tenantId) {
    if (Objects.isNull(tenantId)) {
      tenantId = TenantEnum.KDSP.getTenantId();
    }
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser == null) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }else{
      if(StringUtils.isNotEmpty(onlineUser.getOrgId())){
        Organization current = organizationService.findById(onlineUser.getOrgId());
        if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
          return Result.failure(ResultCode.ORGANIZATION_FROZEN, "请联系管理员开通权限！");
        }
      }
    }
    log.info("用户:{},请求路径:{}", onlineUser.getName(), referer);
    if (permissionService.checkUri(tenantId, referer)) {
      // 通过redis中缓存的权限匹配
      return Result.success(onlineUser.toJson());
    } else {
      return Result.failure(ResultCode.FORBIDDEN);
    }
  }
}
