package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dto.RoleDto;
import cn.quantgroup.big.stms.sys.dto.RoleResourceDto;
import cn.quantgroup.big.stms.sys.model.Resource;
import cn.quantgroup.big.stms.sys.model.Role;
import cn.quantgroup.big.stms.sys.model.RoleResource;
import cn.quantgroup.big.stms.sys.service.ResourceService;
import cn.quantgroup.big.stms.sys.service.RoleResourceService;
import cn.quantgroup.big.stms.sys.service.RoleService;
import cn.quantgroup.big.stms.sys.service.TenantService;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.List;

@RestController
@RequestMapping("/role")
@Slf4j
public class RoleController extends BaseController {

  @Autowired
  private RoleService service;
  @Autowired
  private RoleResourceService roleResourceService;
  @Autowired
  private ResourceService resourceService;

  @Autowired
  private TenantService tenantService;

  @RequestMapping(value = "/get/{id}", method = RequestMethod.POST)
  public Result get(@PathVariable String id) {
    return Result.success(service.findById(id));
  }

  /**
   * 查询指定系统所有角色（目前应该用不到）
   *
   * @param appId
   * @return
   */
  @RequestMapping(value = "/listall/{appId}", method = RequestMethod.POST)
  public Result listAll(@PathVariable String appId) {
    // TODO
    return Result.success();
  }

  @RequestMapping(value = "/list/{page}/{size}", method = RequestMethod.POST)
  public Result list(@RequestBody Role role, @PathVariable("page") int page,
      @PathVariable("size") int size) {
    PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
    Example<Role> example = Example.of(role, getExampleMatcher());
    Page<Role> pageRole = service.findAll(example, pageRequest);
    List<Role> content = pageRole.getContent();

    tenantService.findAll()
        .forEach(i -> content.forEach(j -> {
          if (j.getTenantId().equals(i.getId())) {
            j.setTenantName(i.getName());
          }
        }));
    return Result.success(pageRole);
  }

  @RequestMapping(value = "/addnew", method = RequestMethod.POST)
  public Result addNew(@Valid @RequestBody RoleDto roleDto) {
    Role role = service.findByCode(roleDto.getCode());
    if (null != role) {
      throw new BizException("角色编码已存在", ResultCode.PARAM_ERROR);
    }
    role = new Role();
    BeanUtils.copyProperties(roleDto, role);
    service.save(role);
    return Result.success();
  }

  @RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
  public Result delete(@PathVariable String id) {
    Role role = service.findById(id);
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.SAVE != role.getStatus()) {
      throw new BizException("新增状态才能删除!", ResultCode.PARAM_ERROR);
    }
    service.delete(role);
    return Result.success();
  }

  @RequestMapping(value = "/update", method = RequestMethod.POST)
  public Result update(@Valid @RequestBody RoleDto roleDto) {
    Role role = service.findById(roleDto.getId());
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.SAVE != role.getStatus()) {
      throw new BizException("新增状态才能修改!", ResultCode.PARAM_ERROR);
    }
    if (!role.getCode().equals(roleDto.getCode())) {
      Role temp = service.findByCode(roleDto.getCode());
      if (null != temp) {
        throw new BizException("角色编码已存在", ResultCode.PARAM_ERROR);
      }
    }
    BeanUtils.copyProperties(roleDto, role);
    service.save(role);
    return Result.success();
  }

  @RequestMapping(value = "/submit/{id}", method = RequestMethod.POST)
  public Result submit(@PathVariable String id) {
    Role role = service.findById(id);
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.SAVE != role.getStatus()) {
      throw new BizException("新增状态才能提交!", ResultCode.PARAM_ERROR);
    }
    service.submit(role);
    return Result.success();
  }

  @RequestMapping(value = "/enabled/{id}", method = RequestMethod.POST)
  public Result enabled(@PathVariable String id) {
    Role role = service.findById(id);
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.INVALID != role.getStatus()) {
      throw new BizException("无效状态才能启用!", ResultCode.PARAM_ERROR);
    }
    service.enabled(role);
    return Result.success();
  }

  @RequestMapping(value = "/disabled/{id}", method = RequestMethod.POST)
  public Result disabled(@PathVariable String id) {
    Role role = service.findById(id);
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.VALID != role.getStatus()) {
      throw new BizException("有效状态才能启用!", ResultCode.PARAM_ERROR);
    }
    service.disabled(role);
    return Result.success();
  }

  /**
   * 获取角色对应的系统资源 & 已选中的资源
   *
   * @param roleId
   * @return
   */
  @RequestMapping(value = "/datacolumnset/list/{roleId}", method = RequestMethod.POST)
  public Result dataColumnSet(@PathVariable String roleId) {
    JSONObject result = new JSONObject();
    Role role = service.findById(roleId);
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }

    // 获取树结构的Resource列表
    Resource resource = new Resource();
    resource.setApp(role.getApp());
    resource.setStatus(BasicDataStatus.VALID);
    Example<Resource> example = Example.of(resource, getExampleMatcher());
    List<Resource> resList = resourceService.findAll(example, getSortByCreateTimeAES());
    result.put("content", resourceService.getTree(resList));

    // 获取已选中的资源ID集合
    RoleResource exampleRR = new RoleResource();
    exampleRR.setRole(role);
    Example<RoleResource> example2 = Example.of(exampleRR, getExampleMatcher());
    List<RoleResource> selected_roleResources = roleResourceService.findAll(example2);
    List<String> selected = new ArrayList<>();
    for (RoleResource rr : selected_roleResources) {
      selected.add(rr.getResource().getId());
    }
    result.put("selected", selected);
    return Result.success(result);
  }

  /**
   * 保存选中的资源
   *
   * @param roleResourceDto
   * @return
   */
  @RequestMapping(value = "/datacolumnset/save", method = RequestMethod.POST)
  public Result dataColumnSetSave(@Valid @RequestBody RoleResourceDto roleResourceDto) {
    if (StringUtils.isBlank(roleResourceDto.getRole().getId())) {
      throw new BizException("所选角色不能为空", ResultCode.PARAM_ERROR);
    }
    service.updateRoleResource(roleResourceDto);
    return Result.success();
  }
}
