package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dto.ResourceDto;
import cn.quantgroup.big.stms.sys.model.Resource;
import cn.quantgroup.big.stms.sys.service.ResourceService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.PageRequest;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * @Author meng.xin@quantgroup.cn
 * @Date 2019-10-30 18:19
 * @Version 1.0
 */
@RestController
@RequestMapping("/resource")
public class ResourceController extends BaseController {
    @Autowired
    private ResourceService service;

    @RequestMapping(value = "/get/{id}", method = RequestMethod.POST)
    public Result get(@PathVariable String id) {
        return Result.success(service.findById(id));
    }

    @RequestMapping(value = "/list/{page}/{size}", method = RequestMethod.POST)
    public Result list(@RequestBody Resource resource, @PathVariable int page, @PathVariable int size) {
        PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
        Example<Resource> example = Example.of(resource, getExampleMatcher());
        return Result.success(service.findAll(example, pageRequest));
    }

    @RequestMapping(value = "/addnew", method = RequestMethod.POST)
    public Result addNew(@Valid @RequestBody ResourceDto resourceDto) {
        Resource resource = service.findByCode(resourceDto.getCode());
        if (null != resource) {
            throw new BizException("资源编码已存在", ResultCode.PARAM_ERROR);
        }
        if (resourceDto.getIsSuper() == null) {
        	resourceDto.setIsSuper(false);
		}
        String sensitive = filterPermission(resourceDto.getPermission());
        if (sensitive != null) {
        	throw new BizException("权限标识字段禁止使用字符: " + sensitive, ResultCode.PARAM_ERROR);
        }
        
        resource = new Resource();
        BeanUtils.copyProperties(resourceDto, resource);
//      PropertyUtils.copyProperties(resourceDto, resource);

        // 验证逻辑
        if (resourceDto.getParent() != null && !StringUtils.isEmpty(resourceDto.getParent().getId())) {
            Resource parent = service.findById(resourceDto.getParent().getId());
        	resource.getType().logicalCheck(parent.getType());
		}
        service.save(resource);
        return Result.success();
    }

	@RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
    public Result delete(@PathVariable String id) {
        Resource resource = service.findById(id);
        if (null == resource) {
            throw new BizException("资源不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.SAVE.equals(resource.getStatus())) {
            throw new BizException("新增状态才能删除", ResultCode.PARAM_ERROR);
        }
        service.delete(resource);
        return Result.success();
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result update(@Valid @RequestBody ResourceDto resourceDto) {
        Resource resource = service.findById(resourceDto.getId());
        if (null == resource) {
            throw new BizException("资源不存在", ResultCode.PARAM_ERROR);
        }

        // 校验资源类型是否修改
        if (resource.getType() != resourceDto.getType()) {
            throw new BizException("资源类型不能修改", ResultCode.PARAM_ERROR);
        }
        String sensitive = filterPermission(resourceDto.getPermission());
        if (sensitive != null) {
        	throw new BizException("权限标识字段禁止使用字符: " + sensitive, ResultCode.PARAM_ERROR);
        }
        // 校验资源编码是否修改
        if (!resource.getCode().equals(resourceDto.getCode())) {
            Resource temp = service.findByCode(resourceDto.getCode());
            if (null != temp) {
                throw new BizException("资源编码已存在", ResultCode.PARAM_ERROR);
            }
        }
        if (resourceDto.getIsSuper() == null) {
        	resourceDto.setIsSuper(false);
		}
        BeanUtils.copyProperties(resourceDto, resource, "parent", "app");// 上级资源、所属应用不能修改
        service.save(resource);
        return Result.success();
    }

    @RequestMapping(value = "/submit/{id}", method = RequestMethod.POST)
    public Result submit(@PathVariable String id) {
        Resource resource = service.findById(id);
        if (null == resource) {
            throw new BizException("资源不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.SAVE.equals(resource.getStatus())) {
            throw new BizException("新增状态才能提交", ResultCode.PARAM_ERROR);
        }

        service.submit(resource);
        return Result.success();
    }

    @RequestMapping(value = "/enabled/{id}", method = RequestMethod.POST)
    public Result enabled(@PathVariable String id) {
        Resource resource = service.findById(id);
        if (null == resource) {
            throw new BizException("资源不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.INVALID.equals(resource.getStatus())) {
            throw new BizException("无效状态才能启用", ResultCode.PARAM_ERROR);
        }
        service.enabled(resource);
        return Result.success();
    }

    @RequestMapping(value = "/disabled/{id}", method = RequestMethod.POST)
    public Result disabled(@PathVariable String id) {
        Resource resource = service.findById(id);
        if (null == resource) {
            throw new BizException("资源不存在", ResultCode.PARAM_ERROR);
        }
        if (!BasicDataStatus.VALID.equals(resource.getStatus())) {
            throw new BizException("有效状态才能禁用", ResultCode.PARAM_ERROR);
        }
        service.disabled(resource);
        return Result.success();
    }

    /**
     * 获取资源树结构（根据应用ID获取）
     *
     * @param resource
     * @return
     */
    @RequestMapping(value = "/tree", method = RequestMethod.POST)
    public Result getResourceTree(@RequestBody Resource resource) {
        Example<Resource> example = Example.of(resource, getExampleMatcher());
        List<Resource> resList = service.findAll(example, getSort());
        return Result.success(service.getTree(resList));
    }

	/** 正则敏感字符 */
	private final String[] SENSITIVE_WORDS = { "\\", "?", "+", ".", "^", "|", "[", "$", "***" };

	/**
	 * 过滤权限敏感字符（正则表达式敏感）
	 * @param permission
	 * @return null为无敏感字符
	 */
	private String filterPermission(String permission) {
		if (StringUtils.isEmpty(permission)) {
			return null;
		} else {
			for (String sw : SENSITIVE_WORDS) {
				if (permission.contains(sw)) {
					return sw;
				}
			}
			return null;
		}
	}
}