package cn.quantgroup.big.stms.common.utils;

import org.apache.commons.lang3.StringUtils;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Calendar;
import java.util.Date;

/**
 * @author aoxq
 * @date 2019/11/21 10:10
 **/
public class DateUtil {

	public static final String DATE_FORMAT_6 = "yyyyMM";
    public static final String DATE_FORMAT_10 = "yyyy-MM-dd";
    public static final String DATE_FORMAT_14 = "yyyyMMddHHmmss";
    public static final String DATE_FORMAT_17 = "yyyyMMdd HH:mm:ss";
    public static final String DATE_FORMAT_19 = "yyyy-MM-dd HH:mm:ss";
    public static final String DATE_FORMAT_19_FORWARD_SLASH = "yyyy/MM/dd HH:mm:ss";

    private DateUtil() {
        throw new RuntimeException("不允许构造DateUtil的实例!");
    }

    /**
     * 将Date转换成LocalDate
     */
    public static LocalDate date2LocalDate(Date date) {
        if (null == date) {
            throw new RuntimeException("date can't be null");
        }
        return date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
    }

    /**
     * 将localDate 转化为 Date
     */
    public static Date localDate2Date(LocalDate localDate) {
        if (localDate == null) {
            throw new RuntimeException("date can't be null");
        }
        return Date.from(localDate.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant());
    }

    /**
     * 计算两个日期之前的月份
     */
    public static int monthsBetween(Date start, Date end) {
        if (start == null || end == null) {
            throw new RuntimeException("date can't be null !");
        }
        LocalDate startLD = date2LocalDate(start);
        LocalDate endLD = date2LocalDate(end);
        if (startLD.compareTo(endLD) > 0) {
            return monthsBetween(end, start);
        }
        int startYear = startLD.getYear();
        int endYear = endLD.getYear();
        int startMonth = startLD.getMonthValue();
        int endMonth = endLD.getMonthValue();
        return startYear == endYear ? (endMonth - startMonth) :
                (endMonth + 12 * (endYear - startYear) - startMonth);
    }

    /**
     * 字符串转时间
     *
     * @param str
     * @return
     * @throws ParseException
     */
    public static Date stringToDate(String str) throws ParseException {
        return stringToDate(str, null);
    }

    /**
     * 字符串转时间
     *
     * @param str
     * @param pattern 时间格式
     * @return
     * @throws ParseException
     */
    public static Date stringToDate(String str, String pattern) throws ParseException {
        if (null == str) {
            return null;
        }
        if (StringUtils.isBlank(pattern)) {
            pattern = "yyyy-MM-dd HH:mm:ss";
        }
        SimpleDateFormat format = new SimpleDateFormat(pattern);
        return format.parse(str);
    }

    /**
     * 计算两个日期之间的天数
     */
    public static long daysBetween(Date start, Date end) {
        if (start == null || end == null) {
            throw new RuntimeException("date can't be null !");
        }
        LocalDate startLD = date2LocalDate(start);
        LocalDate endLD = date2LocalDate(end);
       return endLD.toEpochDay() - startLD.toEpochDay();
    }

    /**
     * 格式化日期
     * @param date
     * @param dateFormat
     * @return
     */
    public static String format(Date date, String dateFormat) {
        if (date == null) {
            return "";
        }
        if (StringUtils.isEmpty(dateFormat)) {
            dateFormat = DATE_FORMAT_19;
        }
        return new SimpleDateFormat(dateFormat).format(date);
    }

    /**
     * 月份控制
     * @param date
     * @param addMonth
     * @return
     */
    public static Date controlMonth(Date date, int addMonth) {
    	if (date == null) {
    		date = new Date();
		}
    	Calendar cal = Calendar.getInstance();
		cal.setTime(date);
    	cal.add(Calendar.MONTH, addMonth);
    	return cal.getTime();
    }
}
