package cn.quantgroup.big.stms.sys.utils;

import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.MailService;
import cn.quantgroup.big.stms.sys.dto.TokenDto;
import cn.quantgroup.big.stms.sys.model.Config;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.service.PasswordService;
import com.google.gson.Gson;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.time.LocalDateTime;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * 跟第三方账户配置相关工具
 */
@Component
public class ConfigUtils {

  public final static String ACCESS_TOKEN = "st:external:access_token";
  public final static String BINDING_CODE = "st:external:code";
  public final static String LOGIN_CODE = "st:login";
  public final static String LOGIN_RESTRICT = "st:login:restrict:";
  public final static String BINDING_CONFIG = "st:external:config";
  public final static int BINDING_CODE_EXPIRE = 60 * 5;

  private final RedisTemplate<String, String> stringRedisTemplate;

  public final static String TOKEN_KEY = "st:token:";
  public final static String EXTERNAL_TOKEN_KEY = "st:token:external";
  private static final int SALT_LENGTH = 16;// 加密盐长度

  private final PasswordService passwordService;
  /**
   * 访问token过期时间 单位秒
   **/
  public final static long ACCESS_TOKEN_EXPIRE = 3600L * 2;
  /**
   * 刷新token过期时间 单位秒
   **/
  public final static long REFRESH_TOKEN_EXPIRE = 3600L * 24 * 30;


  private static final String MAIL_TEMPLATE = "账号：#account#\n" +
      "密码：#password#\n" +
      "支持公司邮箱、密码登录";

  private static final String MERCHANT_MAIL_TEMPLATE = "尊敬的商户，您好！您在量化派平台入驻已完成，可登录进行相关操作。\n"
      + "\n"
      + "商户管理后台地址：https://merchant-manage.q-gp.com/\n" + "账号：#account#\n" +
      "密码：#password#\n";

  private final MailService mailService;
  private final Gson gson;

  public ConfigUtils(
      @Qualifier("stringRedisTemplate") RedisTemplate<String, String> stringRedisTemplate,
      PasswordService passwordService, MailService mailService, Gson gson) {
    this.stringRedisTemplate = stringRedisTemplate;
    this.passwordService = passwordService;
    this.mailService = mailService;
    this.gson = gson;
  }

  /**
   * 保存Access token
   *
   * @param id          t_sys_config_external 表id
   * @param accessToken 访问token
   */
  public void saveExternalAccessToken(Integer id, String accessToken) {
    stringRedisTemplate.opsForValue().set(getAccessTokenKey(id), accessToken);
  }

  /**
   * 获取external access token
   *
   * @param id t_sys_config_external 表id
   */
  public String getExternalAccessToken(Integer id) {
    return stringRedisTemplate.opsForValue().get(getAccessTokenKey(id));
  }

  /**
   * 保存access token
   *
   * @param onlineUser onlineUser
   * @param tokenDto   访问token DTO
   */
  public void saveLoginAccessToken(OnlineUser onlineUser, TokenDto tokenDto) {
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + onlineUser.getId(), tokenDto.getAccessToken(), ACCESS_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + tokenDto.getAccessToken(), onlineUser.toJson(), ACCESS_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
  }

  /**
   * 通过userId获取accessToken
   *
   * @param userId accessToken
   */
  public String getAccessTokenByUserId(String userId) {
    return stringRedisTemplate.opsForValue().get(TOKEN_KEY + userId);
  }

  /**
   * 通过refreshToken获取OnlineUser
   *
   * @param refreshToken refreshToken
   */
  public OnlineUser getOnlineUserByRefreshToken(String refreshToken) {
    String onlineData = stringRedisTemplate.opsForValue().get(TOKEN_KEY + refreshToken);
    if (StringUtils.isBlank(onlineData)) {
      throw new BizException(ResultCode.REFRESH_TOKEN_EXPIRE);
    } else {
      return new OnlineUser(onlineData);
    }
  }

  /**
   * 保存token
   *
   * @param onlineUser onlineUser
   * @param tokenDto   访问token DTO
   */
  public void saveLoginToken(OnlineUser onlineUser, TokenDto tokenDto) {
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + onlineUser.getId(), tokenDto.getAccessToken(), ACCESS_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + tokenDto.getAccessToken(), onlineUser.toJson(), ACCESS_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + tokenDto.getRefreshToken(), onlineUser.toJson(), REFRESH_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + ":refreshToken:" + onlineUser.getId(), tokenDto.getRefreshToken(),
            REFRESH_TOKEN_EXPIRE, TimeUnit.SECONDS);

    stringRedisTemplate.boundHashOps(LOGIN_CODE)
        .put(onlineUser.getId(), LocalDateTime.now().toString());
  }

  public LocalDateTime getLastLoginByUserId(String userId) {
    Object object = stringRedisTemplate.boundHashOps(LOGIN_CODE).get(userId);

    return object == null ? null : LocalDateTime.parse(
        (CharSequence) object);
  }

  /**
   * 绑定供应链时候需要更新redis缓存
   *
   * @param userId       用户id
   * @param supplierCode 供应商code
   */
  public void updateSupplierCode(String userId, String supplierCode) {
    String accessToken = stringRedisTemplate.opsForValue().get(TOKEN_KEY + userId);
    String userInfo = stringRedisTemplate.opsForValue()
        .get(TOKEN_KEY + accessToken);
    OnlineUser cache = new OnlineUser(userInfo);
    cache.setSupplierCode(supplierCode);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + accessToken, cache.toJson(), ACCESS_TOKEN_EXPIRE,
            TimeUnit.SECONDS);

    String refreshToken = stringRedisTemplate.opsForValue()
        .get(TOKEN_KEY + ":refreshToken:" + userId);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + refreshToken, cache.toJson(), REFRESH_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
  }

  /**
   * 绑定微信时候需要更新redis缓存
   *
   * @param userId 用户id
   */
  public void updateLoginBindingStatus(String userId) {
    String accessToken = stringRedisTemplate.opsForValue().get(TOKEN_KEY + userId);
    String userInfo = stringRedisTemplate.opsForValue()
        .get(TOKEN_KEY + accessToken);
    OnlineUser cache = new OnlineUser(userInfo);
    cache.setBindStatus(true);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + accessToken, cache.toJson(), ACCESS_TOKEN_EXPIRE,
            TimeUnit.SECONDS);

    String refreshToken = stringRedisTemplate.opsForValue()
        .get(TOKEN_KEY + ":refreshToken:" + userId);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + refreshToken, cache.toJson(), REFRESH_TOKEN_EXPIRE,
            TimeUnit.SECONDS);
  }

  /**
   * 保存token
   *
   * @param onlineUser onlineUser
   * @param tokenDto   访问token DTO
   */
  public TokenDto updateLoginToken(OnlineUser onlineUser, TokenDto tokenDto) {
    if (StringUtils.isNotEmpty(tokenDto.getAccessToken())) {
      stringRedisTemplate.delete(TOKEN_KEY + onlineUser.getId());
      stringRedisTemplate.delete(TOKEN_KEY + tokenDto.getAccessToken());
    }
    if (StringUtils.isNotEmpty(tokenDto.getRefreshToken())) {
      stringRedisTemplate.delete(TOKEN_KEY + tokenDto.getRefreshToken());
      stringRedisTemplate.delete(TOKEN_KEY + ":refreshToken:" + onlineUser.getId());
    }

    long now = System.currentTimeMillis();
    long atExpire = now + ACCESS_TOKEN_EXPIRE * 1000L;
    long rtExpire = now + REFRESH_TOKEN_EXPIRE * 1000L;
    tokenDto.setAccessToken(UUID.randomUUID().toString());
    tokenDto.setAtExpire(Long.toString(atExpire));
    tokenDto.setRefreshToken(UUID.randomUUID().toString());
    tokenDto.setRtExpire(Long.toString(rtExpire));
    saveLoginToken(onlineUser, tokenDto);
    return tokenDto;
  }


  private String getAccessTokenKey(Integer id) {
    return ACCESS_TOKEN + "_" + id;
  }


  /**
   * 设置用户密码
   *
   * @param user
   */
  public void setPassword(User user) {
    String salt = getSalt(SALT_LENGTH);
    String password = DigestUtils.sha512Hex(user.getPassword() + salt);
    user.setSalt(salt);
    user.setPassword(password);
  }

  /**
   * 获取指定长度的加密盐
   *
   * @param length
   * @return
   */
  public String getSalt(int length) {
    return passwordService.getSalt(length);
  }


  /**
   * 发送邮件
   *
   * @param account  账号
   * @param password 密码
   * @param toEmail  目标邮箱
   * @param subject  主题
   */
  public void sendMail(String account, String password, String toEmail, String subject) {
    String content = MAIL_TEMPLATE.substring(0, MAIL_TEMPLATE.indexOf("#account#")) + account
        + MAIL_TEMPLATE.substring(MAIL_TEMPLATE.indexOf("#account#") + 9);
    content = content.substring(0, content.indexOf("#password#")) + password + content.substring(
        content.indexOf("#password#") + 10);

    mailService.sendSimpleMail(toEmail, subject, content);
  }

  /**
   * 发送商户开通主账户邮件
   *
   * @param account  账号
   * @param password 密码
   * @param toEmail  目标邮箱
   * @param subject  主题
   */
  public void sendMasterMail(String account, String password, String toEmail, String subject) {
    String content =
        MERCHANT_MAIL_TEMPLATE.substring(0, MERCHANT_MAIL_TEMPLATE.indexOf("#account#")) + account
            + MERCHANT_MAIL_TEMPLATE.substring(MERCHANT_MAIL_TEMPLATE.indexOf("#account#") + 9);
    content = content.substring(0, content.indexOf("#password#")) + password + content.substring(
        content.indexOf("#password#") + 10);

    mailService.sendSimpleMail(toEmail, subject, content);
  }


  /**
   * 保存微信账户绑定场景码
   *
   * @param scene      场景码
   * @param onlineUser 用户信息
   */
  public void saveBindingCode(String scene, OnlineUser onlineUser, Config config) {
    stringRedisTemplate.opsForValue()
        .set(BINDING_CODE + scene, gson.toJson(onlineUser), BINDING_CODE_EXPIRE,
            TimeUnit.SECONDS);
    stringRedisTemplate.opsForValue()
        .set(BINDING_CONFIG + scene, gson.toJson(config), BINDING_CODE_EXPIRE,
            TimeUnit.SECONDS);

  }

  /**
   * 获取微信绑定的用户信息
   *
   * @param scene
   */
  public OnlineUser getBindingScene(String scene) {
    return gson.fromJson(stringRedisTemplate.opsForValue()
        .get(BINDING_CODE + scene), OnlineUser.class);
  }

  /**
   * 获取微信绑定的配置信息
   *
   * @param scene
   */
  public Config getBindingConfig(String scene) {
    return gson.fromJson(stringRedisTemplate.opsForValue()
        .get(BINDING_CONFIG + scene), Config.class);
  }

  /**
   * 检查是否可以登录
   * @param account 用户名
   * @return true可以登录，false不能登录
   */
  public boolean checkLoginFailTimes(String account) {
    String times = stringRedisTemplate.opsForValue().get(LOGIN_RESTRICT + account);
    if (StringUtils.isEmpty(times)) {
      return true;
    } else {
      return Integer.parseInt(times) <= 5;
    }
  }

  public void addLoginFailTimes(String account) {
    String times = stringRedisTemplate.opsForValue().get(LOGIN_RESTRICT + account);
    if (StringUtils.isEmpty(times)) {
      stringRedisTemplate.opsForValue().set(LOGIN_RESTRICT+account, String.valueOf(1),10,TimeUnit.MINUTES);
    }else{
      stringRedisTemplate.opsForValue().increment(LOGIN_RESTRICT + account,1);
    }
  }

}
