package cn.quantgroup.big.stms.sys.service.v2.impl;

import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dao.ConfigDao;
import cn.quantgroup.big.stms.sys.dao.UserDao;
import cn.quantgroup.big.stms.sys.dao.UserExternalDao;
import cn.quantgroup.big.stms.sys.dto.TokenDto;
import cn.quantgroup.big.stms.sys.dto.UserPhoneDto;
import cn.quantgroup.big.stms.sys.model.Config;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.model.UserExternal;
import cn.quantgroup.big.stms.sys.service.remote.WechatRemoteService;
import cn.quantgroup.big.stms.sys.service.v2.IUserService;
import cn.quantgroup.big.stms.sys.service.v2.strategy.AccountStrategy;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

import static cn.quantgroup.big.stms.common.result.ResultCode.INVALID_PHONE;

@Service
@Slf4j
public class UserServiceV2Impl implements IUserService {

  private final ConfigDao configDao;
  private final WechatRemoteService wechatRemoteService;
  private final UserExternalDao userExternalDao;

  private final ConfigUtils configUtils;
  private final UserDao userDao;

  private final AccountStrategy accountStrategy;

  public UserServiceV2Impl(ConfigDao configDao, WechatRemoteService wechatRemoteService,
      UserExternalDao userExternalDao, ConfigUtils configUtils, UserDao userDao,
      AccountStrategy accountStrategy) {
    this.configDao = configDao;
    this.wechatRemoteService = wechatRemoteService;
    this.userExternalDao = userExternalDao;
    this.configUtils = configUtils;
    this.userDao = userDao;
    this.accountStrategy = accountStrategy;
  }

  @Override
  public boolean bindPhone(String userId, int tenantId, String appId, String code) {
    Config config = configDao.findByTenantIdAndAppId(tenantId, appId);

    Map<String, String> param = new HashMap<>();
    param.put("code", code);
    UserPhoneDto userPhoneDto = wechatRemoteService.getUserPhoneNumber(
        configUtils.getExternalAccessToken(config.getId()), param);
    if (userPhoneDto.getErrcode() == 0) {
      UserExternal userExternal = userExternalDao.findOne(userId);
      if (StringUtils.isNotEmpty(userExternal.getPhone())) {
        throw new BizException(ResultCode.DUPLICATION_DATA);
      }

      userExternal.setPhone(userPhoneDto.getPhone_info().getPurePhoneNumber());

      try {
        userExternalDao.save(userExternal);
        return true;
      } catch (Exception ex) {
        log.info("bindPhone:{}",ex.getMessage(),ex);
        if (ex instanceof DataIntegrityViolationException) {
          throw new BizException(ResultCode.DUPLICATION_DATA);
        }
        throw new BizException(ResultCode.BIND_PHONE_ERROR);
      }
    }
    return false;
  }

  @Override
  public boolean bind(String userId, String supplierCode) {
    UserExternal userExternal = userExternalDao.findOne(userId);
    if (StringUtils.isNotEmpty(userExternal.getUserId())) {
      userDao.bindSupplierCode(userId, supplierCode);
    }
    try {
      boolean result = userExternalDao.bindSupplierCode(userId, supplierCode) > 0;
      if (result) {
        configUtils.updateSupplierCode(userId, supplierCode);
      }
    } catch (Exception ex) {
      throw new BizException(ResultCode.BIND_SUPPLIER_CODE_ERROR);
    }
    return true;
  }

  @Override
  public boolean bindCode(String code, String scene) {
    Config config = configUtils.getBindingConfig(scene);
    TokenDto tokenDto =  accountStrategy.loginByCode(code,config.getAppId(),config.getTenantId());

    OnlineUser externalUser = configUtils.getOnlineUserByRefreshToken(tokenDto.getRefreshToken());

    OnlineUser onlineUser =  configUtils.getBindingScene(scene);

    userExternalDao.updateUserId(externalUser.getId(), onlineUser.getId());
    configUtils.updateLoginBindingStatus(onlineUser.getId());
    return true;
  }

  @Override
  public OnlineUser getOnlineUserByPhone(int tenantId, String appId, String phone) {
    UserExternal userExternal = userExternalDao.findByTenantIdAndAppIdAndPhone(tenantId, appId,
        phone);
    if (userExternal != null) {
      log.info("用户信息:{},{}",phone,userExternal.getPhone());
      OnlineUser result = new OnlineUser();
      if (StringUtils.isNotEmpty(userExternal.getUserId())) {
        result.setId(userExternal.getUserId());
        result.setAccountType(0);
        User user = userDao.findOne(userExternal.getUserId());
        result.setName(user.getName());
        result.setAccount(user.getAccount());
        result.setOrgName(user.getOrganization().getName());
        result.setOrgId(user.getOrganization().getId());
      } else {
        result.setId(userExternal.getId());
        result.setAccountType(1);
      }
      result.setSupplierCode(userExternal.getSupplierCode());
      result.setTenantIds(String.valueOf(tenantId));

      return result;

    } else {
      log.info("手机号码没有找到：{}",phone);
      throw new BizException(INVALID_PHONE);
    }
  }
}
