package cn.quantgroup.big.stms.sys.controller.v2;

import cn.quantgroup.big.stms.common.config.IAuthentication;
import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.enums.TenantEnum;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.utils.Constants;
import cn.quantgroup.big.stms.sys.dto.UserDto;
import cn.quantgroup.big.stms.sys.dto.UserRoleDto;
import cn.quantgroup.big.stms.sys.dto.UserV2Dto;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import cn.quantgroup.big.stms.sys.service.PasswordService;
import cn.quantgroup.big.stms.sys.service.UserService;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import cn.quantgroup.big.stms.sys.utils.PermissionUtils;
import com.google.code.kaptcha.impl.DefaultKaptcha;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.web.bind.annotation.*;

import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static cn.quantgroup.big.stms.common.utils.Constants.CAPTCHA_KEY;

@RestController
@RequestMapping(value = "/v2/user", produces = {"application/json"})
public class UserV2Controller {

  @Autowired
  private UserService userService;

  @Autowired
  @Qualifier("LDAPAuthentication")
  private IAuthentication authentication;

  @Autowired
  private OrganizationService organizationService;

  @Autowired
  private PasswordService passwordService;

  @Autowired
  private ConfigUtils configUtils;


  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private DefaultKaptcha defaultKaptcha;

  /**
   * 商户管理后台-修改密码
   *
   * @param map
   * @return
   */
  @PostMapping("/password/reset")
  public Result merchantResetPassword(@RequestBody HashMap<String, String> map) {
    // 获取用户信息
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser == null) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }

    // 校验 新密码规则
    String newPassword = map.get("newPassword");
    validatePassword(newPassword);
    String password = map.get("password");

    String account = onlineUser.getAccount();

    //验证密码是否正确
    boolean isPass = userService.checkLogin(account, password);
    if (!isPass) {
      //登录验证
      isPass = authentication.authenticate(account, password);
      if (!isPass) {
        return Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD_ORIGINAL);
      }
    }

    // 设置新密码
    User user = userService.findById(onlineUser.getId());
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    user.setPassword(newPassword);
    // 设置密码
    setPassword(user);
    userService.save(user);

    return Result.success();

  }


  /**
   * 量星球-重置密码
   *
   * @param map
   * @return
   */
  @RequestMapping(value = "/password/resetpassword", method = RequestMethod.POST)
  public Result resetPassword(@RequestBody HashMap<String, String> map) {
    if (null == map) {
      throw new BizException("请求参数不能为空", ResultCode.PARAM_ERROR);
    }

    String password = map.get("password");
    validatePassword(password);
    String id = map.get("id");
    if (StringUtils.isBlank(id)) {
      throw new BizException("用户ID不能为空", ResultCode.PARAM_ERROR);
    }
    User user = userService.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (PermissionUtils.checkMaster(user)) {
      user.setPassword(password);
      // 设置密码
      setPassword(user);
      userService.save(user);
      // 发送密码邮件
      try {
        configUtils.sendMail(user.getAccount(), password, user.getEmail(), Constants.SUBJECT_OPEN);
      } catch (Exception ex) {
        return Result.success("账户开通成功，但是邮件发送失败！");
      }
    }

    return Result.success();
  }

  /**
   * 量星球-校验密码规则
   *
   * @param map
   * @return
   */
  @PostMapping("/password/validate")
  public Result validatePasswordRule(@RequestBody HashMap<String, String> map) {
    String password = map.get("password");
    validatePassword(password);
    return Result.success();
  }

  /**
   * 获取登录验证码
   *
   * @param request
   * @param response
   * @throws Exception
   */
  @RequestMapping("/captcha")
  public void defaultKaptcha(@NotNull @RequestParam(name = "username") String username,
      HttpServletRequest request, HttpServletResponse response)
      throws Exception {
    byte[] captchaChallengeAsJpeg = null;
    ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
    try {
      // 生成验证码字符串并保存到redis中
      String createText = defaultKaptcha.createText();
      stringRedisTemplate.opsForValue()
          .set(CAPTCHA_KEY + username, createText, Constants.CAPTCHA_EXPIRE, TimeUnit.SECONDS);
      // 生成的验证码字符串返回一个BufferedImage对象并转为byte写入到byte数组中
      BufferedImage challenge = defaultKaptcha.createImage(createText);
      ImageIO.write(challenge, "jpg", jpegOutputStream);
    } catch (IllegalArgumentException e) {
      response.sendError(HttpServletResponse.SC_NOT_FOUND);
      return;
    }

    // 定义response输出类型为image/jpeg类型，使用response输出流输出图片的byte数组
    captchaChallengeAsJpeg = jpegOutputStream.toByteArray();
    response.setHeader("Cache-Control", "no-store");
    response.setHeader("Pragma", "no-cache");
    response.setDateHeader("Expires", 0);
    response.setContentType("image/jpeg");
    ServletOutputStream responseOutputStream = response.getOutputStream();
    responseOutputStream.write(captchaChallengeAsJpeg);
    responseOutputStream.flush();
    responseOutputStream.close();
  }


  /**
   * 校验密码规则
   *
   * @param password
   */
  private void validatePassword(String password) {
    if (StringUtils.isBlank(password)) {
      throw new BizException("新密码不能为空", ResultCode.PARAM_ERROR);
    }
    if (password.length() < Constants.PASSWORD_LENGTH_MIN
        || password.length() > Constants.PASSWORD_LENGTH_MAX) {
      throw new BizException(
          "新密码必须在" + Constants.PASSWORD_LENGTH_MIN + "-" + Constants.PASSWORD_LENGTH_MAX
              + "之间",
          ResultCode.PARAM_ERROR);
    }
    String reg = "(?=.*\\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[^a-zA-Z0-9]).{13,18}";
    if (!password.matches(reg)) {
      throw new BizException("新密码必须包含大小写字母、数字、特殊字符四类");
    }
  }

  /**
   * 设置用户密码
   *
   * @param user
   */
  private void setPassword(User user) {
    String salt = getSalt(Constants.SALT_LENGTH);
    String password = DigestUtils.sha512Hex(user.getPassword() + salt);
    user.setSalt(salt);
    user.setPassword(password);
  }

  /**
   * 获取指定长度的加密盐
   *
   * @param length
   * @return
   */
  public String getSalt(int length) {
    return passwordService.getSalt(length);
  }


  /**
   * 用户分配权限
   */
  @PostMapping(value = "/roles/save")
  public Result saveRoles(@Valid @RequestBody UserRoleDto userRoleDto) {
    if (StringUtils.isBlank(userRoleDto.getUser().getId())) {
      throw new BizException("所选用户不能为空", ResultCode.PARAM_ERROR);
    }
    User u = userService.findById(userRoleDto.getUser().getId());
    if (null == u) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    userService.updateUserRole(userRoleDto);
    return Result.success();
  }


  @PostMapping(value = "/list/{page}/{size}")
  public Result list(@RequestBody UserV2Dto userV2Dto, @PathVariable int page,
      @PathVariable int size) {
    PageRequest pageRequest = new PageRequest(page - 1, size);
    Page<UserDto> userPage = userService.findAllByOrganizationDto(userV2Dto, pageRequest);

    return Result.success(userPage);
  }

  @GetMapping(value = "/info/{id}")
  public Result get(@PathVariable String id) {
    User user = userService.findById(id);
    user.setPassword(null);
    user.setSalt(null);
    return Result.success(user);
  }


  @PostMapping(value = "/enabled/{id}")
  public Result enabled(@PathVariable String id) {
    User user = userService.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.INVALID != user.getStatus()) {
      throw new BizException("无效状态才能启用", ResultCode.PARAM_ERROR);
    }
    userService.enabled(user);
    return Result.success();
  }

  @PostMapping(value = "/disabled/{id}")
  public Result disabled(@PathVariable String id) {
    User user = userService.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.VALID != user.getStatus()) {
      throw new BizException("有效状态才能禁用", ResultCode.PARAM_ERROR);
    }
    userService.disabled(user);
    return Result.success();
  }


  /**
   * 新建主账户
   *
   * @param userDto 用户信息
   * @return Result
   */
  @RequestMapping(value = "/addmaster", method = RequestMethod.POST)
  public Result addMaster(@Valid @RequestBody UserDto userDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    User user = userService.findByCode(userDto.getCode());
    if (null != user) {
      throw new BizException("用户编码已存在", ResultCode.PARAM_ERROR);
    }
    if (StringUtils.isNotEmpty(userDto.getAccount())) {
      user = userService.getByAccount(userDto.getAccount());
      if (null != user) {
        throw new BizException("登录账号已存在", ResultCode.PARAM_ERROR);
      }
    }

    if (StringUtils.isNotEmpty(userDto.getEmail())) {
      user = userService.getByEmail(userDto.getEmail());
      if (null != user) {
        throw new BizException("邮箱已存在", ResultCode.PARAM_ERROR);
      }
    }
    if (StringUtils.isNotEmpty(userDto.getMobile())) {
      user = userService.getByMobile(userDto.getMobile());
      if (null != user) {
        throw new BizException("手机号码已存在", ResultCode.PARAM_ERROR);
      }
    }

    user = new User();
    BeanUtils.copyProperties(userDto, user);
    if (StringUtils.isNotEmpty(userDto.getId())) {
      user.setId(userDto.getId());
    } else {
      user.setId(UUID.randomUUID().toString());
    }

    String password = user.getPassword();
    // 设置密码
    configUtils.setPassword(user);
    user.setTenantId(TenantEnum.KDSP.getTenantId());
    user.setStatus(BasicDataStatus.VALID);
    user.setCreateTime(Date.from(LocalDateTime.now().toInstant(ZoneOffset.UTC)));
    user.setCreator(onlineUser.getName());
    user.setUpdateUser(onlineUser.getName());
    user.setUpdateTime(Date.from(LocalDateTime.now().toInstant(ZoneOffset.UTC)));

    userService.addMasterAccount(user, userDto.getId(), userDto.getRoleId());

    // 发送密码邮件
    try {
      configUtils.sendMasterMail(user.getAccount(), password, user.getEmail(),
          Constants.SUBJECT_OPEN);
    } catch (Exception ex) {
      return Result.success("账户开通成功，但是邮件发送失败！");
    }


    return Result.success();
  }


  /**
   * 新建主账户
   *
   * @param userDto 用户信息
   * @return Result
   */
  @PostMapping(value = "/addnew")
  public Result addNew(@Valid @RequestBody UserDto userDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    User user = userService.findByCode(userDto.getCode());
    if (null != user) {
      throw new BizException("用户编码已存在", ResultCode.PARAM_ERROR);
    }
    if (StringUtils.isNotEmpty(userDto.getAccount())) {
      user = userService.getByAccount(userDto.getAccount());
      if (null != user) {
        throw new BizException("登录账号已存在", ResultCode.PARAM_ERROR);
      }
    }

    if (StringUtils.isNotEmpty(userDto.getEmail())) {
      user = userService.getByEmail(userDto.getEmail());
      if (null != user) {
        throw new BizException("邮箱已存在", ResultCode.PARAM_ERROR);
      }
    }
    if (StringUtils.isNotEmpty(userDto.getMobile())) {
      user = userService.getByMobile(userDto.getMobile());
      if (null != user) {
        throw new BizException("手机号码已存在", ResultCode.PARAM_ERROR);
      }
    }

    user = new User();
    BeanUtils.copyProperties(userDto, user);
    if (StringUtils.isNotEmpty(userDto.getId())) {
      user.setId(userDto.getId());
    } else {
      user.setId(UUID.randomUUID().toString());
    }

    String password = user.getPassword();
    // 设置密码
    configUtils.setPassword(user);
    user.setTenantId(TenantEnum.KDSP.getTenantId());
    user.setSupplierCode(onlineUser.getSupplierCode());
    user.setStatus(BasicDataStatus.VALID);
    user.setCreateTime(Date.from(LocalDateTime.now().toInstant(ZoneOffset.UTC)));
    user.setCreator(onlineUser.getName());
    user.setUpdateUser(onlineUser.getName());
    user.setUpdateTime(Date.from(LocalDateTime.now().toInstant(ZoneOffset.UTC)));

    //创建子账户必须是子账户
    if (userDto.getOrganization() != null && userDto.getOrganization().getId() != null) {
      List<String> children = null;
      if (onlineUser.getOrgId() != null) {
        children = organizationService.findAllByParentId(onlineUser.getOrgId()).stream().map(
            Organization::getId).collect(
            Collectors.toList());
      }
      //如果是超级管理员或者主账户才能操作
      if (onlineUser.getOrgId() == null || onlineUser.getOrgId()
          .equals(userDto.getOrganization().getId())
          || Objects.requireNonNull(children).contains(userDto.getOrganization().getId())) {
        userService.addAccount(user, userDto.getRoleId());
        // 发送密码邮件
        try {
          configUtils.sendMasterMail(user.getAccount(), password, user.getEmail(),
              Constants.SUBJECT_OPEN);
        } catch (Exception ex) {
          return Result.success("账户开通成功，但是邮件发送失败！");
        }

        return Result.success();
      }

    }

    return Result.failure(ResultCode.FORBIDDEN);

  }


  /**
   * 修改账户
   *
   * @param userDto 用户信息
   * @return Result
   */
  @PostMapping(value = "/update")
  public Result update(@Valid @RequestBody UserV2Dto userDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    User user = userService.findById(userDto.getId());
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (StringUtils.isNotEmpty(userDto.getName())) {
      user.setName(userDto.getName());
    }
    if (StringUtils.isNotEmpty(userDto.getMobile())) {
      user.setMobile(userDto.getMobile());
    }
    if (StringUtils.isNotEmpty(user.getEmail())) {
      user.setEmail(user.getEmail());
    }
    if (StringUtils.isNotEmpty(userDto.getDescription())) {
      user.setDescription(userDto.getDescription());
    }
    user.setOrganization(userDto.getOrganization());

    //创建子账户必须是子账户
    if (userDto.getOrganization() != null && userDto.getOrganization().getId() != null) {
      List<String> children = null;
      if (onlineUser.getOrgId() != null) {
        children = organizationService.findAllByParentId(onlineUser.getOrgId()).stream().map(
            Organization::getId).collect(
            Collectors.toList());
      }
      //如果是超级管理员或者主账户才能操作
      if (onlineUser.getOrgId() == null || onlineUser.getOrgId()
          .equals(userDto.getOrganization().getId())
          || Objects.requireNonNull(children).contains(userDto.getOrganization().getId())) {
        userService.updateAccount(user, userDto.getRoleId());
        // 发送密码邮件
        return Result.success();
      }

    }

    return Result.failure(ResultCode.FORBIDDEN);

  }

  @PostMapping(value = "/delete/{id}")
  public Result delete(@PathVariable String id) {

    User user = userService.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }

    if (PermissionUtils.checkMaster(user)) {
      userService.delete(user);
    }

    return Result.success();
  }

}
