package cn.quantgroup.big.stms.sys.controller.v2;

import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dao.UserRoleDao;
import cn.quantgroup.big.stms.sys.dao.V2.RoleV2Dao;
import cn.quantgroup.big.stms.sys.dto.RoleDto;
import cn.quantgroup.big.stms.sys.dto.RoleQueryDto;
import cn.quantgroup.big.stms.sys.dto.RoleSelectedDto;
import cn.quantgroup.big.stms.sys.dto.RoleV2Dto;
import cn.quantgroup.big.stms.sys.model.App;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.model.Role;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import cn.quantgroup.big.stms.sys.service.RoleService;
import cn.quantgroup.big.stms.sys.service.TenantService;
import cn.quantgroup.big.stms.sys.utils.PermissionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@RestController
@RequestMapping(value = "/v2/role", produces = {"application/json"})
public class RoleV2Controller extends BaseController {

  private final TenantService tenantService;
  private final RoleV2Dao roleDao;

  @Autowired
  private UserRoleDao userRoleDao;

  @Autowired
  private RoleService roleService;

  @Autowired
  private OrganizationService organizationService;


  public RoleV2Controller(TenantService tenantService, RoleV2Dao roleDao) {
    this.tenantService = tenantService;
    this.roleDao = roleDao;
  }

  @PostMapping(value = "/current")
  public Result list(@RequestParam(value = "tenantId", defaultValue = "560761") Integer tenantId,
      @RequestParam("applicationId") String applicationId) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();

    List<Role> roles = roleDao.findByCurrent(tenantId, applicationId, onlineUser.getId());

    tenantService.findAll()
        .forEach(i -> roles.forEach(j -> {
          if (j.getTenantId().equals(i.getId())) {
            j.setTenantName(i.getName());
          }
        }));
    roles.forEach(i -> i.setApp(null));
    return Result.success(roles);
  }


  /**
   * 查询当前组织下所有角色列表
   *
   * @param page 页码
   * @param size 每页数量
   * @return Result
   */
  @PostMapping(value = "/list/{page}/{size}")
  public Result list(@PathVariable("page") int page,
      @PathVariable("size") int size, @RequestParam("appCode") String appCode, @RequestBody
  RoleQueryDto roleQueryDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();

    PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
    Role role = new Role();
    role.setOrganizationId(onlineUser.getOrgId());
    App app = new App();
    app.setCode(appCode);
    role.setApp(app);
    if (roleQueryDto != null) {
      if (StringUtils.isNotEmpty(roleQueryDto.getName())) {
        role.setName(roleQueryDto.getName());
      }
      if (StringUtils.isNotEmpty(roleQueryDto.getUpdateUser())) {
        role.setUpdateUser(roleQueryDto.getUpdateUser());
      }
    }

    Example<Role> query = Example.of(role);
    Page<Role> pageRole = roleService.findAll(query, pageRequest);
    tenantService.findAll()
        .forEach(i -> pageRole.getContent().forEach(j -> {
          if (j.getTenantId().equals(i.getId())) {
            j.setTenantName(i.getName());
          }
        }));

    return Result.success(pageRole.map(i -> {
      int count = userRoleDao.countByRoleId(i.getId());
      return RoleV2Dto.from(i, count);
    }));
  }


  /**
   * 新建角色
   *
   * @param roleDto RoleDto
   * @return Result
   */

  @PostMapping(value = "/addnew")
  public Result addNew(@Valid @RequestBody RoleDto roleDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    Role role = roleService.findByCode(roleDto.getCode());
    if (null != role) {
      throw new BizException("角色编码已存在", ResultCode.PARAM_ERROR);
    }
    BeanUtils.copyProperties(roleDto, role);
    if (roleDto.getOrgId() != null) {
      List<String> children = null;

      if (onlineUser.getOrgId() != null) {
        children = organizationService.findAllByParentId(onlineUser.getId()).stream().map(
            Organization::getId).collect(
            Collectors.toList());
      }

      //如果是超级管理员或者主账户才能操作
      if (onlineUser.getOrgId() == null || onlineUser.getOrgId().equals(roleDto.getOrgId())
          || Objects.requireNonNull(children).contains(roleDto.getOrgId())) {
        role.setOrganizationId(onlineUser.getOrgId());
      }
    }

    role.setStatus(BasicDataStatus.VALID);
    roleService.save(role);
    return Result.success();
  }

  @RequestMapping(value = "/addResource", method = RequestMethod.POST)
  public Result addResource(@Valid @RequestBody RoleSelectedDto roleDto) {
    roleService.addRoleAndBindResource(roleDto);
    return Result.success();
  }

  @RequestMapping(value = "/updateResource", method = RequestMethod.POST)
  public Result updateResource(@Valid @RequestBody RoleSelectedDto roleDto) {
    roleService.updateRoleAndBindResource(roleDto);
    return Result.success();
  }


  /**
   * 删除角色
   *
   * @param id 角色id
   * @return
   */
  @PostMapping(value = "/delete/{id}")
  public Result delete(@PathVariable String id) {
    roleService.deleteRoleAndBindResource(id);
    return Result.success();
  }

  /**
   * 修改角色
   *
   * @param roleDto
   * @return
   */
  @PostMapping(value = "/update")
  public Result update(@Valid @RequestBody RoleDto roleDto) {
    Role role = roleService.findById(roleDto.getId());
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }

    if (!role.getCode().equals(roleDto.getCode())) {
      Role temp = roleService.findByCode(roleDto.getCode());
      if (null != temp) {
        throw new BizException("角色编码已存在", ResultCode.PARAM_ERROR);
      }
    }
    BeanUtils.copyProperties(roleDto, role);
    if (PermissionUtils.checkMaster(role)) {
      roleService.save(role);
    }
    return Result.success();
  }

  @GetMapping(value = "/info/{id}")
  public Result get(@PathVariable String id) {
    return Result.success(roleService.findById(id));
  }


}
