package cn.quantgroup.big.stms.common.config;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.CacheManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisNode;
import org.springframework.data.redis.connection.RedisSentinelConfiguration;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.util.Assert;
import redis.clients.jedis.JedisPoolConfig;

import java.util.ArrayList;
import java.util.List;

@Configuration
public class RedisConfig {

    @Value("${redis.defaultExpiration}")
    private Long defaultExpiration;
    @Value("${redis.master.host}")
    private String masterHost;
    @Value("${redis.master.port}")
    private Integer masterPort;

    @Value("${redis.sentinel.nodes}")
    private String sentinelNodes;
    @Value("${redis.sentinel.master.name}")
    private String sentinelMasterName;
    @Value("${spring.redis.password:}")
    private String passWord;

    private RedisConnectionFactory getConnectionFactoryOfSingleton() {
        JedisConnectionFactory factory = new JedisConnectionFactory();
        factory.setHostName(masterHost);
        factory.setPort(masterPort);
        factory.setUsePool(true);
        factory.setConvertPipelineAndTxResults(true);
        JedisPoolConfig poolConfig = JedisPoolUtil.getPoolConfig();
        factory.setPoolConfig(poolConfig);
        if(StringUtils.isNotEmpty(passWord)){
            factory.setPassword(passWord);
        }
        factory.afterPropertiesSet();
        return factory;
    }

    private RedisConnectionFactory getConnectionFactoryOfCluster() {
        List<RedisNode> sentinels = createSentinels(sentinelNodes);

        RedisSentinelConfiguration sentinelConfiguration = new RedisSentinelConfiguration();
        sentinelConfiguration.setMaster(sentinelMasterName);
        sentinelConfiguration.setSentinels(sentinels);
        JedisPoolConfig poolConfig = JedisPoolUtil.getPoolConfig();

        JedisConnectionFactory factory = new JedisConnectionFactory(sentinelConfiguration, poolConfig);
        factory.setTimeout(2000);
        factory.setUsePool(true);
        factory.setConvertPipelineAndTxResults(true);
        factory.afterPropertiesSet();
        return factory;
    }

    private List<RedisNode> createSentinels(String sentinelNodes) {
        List<RedisNode> nodes = new ArrayList<RedisNode>();
        for (String node : org.springframework.util.StringUtils
                .commaDelimitedListToStringArray(sentinelNodes)) {
            try {
                String[] parts = org.springframework.util.StringUtils.split(node, ":");
                Assert.state(parts.length == 2, "Must be defined as 'host:port'");
                nodes.add(new RedisNode(parts[0], Integer.valueOf(parts[1])));
            } catch (RuntimeException ex) {
                throw new IllegalStateException(
                        "Invalid redis sentinel " + "property '" + node + "'", ex);
            }
        }
        return nodes;
    }

    @Bean(name = "redisConnectionFactory")
    public RedisConnectionFactory redisConnectionFactory() {
        if (StringUtils.isEmpty(sentinelMasterName)) {
            return getConnectionFactoryOfSingleton();
        } else {
            return getConnectionFactoryOfCluster();
        }
    }

    @Bean(name = "redisTemplate")
    public RedisTemplate<String, Object> redisTemplate(
            RedisConnectionFactory factory) {
        final RedisTemplate<String, Object> template = new RedisTemplate<>();
        StringRedisSerializer stringRedisSerializer = new StringRedisSerializer();
        GenericJackson2JsonRedisSerializer genericJackson2JsonRedisSerializer = new GenericJackson2JsonRedisSerializer();
        template.setEnableTransactionSupport(false);
        template.setKeySerializer(stringRedisSerializer);
        template.setHashKeySerializer(stringRedisSerializer);
        template.setValueSerializer(genericJackson2JsonRedisSerializer);
        template.setDefaultSerializer(genericJackson2JsonRedisSerializer);
        template.setConnectionFactory(factory);
        return template;
    }

    @Bean(name = "stringRedisTemplate")
    public RedisTemplate<String, String> stringRedisTemplate(
            RedisConnectionFactory factory) {
        final RedisTemplate<String, String> template = new RedisTemplate<>();
        StringRedisSerializer stringRedisSerializer = new StringRedisSerializer();
        template.setEnableTransactionSupport(false);
        template.setKeySerializer(stringRedisSerializer);
        template.setHashKeySerializer(stringRedisSerializer);
        template.setValueSerializer(stringRedisSerializer);
        template.setDefaultSerializer(stringRedisSerializer);
        template.setConnectionFactory(factory);
        return template;
    }

    @Bean(name = "cacheManager")
    public CacheManager cacheManager(RedisTemplate<String, Object> redisTemplate) {
        RedisCacheManager cacheManager = new RedisCacheManager(redisTemplate);
        cacheManager.setDefaultExpiration(defaultExpiration);
        cacheManager.setUsePrefix(true);
        return cacheManager;
    }
}
