package cn.quantgroup.big.stms.sys.service.v2.impl;

import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.PermissionService;
import cn.quantgroup.big.stms.sys.dao.ConfigDao;
import cn.quantgroup.big.stms.sys.dao.UserExternalDao;
import cn.quantgroup.big.stms.sys.dto.TokenDto;
import cn.quantgroup.big.stms.sys.enums.ConfigType;
import cn.quantgroup.big.stms.sys.model.Config;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.model.UserExternal;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import cn.quantgroup.big.stms.sys.service.UserService;
import cn.quantgroup.big.stms.sys.service.remote.WechatRemoteService;
import cn.quantgroup.big.stms.sys.service.v2.AccountService;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Example;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static cn.quantgroup.big.stms.common.result.ResultCode.DISABLED_USER;
import static cn.quantgroup.big.stms.common.result.ResultCode.EXTERNAL_LOGIN_ERROR;
import static cn.quantgroup.big.stms.sys.utils.ConfigUtils.ACCESS_TOKEN_EXPIRE;
import static cn.quantgroup.big.stms.sys.utils.ConfigUtils.REFRESH_TOKEN_EXPIRE;

/**
 * 微信对接
 */
@Component
@Slf4j
public class WeChatMiniAccountServiceImpl implements AccountService {

  private final ConfigDao configDao;

  private final UserExternalDao userExternalDao;
  private final WechatRemoteService wechatRemoteService;
  @Autowired
  private PermissionService permissionService;

  @Autowired
  private OrganizationService organizationService;
  private final ConfigUtils configUtils;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private UserService userService;
  private final Gson gson;

  public WeChatMiniAccountServiceImpl(ConfigDao configDao, UserExternalDao userExternalDao,
      WechatRemoteService wechatRemoteService, ConfigUtils configUtils, Gson gson) {
    this.configDao = configDao;
    this.userExternalDao = userExternalDao;
    this.wechatRemoteService = wechatRemoteService;
    this.configUtils = configUtils;
    this.gson = gson;
  }

  @Override
  public ConfigType getType() {
    return ConfigType.WECHAT_MINI;
  }

  @Override
  public void reloadExternalAccessToken() {
    configDao.findByType(ConfigType.WECHAT_MINI).forEach(i -> {

      Map<String, String> accessToken = wechatRemoteService.getAccessToken(i.getAppId(),
          i.getSecret());

      if (accessToken.containsKey("errcode")) {
        log.error("微信获取token失败:{}", i.getAppId());
      } else {
        configUtils.saveExternalAccessToken(i.getId(), accessToken.get("access_token"));
      }
    });
  }

  @Override
  public TokenDto loginByCode(String code, Config config) {

    //1.请求微信接口获取
    String openid;
    String sessionKey = null;
    String unionId = null;
    String response = wechatRemoteService.login(config.getAppId(), config.getSecret(), code);

    Type resultType = new TypeToken<Map<String, String>>() {
    }.getType();

    Map<String, String> responseMap = gson.fromJson(response, resultType);

    assert responseMap != null;
    if (responseMap.containsKey("errcode") && Integer.parseInt(responseMap.get("errcode")) != 0) {
      log.error("code:{},errcode:{},errmsg:{}", code, responseMap.get("errcode"),
          responseMap.get("errmsg"));
      throw new BizException(EXTERNAL_LOGIN_ERROR);
    } else {
      openid = responseMap.get("openid");
      if (responseMap.containsKey("session_key")) {
        sessionKey = responseMap.get("session_key");
      }
      if (responseMap.containsKey("unionid")) {
        unionId = responseMap.get("unionid");
      }
    }

    //2.判断数据库存在该用户，如果有，那么更新相关配置；否则新建
    UserExternal userExternal;
    UserExternal userExternalQuery = new UserExternal();

    if (StringUtils.isNotEmpty(unionId)) {
      userExternalQuery.setUnionId(unionId);
    } else {
      userExternalQuery.setOpenId(openid);
    }
    userExternal = userExternalDao.findOne(Example.of(userExternalQuery));

    if (userExternal == null) {
      userExternal = new UserExternal();
      userExternal.setOpenId(openid);
      userExternal.setAppId(config.getAppId());
      userExternal.setUnionId(unionId);
      userExternal.setState(2);
      userExternal.setSessionKey(sessionKey);
    }
    userExternal.setSessionKey(sessionKey);
    userExternal.setTenantId(config.getTenantId());
    userExternalDao.save(userExternal);

    // 同一个用户，不同的系统，公用一个token
    // 根据用户id获取 用户token
    String accessToken = null;
    User user = null;
    if (StringUtils.isNotEmpty(userExternal.getUserId())) {
      user = userService.findById(userExternal.getUserId());
      accessToken = stringRedisTemplate.opsForValue()
          .get(ConfigUtils.TOKEN_KEY + userExternal.getUserId());
    }

    if (StringUtils.isEmpty(accessToken)) {
      accessToken = UUID.randomUUID().toString();
    }
    String refreshToken = UUID.randomUUID().toString();
    long now = System.currentTimeMillis();
    Long atExpire = now + ACCESS_TOKEN_EXPIRE * 1000L;
    Long rtExpire = now + REFRESH_TOKEN_EXPIRE * 1000L;

    String orgId = "";
    String orgName = "";
    if (user != null && user.getOrganization() != null) {
      if (BasicDataStatus.INVALID.equals(user.getOrganization().getStatus())) {
        throw new BizException("请联系管理员开通权限！",
            ResultCode.ORGANIZATION_FROZEN);
      }
      orgId = user.getOrganization().getId();
      orgName = user.getOrganization().getName();
    }

   //如果用户被冻结，禁止登录
    if (!userExternal.getState().equals(2)) {
      throw new BizException(DISABLED_USER);
    }

    //如果绑定了supplierCode，这时候由于入驻过程中组织已经被禁用，禁止登录
    if (StringUtils.isEmpty(orgId) && StringUtils.isNotEmpty(userExternal.getSupplierCode())) {
      List<Organization> organizations = organizationService.findBySupplierCode(
          userExternal.getSupplierCode());
      if (organizations.stream().anyMatch(i -> BasicDataStatus.INVALID.equals(i.getStatus()))) {
        throw new BizException("请联系管理员开通权限！", ResultCode.ORGANIZATION_FROZEN);
      }
    }

    TokenDto tokenDto = new TokenDto();
    tokenDto.setAccessToken(accessToken);
    tokenDto.setAtExpire(atExpire.toString());
    tokenDto.setRefreshToken(refreshToken);
    tokenDto.setRtExpire(rtExpire.toString());

    if (user != null) {
      OnlineUser onlineUser = new OnlineUser(user.getId(), user.getAccount(), user.getName(), orgId,
          orgName, "", String.valueOf(user.getTenantId()), user.getSupplierCode());
      String supplierCode = user.getSupplierCode();
      if(StringUtils.isEmpty(supplierCode)&&user.getOrganization()!=null && StringUtils.isNotEmpty(user.getOrganization().getId())){
        Organization organization = organizationService.findById(user.getOrganization().getId());
        supplierCode = organization.getSupplierCode();
      }

      onlineUser.setSupplierCode(supplierCode);
      onlineUser.setBindStatus(true);
      configUtils.saveLoginToken(onlineUser, tokenDto);
      permissionService.refreshPermission(user.getId(), null);
    } else {
      OnlineUser onlineUser = new OnlineUser(userExternal.getId(), "", "", orgId, orgName, "",
          String.valueOf(userExternal.getTenantId()), userExternal.getSupplierCode());
      onlineUser.setAccountType(1);
      onlineUser.setSupplierCode(userExternal.getSupplierCode());
      configUtils.saveLoginToken(onlineUser, tokenDto);
    }
    return tokenDto;
  }
}
