package cn.quantgroup.big.stms.sys.service.impl;

import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.PermissionService;
import cn.quantgroup.big.stms.common.service.impl.BaseBasicDataServiceImpl;
import cn.quantgroup.big.stms.sys.dao.AppDao;
import cn.quantgroup.big.stms.sys.dao.RoleDao;
import cn.quantgroup.big.stms.sys.dao.RoleResourceDao;
import cn.quantgroup.big.stms.sys.dao.V2.RoleV2Dao;
import cn.quantgroup.big.stms.sys.dto.RoleResourceDto;
import cn.quantgroup.big.stms.sys.dto.RoleSelectedDto;
import cn.quantgroup.big.stms.sys.model.App;
import cn.quantgroup.big.stms.sys.model.Resource;
import cn.quantgroup.big.stms.sys.model.Role;
import cn.quantgroup.big.stms.sys.model.RoleResource;
import cn.quantgroup.big.stms.sys.service.ResourceService;
import cn.quantgroup.big.stms.sys.service.RoleResourceService;
import cn.quantgroup.big.stms.sys.service.RoleService;
import cn.quantgroup.big.stms.sys.service.UserService;
import cn.quantgroup.big.stms.sys.utils.PermissionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.Instant;
import java.util.*;

@Service
public class RoleServiceImpl extends BaseBasicDataServiceImpl<Role, String> implements RoleService {

  @Autowired
  private RoleDao roleDao;
  @Autowired
  private ResourceService resourceService;
  @Autowired
  private RoleResourceService roleResourceService;
  @Autowired
  private UserService userService;
  @Autowired
  private PermissionService permissionService;

  @Autowired
  private AppDao appDao;

  @Autowired
  private RoleResourceDao roleResourceDao;

  @Autowired
  private RoleV2Dao roleV2Dao;

  @Override
  public RoleDao getRepository() {
    return roleDao;
  }

  @Override
  public void updateRoleResource(RoleResourceDto roleResourceDto) {
    Role r = roleDao.findOne(roleResourceDto.getRole().getId());
    if (null == r) {
      throw new BizException("角色不存在", ResultCode.PARAM_ERROR);
    }
    // 检查当前角色所选资源列表
    List<RoleResource> cur_rrArr = roleResourceDao.findByRoleId(r.getId());

    if (roleResourceDto.getSelectedResourceIds().size() == 0) {
      // 清空角色所有资源
      roleResourceService.deleteInBatch(cur_rrArr);
    } else {
      // 检查目标角色的资源列表
      Iterable<Resource> selected_Resources = resourceService.findAllById(
          roleResourceDto.getSelectedResourceIds());
      Set<String> selectedIds = new HashSet<>();
      for (Resource resource : selected_Resources) {
        selectedIds.add(resource.getId());
      }
      // 对比集合，记录删除资源关联
      if (cur_rrArr.size() > 0) {
        List<RoleResource> del_roleResources = new ArrayList<>();
        for (RoleResource roleResource : cur_rrArr) {
          if (selectedIds.contains(roleResource.getResource().getId())) {
            selectedIds.remove(roleResource.getResource().getId());
          } else {
            del_roleResources.add(roleResource);
          }
        }
        roleResourceService.deleteInBatch(del_roleResources);
      }
      // 对比后剩余选中角色selected_roleIds执行新增操作
      if (selectedIds.size() > 0) {
        List<RoleResource> add_roleResources = new ArrayList<>();
        for (String rourceId : selectedIds) {
          RoleResource newRR = new RoleResource();
          newRR.setRole(r);
          newRR.setResource(resourceService.getOne(rourceId)); // 通过Hibernate缓存机制获取
          add_roleResources.add(newRR);
        }
        roleResourceService.saveAll(add_roleResources);
      }
    }

    // 清理用户权限缓存
    permissionService.clearPermission(userService.relevantUser(r).toArray(new String[0]));
  }


  @Override
  @Transactional
  public void addRoleAndBindResource(RoleSelectedDto roleSelectedDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    Role role = roleDao.findByCode(roleSelectedDto.getCode());
    App app = appDao.findByCode(roleSelectedDto.getApp().getAppCode());
    if (null != role) {
      throw new BizException("角色编码已存在", ResultCode.PARAM_ERROR);
    }

    if (null == app) {
      throw new BizException("应用编码不存在", ResultCode.PARAM_ERROR);
    }
    role = new Role();

    BeanUtils.copyProperties(roleSelectedDto, role);
    role.setStatus(BasicDataStatus.VALID);
    role.setOrganizationId(onlineUser.getOrgId());
    role.setApp(app);
    role.setCreator(onlineUser.getName());
    role.setCreateTime(Date.from(Instant.now()));
    role.setUpdateUser(onlineUser.getName());
    role.setUpdateTime(Date.from(Instant.now()));
    roleDao.save(role);
    //绑定权限
    if (!ObjectUtils.isEmpty(roleSelectedDto.getApp())) {

      RoleResourceDto resourceDto = new RoleResourceDto();
      resourceDto.setRole(role);
      resourceDto.setSelectedResourceIds(roleSelectedDto.getApp().getSelected());
      this.updateRoleResource(resourceDto);
    }

    // 清理用户权限缓存
    permissionService.clearPermission(userService.relevantUser(role).toArray(new String[0]));
  }

  @Override
  @Transactional
  public void updateRoleAndBindResource(RoleSelectedDto roleSelectedDto) {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    App app = appDao.findByCode(roleSelectedDto.getApp().getAppCode());
    if (null == app) {
      throw new BizException("应用编码不存在", ResultCode.PARAM_ERROR);
    }
    Role role = roleDao.findOne(roleSelectedDto.getId());
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }

    if (!role.getCode().equals(roleSelectedDto.getCode())) {
      Role temp = roleDao.findByCode(roleSelectedDto.getCode());
      if (null != temp) {
        throw new BizException("角色编码已存在", ResultCode.PARAM_ERROR);
      }
    }
    BeanUtils.copyProperties(roleSelectedDto, role);
    role.setOrganizationId(onlineUser.getOrgId());
    role.setApp(app);
    role.setUpdateUser(onlineUser.getName());
    role.setUpdateTime(Date.from(Instant.now()));
    roleDao.save(role);
    //绑定权限
    if (!ObjectUtils.isEmpty(roleSelectedDto.getApp())) {
      RoleResourceDto resourceDto = new RoleResourceDto();
      resourceDto.setRole(role);
      resourceDto.setSelectedResourceIds(roleSelectedDto.getApp().getSelected());
      this.updateRoleResource(resourceDto);
    }
    // 清理用户权限缓存
    permissionService.clearPermission(userService.relevantUser(role).toArray(new String[0]));

  }

  @Override
  @Transactional
  public void deleteRoleAndBindResource(String roleId) {
    Role role = roleDao.findOne(roleId);
    if (null == role) {
      throw new BizException("角色不存在!", ResultCode.PARAM_ERROR);
    }

    if (PermissionUtils.checkMaster(role)) {
      roleDao.delete(role);
    }
    roleV2Dao.deleteAllByRoleId(roleId);

    // 清理用户权限缓存
    permissionService.clearPermission(userService.relevantUser(role).toArray(new String[0]));
  }
}
