package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.utils.Constants;
import cn.quantgroup.big.stms.sys.dto.OrganizationDto;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.web.bind.annotation.*;

import javax.persistence.criteria.Predicate;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @Author meng.xin@quantgroup.cn
 * @Date 2019-10-20 13:25
 * @Version 1.0
 */
@RestController
@RequestMapping("/organization")
public class OrganizationController extends BaseController {

    @Autowired
    private OrganizationService service;

    @RequestMapping(value = "/get/{id}", method = RequestMethod.POST)
    public Result get(@PathVariable String id) {
        return Result.success(service.findById(id));
    }

    @RequestMapping(value = "/listall", method = RequestMethod.POST)
    public Result listAll() {
        List<Organization> list = service.findAll(getSort());
        List<Organization> validList = list.stream().filter(t -> t.getStatus() == BasicDataStatus.VALID).collect(Collectors.toList());
        return Result.success(validList);
    }

    @RequestMapping(value = "/list/{page}/{size}", method = RequestMethod.POST)
    public Result list(@RequestBody Organization organization, @PathVariable("page") int page, @PathVariable("size") int size) {
        PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
        Page<Organization> organizations = service.findAll((root, query, cb) -> {
            Organization parent = organization.getParent();
            // 未选中父节点 & 没有查询条件 --> 查询 所有一级节点
            if (null == parent || StringUtils.isBlank(parent.getId())) {
                if (StringUtils.isBlank(organization.getCode()) && StringUtils.isBlank(organization.getName())) {
                    return cb.isNull(root.get("parent").get("id"));
                }
            }
            // 选中父节点 || 有查询条件 --> 查询 选择节点子节点 || 所有节点
            List<Predicate> predicates = new ArrayList<>();
            if (StringUtils.isNotBlank(organization.getCode())) {
                predicates.add(cb.like(root.get("code"), "%" + organization.getCode() + "%"));
            }
            if (StringUtils.isNotBlank(organization.getName())) {
                predicates.add(cb.like(root.get("name"), "%" + organization.getName() + "%"));
            }
            if (null != parent && (StringUtils.isNotBlank(parent.getId()))) {
//              predicates.add(cb.equal(root.get("parent").get("id"), parent.getId()));
            	// 获取父节点本身
                predicates.add(cb.or(cb.equal(root.get("parent").get("id"), parent.getId()), cb.equal(root.get("id"), parent.getId())));
            }
            Predicate[] pre = new Predicate[predicates.size()];
            return query.where(predicates.toArray(pre)).getRestriction();
        }, pageRequest);

        // 选中父节点 && 没有查询条件 --> 未查询到子节点，返回选中节点
//        if (organizations.getTotalElements() == 0) {
//            if (StringUtils.isBlank(organization.getCode()) && StringUtils.isBlank(organization.getName())) {
//                if (null != organization.getParent() && StringUtils.isNotBlank(organization.getParent().getId())) {
//                    return Result.success(service.findAll((root, query, cb) -> {
//                        return cb.equal(root.get("id"), organization.getParent().getId());
//                    }, pageRequest));
//                }
//            }
//        }

        return Result.success(organizations);
    }

    @RequestMapping(value = "/addnew", method = RequestMethod.POST)
    public Result addNew(@Valid @RequestBody OrganizationDto organizationDto, HttpServletRequest request) {
        Organization organization = service.findByCode(organizationDto.getCode());
        if (null != organization) {
            throw new BizException("组织编码已存在", ResultCode.PARAM_ERROR);
        }
        Organization byName = service.findByName(organizationDto.getName());
        if(Objects.nonNull(byName)){
            throw new BizException("组织名已存在", ResultCode.PARAM_ERROR);
        }
        organization = new Organization();
        BeanUtils.copyProperties(organizationDto, organization);
        // 添加个校验,羊小咩商品供应商(a6ed0680-849d-4821-bb8e-c3398ddb1b5a)下添加组织，
        // 需要 在供应链 下先添加商户,并且通过名称校验
        if(Objects.nonNull(organizationDto) && Objects.nonNull(organizationDto.getParent())
                && Objects.equals(organizationDto.getParent().getId(),"a6ed0680-849d-4821-bb8e-c3398ddb1b5a")){
            String code = service.validateSupplierName(request.getHeader(Constants.ACCESS_TOKEN), organization.getName());
            if(StringUtils.isEmpty(code)){
                throw new BizException("此供应商不存在，请先添加供应商", ResultCode.ORGANIZATION_FROZEN);
            }
            // 将组织编码设置为商户系统的供应商编码
            organization.setSupplierCode(code);
        }

        service.save(organization);
        return Result.success();
    }

    @RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
    public Result delete(@PathVariable String id) {
        Organization organization = service.findById(id);
        if (null == organization) {
            throw new BizException("组织机构不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.SAVE != organization.getStatus()) {
            throw new BizException("新增状态才能删除", ResultCode.PARAM_ERROR);
        }
        service.delete(organization);
        return Result.success();
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result update(@Valid @RequestBody OrganizationDto organizationDto) {
        Organization organization = service.findById(organizationDto.getId());
        if (null == organization) {
            throw new BizException("组织机构不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.SAVE != organization.getStatus()) {
            throw new BizException("新增状态才能修改", ResultCode.PARAM_ERROR);
        }
        // 校验组织编码是否修改
        if (!organization.getCode().equals(organizationDto.getCode())) {
            Organization temp = service.findByCode(organizationDto.getCode());
            if (null != temp) {
                throw new BizException("组织编码已存在", ResultCode.PARAM_ERROR);
            }
        }
        BeanUtils.copyProperties(organizationDto, organization, "parent");// 上级组织结构不能修改
        service.save(organization);
        return Result.success();
    }

    @RequestMapping(value = "/submit/{id}", method = RequestMethod.POST)
    public Result submit(@PathVariable String id) {
        Organization organization = service.findById(id);
        if (null == organization) {
            throw new BizException("组织机构不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.SAVE != organization.getStatus()) {
            throw new BizException("新增状态才能提交", ResultCode.PARAM_ERROR);
        }
        service.submit(organization);
        return Result.success();
    }

    @RequestMapping(value = "/enabled/{id}", method = RequestMethod.POST)
    public Result enabled(@PathVariable String id) {
        Organization organization = service.findById(id);
        if (null == organization) {
            throw new BizException("组织机构不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.INVALID != organization.getStatus()) {
            throw new BizException("无效状态才能启用", ResultCode.PARAM_ERROR);
        }
        service.enabled(organization);
        return Result.success();
    }

    @RequestMapping(value = "/disabled/{id}", method = RequestMethod.POST)
    public Result disabled(@PathVariable String id) {
        Organization organization = service.findById(id);
        if (null == organization) {
            throw new BizException("组织机构不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.VALID != organization.getStatus()) {
            throw new BizException("有效状态才能禁用", ResultCode.PARAM_ERROR);
        }
        service.disabled(organization);
        return Result.success();
    }

    @RequestMapping(value = "/tree", method = RequestMethod.POST)
    public Result getOrganizationTree() {
        List<Organization> orgList = (List<Organization>) service.findAll(getSort());
        return Result.success(service.getTree(orgList));
    }

    @RequestMapping(value = "/treeValid", method = RequestMethod.POST)
    public Result getOrganizationTreeByValid() {
    	Organization exo = new Organization();
    	exo.setStatus(BasicDataStatus.VALID);
    	Example<Organization> example = Example.of(exo, getExampleMatcher());
    	return Result.success(service.getTree(service.findAll(example, getSort())));
    }
}
