package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.config.IAuthentication;
import cn.quantgroup.big.stms.common.constants.StmsConstants;
import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.enums.TenantEnum;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.PermissionService;
import cn.quantgroup.big.stms.common.utils.IpAddrUtils;
import cn.quantgroup.big.stms.sys.dto.TokenDto;
import cn.quantgroup.big.stms.sys.model.Organization;
import cn.quantgroup.big.stms.sys.model.Tenant;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.service.*;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import com.google.code.kaptcha.impl.DefaultKaptcha;
import com.google.common.collect.Lists;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.validator.constraints.NotBlank;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.imageio.ImageIO;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.validation.constraints.NotNull;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Controller
@RequestMapping("/oauth")
@Slf4j
public class OauthController extends BaseController {

  @Autowired
  private DefaultKaptcha defaultKaptcha;

  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  @Autowired
  private UserService userService;

  @Autowired
  @Qualifier("LDAPAuthentication")
  private IAuthentication authentication;

  @Autowired
  private PermissionService permissionService;

  @Autowired
  private TenantService tenantService;

  @Autowired
  private RoleService service;
  @Autowired
  private RoleResourceService roleResourceService;

  @Autowired
  private OrganizationService organizationService;

  @Autowired
  private ConfigUtils configUtils;


  /**
   * 验证码过期时间 单位秒
   **/
  private final static long CAPTCHA_EXPIRE = 120;
  /**
   * 访问token过期时间 单位秒
   **/
  private final static long ACCESS_TOKEN_EXPIRE = 3600L * 2;
  /**
   * 刷新token过期时间 单位秒
   **/
  private final static long REFRESH_TOKEN_EXPIRE = 3600L * 24 * 30;

  //redis key 前缀
  private final static String CAPTCHA_KEY = "st:captcha:";
  private final static String TOKEN_KEY = "st:token:";

  /**
   * 获取登录验证码
   *
   * @param request
   * @param response
   * @throws Exception
   */
  @RequestMapping("/captcha")
  public void defaultKaptcha(@NotNull @RequestParam(name = "username") String username,
      HttpServletRequest request, HttpServletResponse response)
      throws Exception {
    log.info("用户[{}]获取登录验证码", username);
    byte[] captchaChallengeAsJpeg = null;
    ByteArrayOutputStream jpegOutputStream = new ByteArrayOutputStream();
    try {
      // 生成验证码字符串并保存到redis中
      String createText = defaultKaptcha.createText();
      stringRedisTemplate.opsForValue()
          .set(CAPTCHA_KEY + username, createText, CAPTCHA_EXPIRE, TimeUnit.SECONDS);
      // 生成的验证码字符串返回一个BufferedImage对象并转为byte写入到byte数组中
      BufferedImage challenge = defaultKaptcha.createImage(createText);
      ImageIO.write(challenge, "jpg", jpegOutputStream);
    } catch (IllegalArgumentException e) {
      response.sendError(HttpServletResponse.SC_NOT_FOUND);
      return;
    }

    // 定义response输出类型为image/jpeg类型，使用response输出流输出图片的byte数组
    captchaChallengeAsJpeg = jpegOutputStream.toByteArray();
    response.setHeader("Cache-Control", "no-store");
    response.setHeader("Pragma", "no-cache");
    response.setDateHeader("Expires", 0);
    response.setContentType("image/jpeg");
    ServletOutputStream responseOutputStream = response.getOutputStream();
    responseOutputStream.write(captchaChallengeAsJpeg);
    responseOutputStream.flush();
    responseOutputStream.close();
  }

  /**
   * 获取登录验证码
   *
   * @param request
   * @throws Exception
   */
  @PostMapping("/login")
  @ResponseBody
  public Result login(@NotNull @RequestParam(name = "username") String username,
      @NotNull @RequestParam(name = "password") String password,
      @RequestParam(name = "vrifycode", required = false) String vrifycode,
      HttpServletRequest request) throws Exception {
    log.info("用户[{}]登录", username);
    if (StringUtils.isNotBlank(vrifycode)) {
      if (!username.equals(stringRedisTemplate.opsForValue().get(CAPTCHA_KEY + username))) {
        return Result.failure(ResultCode.INVALID_VERIFICATION_CODE);
      }
    }

    //判断用户是否存在
    User user = userService.getByAccount(username);
    if (null == user) {
      return Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD);
    }

    if (!BasicDataStatus.VALID.getValue().equals(user.getStatus().getValue())) {
      return Result.failure(ResultCode.FORBIDDEN, "请联系管理员开通权限！");
    }

    if (!user.getIsEnable()) {
      return Result.failure(ResultCode.FORBIDDEN, "请联系管理员开通权限！");
    }

    //验证密码是否正确
    boolean isPass = userService.checkLogin(username, password);
    if (!isPass) {
      //登录验证
      isPass = authentication.authenticate(username, password);
      if (!isPass) {
        return Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD);
      }
    }

    // 同一个用户，不同的系统，公用一个token
    // 根据用户id获取 用户token
    String accessToken = stringRedisTemplate.opsForValue().get(TOKEN_KEY + user.getId());

    if (StringUtils.isBlank(accessToken)) {
      accessToken = UUID.randomUUID().toString();
    }
    String refreshToken = UUID.randomUUID().toString();
    long now = System.currentTimeMillis();
    Long atExpire = now + ACCESS_TOKEN_EXPIRE * 1000L;
    Long rtExpire = now + REFRESH_TOKEN_EXPIRE * 1000L;

    String orgId = "";
    String orgName = "";
    String supplierCode = user.getSupplierCode();

    if (null != user.getOrganization()) {
      Organization current = organizationService.findById(user.getOrganization().getId());

      if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
        return Result.failure(ResultCode.ORGANIZATION_FROZEN, "请联系管理员开通权限！");
      }

      orgId = user.getOrganization().getId();
      orgName = user.getOrganization().getName();

      if(StringUtils.isEmpty(supplierCode) && StringUtils.isNotEmpty(user.getOrganization().getId())){
        Organization organization = organizationService.findById(user.getOrganization().getId());
        supplierCode = organization.getSupplierCode();
      }
    }

    String ip = IpAddrUtils.getIpAddr(request);


    OnlineUser onlineUser = new OnlineUser(user.getId(), user.getAccount(), user.getName(), orgId,
        orgName, ip, String.valueOf(user.getTenantId()), supplierCode);
    TokenDto tokenDto = new TokenDto();
    tokenDto.setAccessToken(accessToken);
    tokenDto.setAtExpire(atExpire.toString());
    tokenDto.setRefreshToken(refreshToken);
    tokenDto.setRtExpire(rtExpire.toString());
    configUtils.saveLoginToken(onlineUser, tokenDto);

    permissionService.refreshPermission(user.getId(), null);

    return Result.success(tokenDto);
  }


  /**
   * 获取新的访问token
   *
   * @param refreshtoken
   */
  @RequestMapping("/refreshtoken")
  @ResponseBody
  public Result refreshtoken(@NotNull @RequestParam(name = "refreshtoken") String refreshtoken) {

    String userInfo = stringRedisTemplate.opsForValue().get(TOKEN_KEY + refreshtoken);
    //refreshtoken 过期后，需要重新登录
    if (StringUtils.isBlank(userInfo)) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }

    // 根据用户id获取 用户token
    OnlineUser onlineUser = new OnlineUser(userInfo);
    String accessToken = stringRedisTemplate.opsForValue().get(TOKEN_KEY + onlineUser.getId());

    if (StringUtils.isBlank(accessToken)) {
      accessToken = UUID.randomUUID().toString();
    }

    long now = System.currentTimeMillis();
    Long atExpire = now + ACCESS_TOKEN_EXPIRE * 1000L;

    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + onlineUser.getId(), accessToken, ACCESS_TOKEN_EXPIRE, TimeUnit.SECONDS);
    stringRedisTemplate.opsForValue()
        .set(TOKEN_KEY + accessToken, userInfo, ACCESS_TOKEN_EXPIRE, TimeUnit.SECONDS);

    Map<String, String> data = new HashMap<>();
    data.put("accessToken", accessToken);
    data.put("atExpire", atExpire.toString());

    return Result.success(data);
  }

  /**
   * 获取新的访问token
   *
   * @param accessToken
   * @throws Exception
   */
  @RequestMapping("/currentuserinfo")
  @ResponseBody
  public Result getCurrentuserinfo(@RequestHeader(value = "Access-Token") String accessToken)
      throws Exception {
    log.info("通过【{}】获取用户信息 ", accessToken);
    String userInfo = stringRedisTemplate.opsForValue().get(TOKEN_KEY + accessToken);

    //accessToken 过期后，需要重新获取新的accessToken
    if (StringUtils.isBlank(userInfo)) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }

    Gson gson = new Gson();
    Map<?, ?> map = gson.fromJson(userInfo, Map.class);

    String tenantIds = (String) map.get("tenantIds");
    Map<String, Object> data = new HashMap<>();
    if (StringUtils.isNotEmpty(tenantIds)) {
      List<Tenant> tenants = Lists.newArrayList(
          tenantService.findAll(Arrays.stream(tenantIds.split(",")).map(
              Integer::valueOf).collect(
              Collectors.toList())));
      data.put("tenantIds", tenants);
    }

    data.put("id", map.get("id"));
    data.put("account", map.get("account"));
    data.put("name", map.get("name"));
    data.put("orgId", map.get("orgId"));
    data.put("orgName", map.get("orgName"));

    return Result.success(data);
  }

  /**
   * 通过服务转发获取鉴权【给外部用的】 获取当前登录用户信息
   *
   * @return
   */
  @PostMapping("/api/onlineuserinfo")
  @ResponseBody
  public Result getOnlineUserInfoByToken() {
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    return Result.success(onlineUser.toJson());
  }

  /**
   * 通过服务转发获取鉴权【给外部用的】 判断请求的URL是否有权限
   *
   * @param referer  请求URL
   * @param tenantId 租户id
   * @return
   */
  @GetMapping("/api/permission")
  @ResponseBody
  public Result checkPermission(@RequestHeader(value = "Referer") @NotBlank String referer,
      Integer tenantId) {
    if (Objects.isNull(tenantId)) {
      tenantId = TenantEnum.KDSP.getTenantId();
    }
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser == null) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }else{
      if(StringUtils.isNotEmpty(onlineUser.getOrgId())){
        Organization current = organizationService.findById(onlineUser.getOrgId());
        if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
          return Result.failure(ResultCode.ORGANIZATION_FROZEN, "请联系管理员开通权限！");
        }
      }
    }
    log.info("用户:{},请求路径:{}", onlineUser.getName(), referer);
    if (permissionService.checkUri(tenantId, referer)) {
      // 通过redis中缓存的权限匹配
      return Result.success(null);
    } else {
      return Result.failure(ResultCode.FORBIDDEN);
    }
  }

  /**
   * 提供给供应链 接口权限查询,判断用户类型,如果是商家 需要进行权限校验
   *
   * @param referer
   * @param tenantId
   * @return
   */
  @GetMapping("/api/supply/permission")
  @ResponseBody
  public Result supplyCheckPermission(@RequestHeader(value = "Referer") @NotBlank String referer,
      Integer tenantId) {
    if (Objects.isNull(tenantId)) {
      tenantId = TenantEnum.KDSP.getTenantId();
    }
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser == null) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }else{
      if(StringUtils.isNotEmpty(onlineUser.getOrgId())){
        Organization current = organizationService.findById(onlineUser.getOrgId());
        if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
          return Result.failure(ResultCode.ORGANIZATION_FROZEN, "请联系管理员开通权限！");
        }
      }
    }
    // 超级管理员 越过权限
    if (onlineUser.getId().equalsIgnoreCase(StmsConstants.ADMINISTRATOR_ID)) {
      return Result.success(onlineUser.toJson());
    }
    // 通过组织id判断是商家
    if (organizationService.validateIsSupplyMerchant(onlineUser.getOrgId())) {
      log.info("supply用户:{},请求路径:{}", onlineUser.getName(), referer);
      if (permissionService.checkUri(tenantId, referer)) {
        // 通过redis中缓存的权限匹配
        return Result.success(onlineUser.toJson());
      } else {
        return Result.failure(ResultCode.FORBIDDEN);
      }
    }
    return Result.success(onlineUser.toJson());
  }
}
