package cn.quantgroup.big.stms.common.aop;

import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.utils.IpAddrUtils;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;

/**
 * 访问日志记录
 * @author rong yang
 *
 */
@Aspect
@Component
public class AccessLogAspect {

    private final static Logger logger = LoggerFactory.getLogger(AccessLogAspect.class);
    
    @Autowired
    private ObjectMapper objectMapper;

    /** 以 controller 包下定义的所有请求为切入点 */
    @Pointcut("execution(public * cn.quantgroup.big.stms..controller.*.*(..))  && !execution(public * cn.quantgroup.big.stms.sys.controller.OauthController.login(..))")
    public void accessPointcut() {}

    /**
     * 在切点之前织入
     * @param joinPoint
     * @throws Throwable
     */
    @Before("accessPointcut()")
    public void doBefore(JoinPoint joinPoint) throws Throwable {
        ServletRequestAttributes attributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = attributes.getRequest();

        logger.info("-------------------------------------------------- start --------------------------------------------------");
        logger.info("IP             : {}", IpAddrUtils.getIpAddr(request));
        
        OnlineUser onlineUser = AppContextHolder.getOnlineUser();
        if(null != onlineUser) {
        	logger.info("Access User    : {}", onlineUser.getAccount());
        }
        
        logger.info("URL            : {}", request.getRequestURL().toString());
        logger.info("HTTP Method    : {}", request.getMethod());
        logger.info("Class Method   : {}.{}", joinPoint.getSignature().getDeclaringTypeName(), joinPoint.getSignature().getName());
        logger.info("Request Args   : {}", tojson(joinPoint.getArgs()));
    }

    @SuppressWarnings("unchecked")
	private final String tojson(Object[] args) throws Throwable{
    	if(null == args) {
    		return null;
    	}
    	
    	@SuppressWarnings("rawtypes")
		List printArgs = new ArrayList();
    	for(Object obj : args) {
    		if(obj instanceof ServletRequest || obj instanceof ServletResponse) {
    			continue;
    		}
    		printArgs.add(obj);
    	}
    	
    	return objectMapper.writeValueAsString(printArgs);
    	
    }
    /**
     * 在切点之后织入
     * @throws Throwable
     */
    @After("accessPointcut()")
    public void doAfter() throws Throwable {
        logger.info("--------------------------------------------------- end ---------------------------------------------------");
    }

    /**
     * 环绕
     * @param proceedingJoinPoint
     * @return
     * @throws Throwable
     */
    @Around("accessPointcut()")
    public Object doAround(ProceedingJoinPoint proceedingJoinPoint) throws Throwable {
        long startTime = System.currentTimeMillis();
        Object result = proceedingJoinPoint.proceed();
        logger.debug("Response Data  : {}", objectMapper.writeValueAsString(result));
        logger.info("Time-Consuming : {} ms", System.currentTimeMillis() - startTime);
        return result;
    }

}
