package cn.quantgroup.big.stms.sys.interceptor;

import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.context.SpringContextHolder;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.utils.Constants;
import cn.quantgroup.big.stms.sys.service.PermissionV2Service;
import com.google.gson.Gson;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Slf4j
@Component
public class TokenInterceptor implements HandlerInterceptor {

  private final static String TOKEN_KEY = "st:token:";
  private final PermissionV2Service permissionV2Service;

  private final List<String> excludes = new ArrayList<>();

  public TokenInterceptor(PermissionV2Service permissionV2Service) {
    this.permissionV2Service = permissionV2Service;
    excludes.add("/v2/resource/APP013");
    excludes.add("/v2/oauth/currentuserinfo");
    excludes.add("/v2/oauth/permission");
    excludes.add("/v2/user/password/reset");
    excludes.add("/v2/oauth/wechat/bindphone");
    excludes.add("/v2/user/password/validate");
    excludes.add("/v2/role/current");

  }


  @Override
  public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler)
      throws Exception {
    String accessToken = request.getHeader(Constants.ACCESS_TOKEN);

    Result result = null;
    if (StringUtils.isBlank(accessToken)) {
      log.error("请求地址:{}",request.getRequestURI());
      result = Result.failure(ResultCode.UNAUTHORIZED);
    }

    String userInfo = null;
    if (null == result) {
      //检查token是否存在，从redis中检查
      RedisTemplate<String, String> stringRedisTemplate = SpringContextHolder.getBean(
          "stringRedisTemplate", RedisTemplate.class);
      userInfo = stringRedisTemplate.opsForValue().get(TOKEN_KEY + accessToken);

      //accessToken 过期后，需要重新获取新的accessToken
      if (StringUtils.isBlank(userInfo)) {
        result = Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
      }

    }

    if (null != result) {
      PrintWriter writer = null;
      response.setCharacterEncoding("UTF-8");
      response.setContentType("application/json; charset=utf-8");
      try {
        writer = response.getWriter();
        String data = new Gson().toJson(result);
        writer.print(data);
        return false;
      } catch (IOException e) {
        log.error("access token check error", e);
      } finally {
        if (writer != null) {
          writer.close();
        }
      }
    }

    //通过access_token获取对应的人员信息
    OnlineUser onlineUser = new OnlineUser(userInfo);
    AppContextHolder.setOnlineUser(onlineUser);

    //新的接口需要验证接口权限
    if (!excludes.contains(request.getRequestURI())&&request.getRequestURI().startsWith("/v2")) {
      boolean p =  permissionV2Service.checkPermission(Integer.parseInt(onlineUser.getTenantIds()),
          onlineUser.getId(), Collections.singletonList(request.getRequestURI()));
      if(p){
        return true;
      }else{
        log.error("请求地址:{},userId:{},account:{}",request.getRequestURI(),onlineUser.getId(),onlineUser.getAccount());

        throw new BizException(ResultCode.FORBIDDEN);
      }
    }

    return true;
  }

  @Override
  public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler,
      ModelAndView modelAndView) throws Exception {
    // TODO Auto-generated method stub

  }

  @Override
  public void afterCompletion(HttpServletRequest request, HttpServletResponse response,
      Object handler, Exception ex) throws Exception {
    AppContextHolder.remove();
  }

}
