package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.config.IAuthentication;
import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.enums.TenantEnum;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.service.MailService;
import cn.quantgroup.big.stms.sys.dto.TenantDto;
import cn.quantgroup.big.stms.sys.dto.UserDto;
import cn.quantgroup.big.stms.sys.dto.UserRoleDto;
import cn.quantgroup.big.stms.sys.model.*;
import cn.quantgroup.big.stms.sys.service.*;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Example;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import java.util.*;
import java.util.stream.Collectors;

@RestController
@RequestMapping("/user")
@Validated
public class UserController extends BaseController {

  private static final int SALT_LENGTH = 16;// 加密盐长度
  private static final int PASSWORD_LENGTH_MIN = 13;// 密码最小长度
  private static final int PASSWORD_LENGTH_MAX = 18;// 密码最大长度
  private static final String SECRET = "*************";// 代替敏感信息
  private static final String SUBJECT_OPEN = "账户开通信息";
  private static final String SUBJECT_RESET = "账号密码重置";
  // 邮件模板
  private static final String MAIL_TEMPLATE = "账号：#account#\n" +
      "密码：#password#\n" +
      "支持公司邮箱、密码登录";

  public static final String ACCESS_TOKEN = "Access-Token";

  @Autowired
  private UserService service;
  @Autowired
  private PasswordService passwordService;
  @Autowired
  private MailService mailService;
  @Autowired
  private ConfigUtils configUtils;
  @Autowired
  private AppService appService;
  @Autowired
  private RoleService roleService;
  @Autowired
  private TenantService tenantService;
  @Autowired
  private OrganizationService organizationService;

  @Autowired
  @Qualifier("LDAPAuthentication")
  private IAuthentication authentication;


  @RequestMapping(value = "/get/{id}", method = RequestMethod.POST)
  public Result get(@PathVariable String id) {
    return Result.success(replace(service.findById(id)));
  }

  @RequestMapping(value = "/list/{page}/{size}", method = RequestMethod.POST)
  public Result list(@RequestBody User user, @PathVariable int page, @PathVariable int size) {
    PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
    return Result.success(replace(service.findAllByOrgChildren(user, pageRequest)));
  }

  /**
   * 走转换器的情况
   *
   * @param user
   * @param page
   * @param size
   * @return
   */
  @RequestMapping(value = "/listconvert/{page}/{size}", method = RequestMethod.POST)
  public Result listConvert(@ModelAttribute User user, @PathVariable int page,
      @PathVariable int size) {
    PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
    return Result.success(replace(service.findAllByOrgChildren(user, pageRequest)));
  }

  /**
   * 替换数据中敏感信息: 密码 & 加密盐
   *
   * @param page
   * @return
   */
  private Object replace(Page<User> page) {
    if (null != page && page.getNumberOfElements() > 0) {
      List<User> list = page.getContent();
      list.forEach((User user) -> {
        replace(user);
      });
    }
    return page;
  }

  private Object replace(User user) {
    if (null != user) {
      user.setPassword(SECRET);
      user.setSalt(SECRET);
    }
    return user;
  }

  @RequestMapping(value = "/addnew", method = RequestMethod.POST)
  public Result addNew(@Valid @RequestBody UserDto userDto, HttpServletRequest request) {
    User user = service.findByCode(userDto.getCode());
    if (null != user) {
      throw new BizException("用户编码已存在", ResultCode.PARAM_ERROR);
    }
    user = service.getByAccount(userDto.getAccount());
    if (null != user) {
      throw new BizException("登录账号已存在", ResultCode.PARAM_ERROR);
    }
    user = new User();
    BeanUtils.copyProperties(userDto, user);
    if(StringUtils.isNotEmpty(userDto.getId())){
      user.setId(userDto.getId());
    }else{
      user.setId(UUID.randomUUID().toString());
    }
    // 暂时 先加个默认的 租户ID 560761
    user.setTenantId(TenantEnum.KDSP.getTenantId());
    //供应链的用户，有supplierCode给用户附上
    Organization organization = organizationService.findById(user.getOrganization().getId());
    if(StringUtils.isNotEmpty(organization.getSupplierCode())) {
      user.setSupplierCode(organization.getSupplierCode());
    }

    String password = user.getPassword();
    // 设置密码
    setPassword(user);
    String accessToken = request.getHeader(ACCESS_TOKEN);
    service.saveUser(user, accessToken);
    // 发送密码邮件
    sendMail(user.getAccount(), password, user.getEmail(), SUBJECT_OPEN);
    return Result.success();
  }

  /**
   * 设置用户密码
   *
   * @param user
   */
  private void setPassword(User user) {
    String salt = getSalt(SALT_LENGTH);
    String password = DigestUtils.sha512Hex(user.getPassword() + salt);
    user.setSalt(salt);
    user.setPassword(password);
  }

  /**
   * 获取指定长度的加密盐
   *
   * @param length
   * @return
   */
  public String getSalt(int length) {
    return passwordService.getSalt(length);
  }

  @RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
  public Result delete(@PathVariable String id) {
    User user = service.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.SAVE != user.getStatus()) {
      throw new BizException("新增状态才能删除", ResultCode.PARAM_ERROR);
    }
    service.delete(user);
    return Result.success();
  }

  @RequestMapping(value = "/update", method = RequestMethod.POST)
  public Result update(@Valid @RequestBody UserDto userDto) {
    User user = service.findById(userDto.getId());
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.SAVE != user.getStatus()) {
      throw new BizException("新增状态才能修改", ResultCode.PARAM_ERROR);
    }
    // 校验用户编码是否修改
    if (!user.getCode().equals(userDto.getCode())) {
      User temp = service.findByCode(userDto.getCode());
      if (null != temp) {
        throw new BizException("用户编码已存在", ResultCode.PARAM_ERROR);
      }
    }
    // 校验用户登录账号是否修改
    if (!user.getAccount().equals(userDto.getAccount())) {
      User temp = service.getByAccount(userDto.getAccount());
      if (null != temp) {
        throw new BizException("登录账号已存在", ResultCode.PARAM_ERROR);
      }
    }
    BeanUtils.copyProperties(userDto, user, "organization", "password");// 所属组织、密码不能修改
    service.save(user);
    return Result.success();
  }

  @RequestMapping(value = "/submit/{id}", method = RequestMethod.POST)
  public Result submit(@PathVariable String id) {
    User user = service.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.SAVE != user.getStatus()) {
      throw new BizException("新增状态才能提交", ResultCode.PARAM_ERROR);
    }
    service.submit(user);
    return Result.success();
  }

  @RequestMapping(value = "/enabled/{id}", method = RequestMethod.POST)
  public Result enabled(@PathVariable String id) {
    User user = service.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.INVALID != user.getStatus()) {
      throw new BizException("无效状态才能启用", ResultCode.PARAM_ERROR);
    }
    service.enabled(user);
    return Result.success();
  }

  @RequestMapping(value = "/disabled/{id}", method = RequestMethod.POST)
  public Result disabled(@PathVariable String id) {
    User user = service.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    if (BasicDataStatus.VALID != user.getStatus()) {
      throw new BizException("有效状态才能禁用", ResultCode.PARAM_ERROR);
    }
    service.disabled(user);
    return Result.success();
  }

  /**
   * 量星球-获取密码
   *
   * @return
   */
  @RequestMapping(value = "/getpassword", method = RequestMethod.GET)
  public Result getPassword() {
    return Result.success(passwordService.getPassword(PASSWORD_LENGTH_MAX));
  }

  /**
   * 量星球-更改密码
   *
   * @param map
   * @return
   */
  @RequestMapping(value = "/resetpassword", method = RequestMethod.POST)
  public Result resetPassword(@RequestBody HashMap<String, String> map) {
    if (null == map) {
      throw new BizException("请求参数不能为空", ResultCode.PARAM_ERROR);
    }
    String password = map.get("password");
    validatePassword(password);
    String id = map.get("id");
    if (StringUtils.isBlank(id)) {
      throw new BizException("用户ID不能为空", ResultCode.PARAM_ERROR);
    }
    User user = service.findById(id);
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    user.setPassword(password);
    // 设置密码
    setPassword(user);
    service.save(user);
    // 发送密码邮件
    sendMail(user.getAccount(), password, user.getEmail(), SUBJECT_RESET);
    return Result.success();
  }

  /**
   * 量星球-校验密码规则
   *
   * @param map
   * @return
   */
  @PostMapping("/ex/validate-password-rule")
  public Result validatePasswordRule(@RequestBody HashMap<String, String> map) {
    String password = map.get("password");
    validatePassword(password);
    return Result.success();
  }

  /**
   * 商户管理后台-修改密码
   *
   * @param map
   * @return
   */
  @PostMapping("/ex/supply/reset-password")
  public Result merchantResetPassword(@RequestBody HashMap<String, String> map) {
    // 获取用户信息
    OnlineUser onlineUser = AppContextHolder.getOnlineUser();
    if (onlineUser == null) {
      return Result.failure(ResultCode.REFRESH_TOKEN_EXPIRE);
    }

    String orgId = onlineUser.getOrgId();
    Organization one = organizationService.findById(orgId);
    if (Objects.isNull(one)) {
      throw new BizException("组织不存在", ResultCode.ORGANIZATION_NO_EXISTENT);
    }

    if (Objects.isNull(one.getParent()) || !Objects.equals(one.getParent().getId(),
        "a6ed0680-849d-4821-bb8e-c3398ddb1b5a")) {
      throw new BizException("非供应商租户", ResultCode.UN_MERCHANT_ORGANIZATION);
    }

    // 校验 新密码规则
    String newPassword = map.get("newPassword");
    validatePassword(newPassword);
    String password = map.get("password");

    String account = onlineUser.getAccount();

    //验证密码是否正确
    boolean isPass = service.checkLogin(account, password);
    if (!isPass) {
      //登录验证
      isPass = authentication.authenticate(account, password);
      if (!isPass) {
        return Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD_ORIGINAL);
      }
    }

    // 设置新密码
    User user = service.findById(onlineUser.getId());
    if (null == user) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    user.setPassword(newPassword);
    // 设置密码
    setPassword(user);
    service.save(user);
    // 发送密码邮件
//        sendMail(user.getAccount(), newPassword, user.getEmail(), SUBJECT_RESET);

    return Result.success();

  }


  /**
   * 校验密码规则
   *
   * @param password
   */
  private void validatePassword(String password) {
    if (StringUtils.isBlank(password)) {
      throw new BizException("新密码不能为空", ResultCode.PARAM_ERROR);
    }
    if (password.length() < PASSWORD_LENGTH_MIN || password.length() > PASSWORD_LENGTH_MAX) {
      throw new BizException(
          "新密码必须在" + PASSWORD_LENGTH_MIN + "-" + PASSWORD_LENGTH_MAX + "之间",
          ResultCode.PARAM_ERROR);
    }
    String reg = "(?=.*\\d)(?=.*[a-z])(?=.*[A-Z])(?=.*[^a-zA-Z0-9]).{13,18}";
    if (!password.matches(reg)) {
      throw new BizException("新密码必须包含大小写字母、数字、特殊字符四类");
    }
  }

  /**
   * 发送邮件
   *
   * @param account  账号
   * @param password 密码
   * @param toEmail  目标邮箱
   * @param subject  主题
   */
  private void sendMail(String account, String password, String toEmail, String subject) {
    String content = MAIL_TEMPLATE.substring(0, MAIL_TEMPLATE.indexOf("#account#")) + account
        + MAIL_TEMPLATE.substring(MAIL_TEMPLATE.indexOf("#account#") + 9);
    content = content.substring(0, content.indexOf("#password#")) + password + content.substring(
        content.indexOf("#password#") + 10);

    mailService.sendSimpleMail(toEmail, subject, content);
  }


  /**
   * 添加租户概念 查询系统所有角色，并标记用户选中的角色
   */
  @RequestMapping(value = "/datacolumnset/list/{userId}/{appId}/{tenantId}", method = RequestMethod.POST)
//  @PostMapping(value = "/datacolumnset/list/{userId}/{appId}")
  public Result dataColumnSetNew(@PathVariable String userId, @PathVariable Integer appId,
      @PathVariable Integer tenantId) {
    JSONObject result = new JSONObject();
    User u = service.findById(userId);
    if (null == u) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }

    List<Role> app_roles = null;
    if (appId == 0) {
      // 查询所有应用下的有效角色
      Role ex_role = new Role();
      ex_role.setStatus(BasicDataStatus.VALID);
      if (tenantId > 0) {
        ex_role.setTenantId(tenantId);
      }
      Example<Role> example = Example.of(ex_role, getExampleMatcher());
      app_roles = roleService.findAll(example);
    } else {
      App app = appService.findById(appId);
      if (null == app) {
        throw new BizException("所选应用不存在", ResultCode.PARAM_ERROR);
      }
      // 查询应用下的所有有效角色
      Role ex_role = new Role();
      ex_role.setApp(app);
      ex_role.setStatus(BasicDataStatus.VALID);
      if (tenantId > 0) {
        ex_role.setTenantId(tenantId);
      }
      Example<Role> example = Example.of(ex_role, getExampleMatcher());
      app_roles = roleService.findAll(example);
      if (app_roles.size() == 0) {
        throw new BizException("所选应用下无有效角色", ResultCode.PARAM_ERROR);
      }
    }
//        result.put("content", app_roles);

    // 已标记角色
    List<UserRole> selected_userRoles = service.getUserRoles(appId, u);
    List<String> selected = new ArrayList<>();
    for (UserRole userRole : selected_userRoles) {
      selected.add(userRole.getRole().getId());
    }

    Map<Integer, String> finalTenantMap = new HashMap<>();

    tenantService.findAll().forEach(i -> {
      finalTenantMap.put(i.getId(), i.getName());
    });

    // 设置角色的 选中状态和 租户id 租户名
    if (CollectionUtils.isNotEmpty(app_roles)) {
      app_roles.forEach(app_role -> {
        app_role.setTenantName(finalTenantMap.get(app_role.getTenantId()));
        app_role.setSelected(selected.contains(app_role.getId()) ? 1 : 0);
      });
    }

    // 对角色进行 租户的分组
    Map<Integer, List<Role>> roleMap = app_roles.stream()
        .collect(Collectors.groupingBy(Role::getTenantId));

    // tenantId =0 表示查询所有,此时,返回有选中的角色
    if (tenantId == 0 && MapUtils.isNotEmpty(roleMap)) {
      Iterator iterator = roleMap.keySet().iterator();
      while (iterator.hasNext()) {
        Integer key = (Integer) iterator.next();
        List<Role> roles = roleMap.get(key);
        long count = roles.stream().filter(role -> Objects.equals(role.getSelected(), 1)).count();
        if (count == 0L) {
          iterator.remove();
        }
      }
    }

//        result.put("selected", selected);

    return Result.success(roleMap);
  }

  /**
   * 查询系统所有角色，并标记用户选中的角色
   */
  @RequestMapping(value = "/datacolumnset/list/{userId}/{appId}", method = RequestMethod.POST)
//  @PostMapping(value = "/datacolumnset/list/{userId}/{appId}")
  public Result dataColumnSet(@PathVariable String userId, @PathVariable Integer appId) {
    JSONObject result = new JSONObject();
    User u = service.findById(userId);
    if (null == u) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }

    List<Role> app_roles = null;
    if (appId == 0) {
      // 查询所有应用下的有效角色
      Role ex_role = new Role();
      ex_role.setStatus(BasicDataStatus.VALID);
      Example<Role> example = Example.of(ex_role, getExampleMatcher());
      app_roles = roleService.findAll(example);
    } else {
      App app = appService.findById(appId);
      if (null == app) {
        throw new BizException("所选应用不存在", ResultCode.PARAM_ERROR);
      }
      // 查询应用下的所有有效角色
      Role ex_role = new Role();
      ex_role.setApp(app);
      ex_role.setStatus(BasicDataStatus.VALID);
      Example<Role> example = Example.of(ex_role, getExampleMatcher());
      app_roles = roleService.findAll(example);
      if (app_roles.size() == 0) {
        throw new BizException("所选应用下无有效角色", ResultCode.PARAM_ERROR);
      }
    }
    result.put("content", app_roles);

    // 已标记角色
    List<UserRole> selected_userRoles = service.getUserRoles(appId, u);
    List<String> selected = new ArrayList<>();
    for (UserRole userRole : selected_userRoles) {
      selected.add(userRole.getRole().getId());
    }
    result.put("selected", selected);

    return Result.success(result);
  }

  /**
   * 给用户分配角色,需要指定应用
   */
  @RequestMapping(value = "/datacolumnset/save", method = RequestMethod.POST)
  public Result dataColumnSetSave(@Valid @RequestBody UserRoleDto userRoleDto) {
    if (StringUtils.isBlank(userRoleDto.getUser().getId())) {
      throw new BizException("所选用户不能为空", ResultCode.PARAM_ERROR);
    }
    User u = service.findById(userRoleDto.getUser().getId());
    if (null == u) {
      throw new BizException("用户不存在", ResultCode.PARAM_ERROR);
    }
    service.updateUserRole(userRoleDto);
    return Result.success();
  }

  /**
   * 获取当前用户，所在应用的所有资源
   *
   * @param appCode 应用编码
   * @return
   */
  @RequestMapping("/getresources/{appCode}")
  public Result getResources(@PathVariable String appCode,
      @RequestHeader(value = "qg-tenant-id", required = false) Integer tenantId) {
    return Result.success(service.getResources(appCode, tenantId));
  }

  /**
   * 获取当前用户，所有分配的应用系统
   *
   * @return
   */
  @RequestMapping("/getapps")
  public Result getApps(@RequestBody(required = false) TenantDto tenantDto) {
    // 租户做兼容
    if (Objects.isNull(tenantDto)) {
      tenantDto = new TenantDto();
    }
    if (Objects.isNull(tenantDto.getTenantId())) {
      tenantDto.setTenantId(TenantEnum.KDSP.getTenantId());
    }
    return Result.success(service.getApps(tenantDto.getTenantId()));
  }

}
