package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.context.OnlineUser;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.model.ClientVO;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.common.utils.Constants;
import cn.quantgroup.big.stms.common.utils.IpAddrUtils;
import cn.quantgroup.big.stms.sys.dto.TokenDto;
import cn.quantgroup.big.stms.sys.model.*;
import cn.quantgroup.big.stms.sys.service.ClientService;
import cn.quantgroup.big.stms.sys.service.OrganizationService;
import cn.quantgroup.big.stms.sys.service.PasswordService;
import cn.quantgroup.big.stms.sys.service.UserService;
import cn.quantgroup.big.stms.sys.utils.ConfigUtils;
import com.google.gson.Gson;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.view.RedirectView;

import javax.servlet.http.HttpServletRequest;
import java.util.*;
import java.util.concurrent.TimeUnit;

@Controller
@RequestMapping("/v1/oauth2")
public class Oauth2Controller {

  @Autowired
  private UserService userService;
  @Autowired
  private ClientService clientService;
  @Autowired
  private ConfigUtils configUtils;
  @Autowired
  private PasswordService passwordService;
  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;
  @Autowired
  private OrganizationService organizationService;
  @Autowired
  private Gson gson;

  @Value("${oauth.authorize.code.length:8}")
  private Integer codeLength;
  @Value("${oauth.authorize.code.expire:300}")
  private Long codeExpire;

  private final static long ACCESS_TOKEN_EXPIRE = 3600L * 2;
  private final static long REFRESH_TOKEN_EXPIRE = 3600L * 24 * 30;
  private final static String DEFAULT_REDIRECT_URL = "/v1/oauth2/code.html";
  private final static String GRANT_TYPE_AUTHORIZATION_CODE = "authorization_code";
  private final static String GRANT_TYPE_REFRESH_TOKEN = "refresh_token";

  @PostMapping("to_login")
  @ResponseBody
  public RedirectView toLogin(@RequestParam(value = "client_id") String clientId,
      @RequestParam(value = "response_type") String responseType,
      @RequestParam(value = "username") String username,
      @RequestParam(value = "password") String password,
      HttpServletRequest request) {
    RedirectView redirectView = new RedirectView();
    redirectView.setContextRelative(true);
    //判断用户是否存在
    User user = userService.getByAccount(username);
    if (null == user) {
      redirectView.setUrl(
          DEFAULT_REDIRECT_URL + "?code=" + ResultCode.INVALID_USERNAME_OR_PASSWORD.getMsg());
      return redirectView;
    }
    if (!BasicDataStatus.VALID.getValue().equals(user.getStatus().getValue())
        || !user.getIsEnable()) {
      redirectView.setUrl(DEFAULT_REDIRECT_URL + "?code=" + "请联系管理员开通权限！");
      return redirectView;
    }
    //判断客户端是否存在
    Client client = clientService.findByClientId(clientId);
    if (Objects.isNull(client)) {
      redirectView.setUrl(DEFAULT_REDIRECT_URL + "?code=" + "请联系管理员开通权限！");
      return redirectView;
    }
    // 判断客户端是否属于该用户
    if (!StringUtils.equals(client.getUserId(), user.getId())) {
      redirectView.setUrl(
          DEFAULT_REDIRECT_URL + "?code=" + ResultCode.INVALID_USERNAME_OR_PASSWORD.getMsg());
      return redirectView;
    }

    // 同一个用户，不同的系统，公用一个token
    // 根据用户id获取 用户token
    String accessToken = stringRedisTemplate.opsForValue().get(Constants.TOKEN_KEY + user.getId());
    if (StringUtils.isBlank(accessToken)) {
      accessToken = UUID.randomUUID().toString();
    }
    String refreshToken = UUID.randomUUID().toString();
    long now = System.currentTimeMillis();
    long atExpire = now + ACCESS_TOKEN_EXPIRE * 1000L;
    long rtExpire = now + REFRESH_TOKEN_EXPIRE * 1000L;
    String orgId = "";
    String orgName = "";
    String ip = IpAddrUtils.getIpAddr(request);

    if (null != user.getOrganization()) {
      Organization current = organizationService.findById(user.getOrganization().getId());
      if (current != null && BasicDataStatus.INVALID.equals(current.getStatus())) {
        redirectView.setUrl(DEFAULT_REDIRECT_URL + "?code=" + "请联系管理员开通权限！");
        return redirectView;
      }
      orgId = user.getOrganization().getId();
      orgName = user.getOrganization().getName();
    }

    // 查询用户权限
    List<UserRole> userRoles = userService.getUserRoles(null, user);
    String userRolesStr = "";
    if (CollectionUtils.isNotEmpty(userRoles)) {
      Set<Integer> tenantIdSet = new HashSet<>();
      userRoles.forEach(userRole -> {
        Role role = userRole.getRole();
        tenantIdSet.add(role.getTenantId());
      });
      userRolesStr = StringUtils.join(tenantIdSet, ",");
    }

    OnlineUser onlineUser = new OnlineUser(user.getId(), user.getAccount(), user.getName(), orgId,
        orgName, ip, userRolesStr, user.getSupplierCode());
    TokenDto tokenDto = new TokenDto();
    tokenDto.setAccessToken(accessToken);
    tokenDto.setAtExpire(Long.toString(atExpire));
    tokenDto.setRefreshToken(refreshToken);
    tokenDto.setRtExpire(Long.toString(rtExpire));
    configUtils.saveLoginToken(onlineUser, tokenDto);

    redirectView.setUrl("/v1/oauth2/authorize?token=" + accessToken + "&client_id=" + clientId);
    return redirectView;
  }


  @RequestMapping("authorize")
  public RedirectView authorize(@RequestParam("token") String token,
      @RequestParam("client_id") String clientId) {
    RedirectView redirectView = new RedirectView();
    redirectView.setContextRelative(true);

    //判断token
    OnlineUser onlineUser = configUtils.getOnlineUserByRefreshToken(token);
    if (Objects.isNull(onlineUser)) {
      redirectView.setUrl(DEFAULT_REDIRECT_URL + "?code=" + "请联系管理员开通权限！");
      return redirectView;
    }
    Client client = clientService.findByClientId(clientId);
    if (!StringUtils.equals(client.getUserId(), onlineUser.getId())) {
      redirectView.setUrl(
          DEFAULT_REDIRECT_URL + "?code=" + ResultCode.INVALID_USERNAME_OR_PASSWORD.getMsg());
      return redirectView;
    }
    String code = getCode(codeLength);
    //保存授权码
    stringRedisTemplate.opsForValue().set(Constants.OAUTH2_CODE_KEY + clientId, code, codeExpire, TimeUnit.SECONDS);
    String redirectUrl = StringUtils.isBlank(client.getRedirectUrl()) ? DEFAULT_REDIRECT_URL
        : client.getRedirectUrl();
    redirectView.setUrl(redirectUrl + "?code=" + code);
    return redirectView;
  }

  @RequestMapping("code.html")
  public ModelAndView redirect(@RequestParam("code") String code) {
    HashMap<String, String> params = new HashMap<>();
    params.put("code", code);
    return new ModelAndView("code", params);
  }

  @PostMapping("token")
  @ResponseBody
  public Result getToken(@RequestParam("client_id") String clientId,
      @RequestParam("client_secret") String clientSecret,
      @RequestParam("grant_type") String grantType,
      @RequestParam(value = "code", required = false) String code,
      @RequestParam(value = "refresh_token", required = false) String refreshToken) {
    Boolean isPass = clientService.checkSecret(clientId, clientSecret);
    if (!isPass){
      Result.failure(ResultCode.INVALID_USERNAME_OR_PASSWORD);
    }
    if (StringUtils.equals(grantType, GRANT_TYPE_AUTHORIZATION_CODE)){
      return Result.success(clientService.getTokenByCode(code, clientId));
    } else if (StringUtils.equals(grantType, GRANT_TYPE_REFRESH_TOKEN)){
      return Result.success(clientService.getTokenByRefreshToken(refreshToken, clientId));
    } else {
      return Result.failure(ResultCode.PARAM_ERROR);
    }
  }

  @PostMapping(value = "/currentclientinfo", produces = {"application/json"})
  @ResponseBody
  public Result currentclientinfo(@RequestHeader(Constants.OAUTH2_ACCESS_TOKEN) String accessToken) {
    Client client = checkAccessToken(accessToken);
    ClientVO clientVO = new ClientVO();
    BeanUtils.copyProperties(client, clientVO);
    return Result.success(clientVO);
  }

  @GetMapping(value = "/user/info/{id}")
  @ResponseBody
  public Result get(@PathVariable String id, @RequestHeader(Constants.OAUTH2_ACCESS_TOKEN) String accessToken) {
    checkAccessToken(accessToken);
    User user = userService.findById(id);
    user.setPassword(null);
    user.setSalt(null);
    return Result.success(user);
  }

  private Client checkAccessToken(String accessToken){
    if (StringUtils.isBlank(accessToken)){
      throw new BizException(ResultCode.UNAUTHORIZED);
    }
    String result = stringRedisTemplate.opsForValue().get(Constants.OAUTH2_TOKEN_KEY + accessToken);
    if (StringUtils.isBlank(result)){
      throw new BizException(ResultCode.ACCESS_TOKEN_EXPIRE);
    }
    return gson.fromJson(result, Client.class);
  }


  private String getCode(int length) {
    StringBuilder result = new StringBuilder(length);
    List<Character> temp = new ArrayList<>(length);
    Random random = new Random(System.nanoTime());
    temp.add((char) (48 + random.nextInt(10)));
    temp.add( (char) (65 + random.nextInt(26)));
    temp.add( (char) (97 + random.nextInt(26)));
    temp.add((char) (48 + random.nextInt(10)));
    temp.add( (char) (65 + random.nextInt(26)));
    temp.add((char) (48 + random.nextInt(10)));
    temp.add( (char) (97 + random.nextInt(26)));
    temp.add((char) (48 + random.nextInt(10)));
    temp.forEach(result::append);
    return result.toString();
  }

}
