package cn.quantgroup.big.stms.sys.service.impl;

import cn.quantgroup.big.stms.common.constants.StmsConstants;
import cn.quantgroup.big.stms.common.context.AppContextHolder;
import cn.quantgroup.big.stms.common.service.PermissionService;
import cn.quantgroup.big.stms.sys.model.Resource;
import cn.quantgroup.big.stms.sys.model.User;
import cn.quantgroup.big.stms.sys.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.AntPathMatcher;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * test 权限校验
 *
 * @author yutong
 * @data 2020年1月19日
 */
@Slf4j
@Service
public class PermissionServiceImpl implements PermissionService {

  /**
   * 访问token过期时间 单位秒
   **/
  private final static long ACCESS_TOKEN_EXPIRE = 3600L * 2;

  private final static String TOKEN_KEY = "st:token:";

  @Autowired
  private UserService userService;


  @Autowired
  @Qualifier("stringRedisTemplate")
  private RedisTemplate<String, String> stringRedisTemplate;

  private final AntPathMatcher pathMatcher = new AntPathMatcher();

  /**
   * TODO 全局忽略权限的接口
   */
  private Set<String> IGNORE_URL = new HashSet<>();

  {
    IGNORE_URL.add("/oauth/**"); // 登录接口
    IGNORE_URL.add("/home/info"); // 主页信息
    IGNORE_URL.add("/user/getresources/**"); // 前端获取指定应用系统的资源
    IGNORE_URL.add("/user/getapps"); // 前端获取当前登录用户的所有分配系统
    IGNORE_URL.add("/dict/getenums/**"); // 数据字典
    //获取租户列表
    IGNORE_URL.add("/tenant/list");
    // 商户后台更改密码
    IGNORE_URL.add("/user/ex/**");
  }

  @Override
  public boolean checkIgnore(String... paths) {
    for (String path : paths) {
      for (String regPermz : IGNORE_URL) {
        if (pathMatcher.match(regPermz, path)) {
          log.debug("忽略权限成功, path:{} --> reg:{}", path, regPermz);
          return true;
        }
      }
    }
    return false;
  }

  @Override
  public boolean checkUsers(String... ids) {
    return false;
  }

  @Override
  public boolean checkUri(Integer tenantId, String... paths) {
    for (String path : paths) {
      for (String regPermz : this.getRegPermission(tenantId, null)) {
        if (pathMatcher.match(regPermz, path)) {
          log.info("匹配权限成功, path:{} --> reg:{}", path, regPermz);
          return true;
        }
      }
    }
    return false;
  }


  @Override
  public Set<String> getRegPermission(Integer tenantId, String userid) {
    userid = StringUtils.isBlank(userid) ? AppContextHolder.getOnlineUser().getId() : userid;
    // 检查token是否存在，从redis中检查
    //TODO：线上授权之后缓存清空失败，暂时直接查询，后续查明在修复
//        String key = ROLE_KEY + userid+":"+tenantId;
//        String roleInfo = stringRedisTemplate.opsForValue().get(key);

//        if (StringUtils.isBlank(roleInfo)) {
//            // 避免重新登录获取权限
    String roleInfo = this.refreshPermission(userid, tenantId);
//        }
    Set<String> permission = new HashSet<>();
    for (String p : roleInfo.split(StmsConstants.SYMBOL_COMMA)) {
      if(StringUtils.isNotEmpty(p)){
        permission.add(p.trim());
      }
    }
    return permission;
  }

  @Override
  public String refreshPermission(String userid, Integer inTenantId) {
    // 获取指定用户或当前用户信息
    User user = null;
    String userName = null;
    String userId = null;
    if (StringUtils.isNotBlank(userid)) {
      user = userService.findById(userid);
      userName = user.getName();
      userId = user.getId();
    } else {
      userName = AppContextHolder.getOnlineUser().getName();
      userId = AppContextHolder.getOnlineUser().getId();
    }
    // 获取资源内后台的接口权限, 每个租户分别放
    Set<String> returnPermission = new HashSet<>();
    Map<Integer, List<Resource>> resourcesTenantList = userService.getResourcesTenantList(null,
        user);
    Set<Map.Entry<Integer, List<Resource>>> entries = resourcesTenantList.entrySet();
    for (Map.Entry<Integer, List<Resource>> entry : entries) {
      Integer tenantId = entry.getKey();
      List<Resource> resources = entry.getValue();
      Set<String> permission = new HashSet<>();
      for (Resource r : resources) {
        if (r.getPermission() != null) {
          for (String p : r.getPermission().split(StmsConstants.SYMBOL_COMMA)) {
            if(StringUtils.isNotEmpty(p)){
              permission.add(p.trim());
            }
          }
        }
      }
      //  返回当前租户 的菜单
      if (Objects.equals(tenantId, inTenantId)) {
        returnPermission = permission;
      }
      permission.add("/oauth/login");
      log.info("[{}]用户加载租户[{}]权限: {}", userName, tenantId, permission.toString());
      String key = ROLE_KEY + userid + ":" + tenantId;
      stringRedisTemplate.opsForValue()
          .set(key, StringUtils.join(permission, ","), ACCESS_TOKEN_EXPIRE, TimeUnit.SECONDS);
    }

//        Set<String> permission = new HashSet<>();
//        for (Resource r : userService.getResourcesList(null, user)) {
//            if (r.getPermission() != null) {
//                for (String p : r.getPermission().split(StmsConstants.SYMBOL_COMMA)) {
//                    permission.add(p);
//                }
//            }
//        }
//        permission.add("/oauth/login");
//        log.info("[{}]用户加载权限: {}", userName, permission.toString());
//        stringRedisTemplate.opsForValue().set(ROLE_KEY + userId, StringUtils.join(permission, ","), ACCESS_TOKEN_EXPIRE, TimeUnit.SECONDS);
    return StringUtils.join(returnPermission, ",");
  }

  @Override
  @Transactional(propagation = Propagation.NOT_SUPPORTED)
  public boolean clearPermission(String... userIds) {
    if (userIds != null && userIds.length > 0) {
      log.info("用户权限清理: {}", Arrays.toString(userIds));
      for (String userId : userIds) {
        stringRedisTemplate.delete(ROLE_KEY + userId + ":*");
        // 清除用户的accesstoken 权限,让用户重新登录
        String accessToken = stringRedisTemplate.opsForValue().get(TOKEN_KEY + userId);
        if (StringUtils.isNotEmpty(accessToken)) {
          stringRedisTemplate.delete(TOKEN_KEY + userId);
          stringRedisTemplate.delete(TOKEN_KEY + accessToken);
        }
        String refreshToken = stringRedisTemplate.opsForValue()
            .get(TOKEN_KEY + ":refreshToken:" + userId);
        if (StringUtils.isNotEmpty(refreshToken)) {
          stringRedisTemplate.delete(TOKEN_KEY + ":refreshToken:" + userId);
          stringRedisTemplate.delete(TOKEN_KEY + refreshToken);
        }
      }
    }
    return true;
  }

  /**
   * 转换成正则表达式
   *
   * @param permission
   * @return
   */
  protected static Set<String> permission2Reg(Set<String> permission) {
    Set<String> regPermission = new HashSet<>();
    for (String p : permission) {
      p = p.replaceAll("\\{", "\\\\{");
      p = p.replaceAll("\\*", "[^/]*");
      p = p.replaceAll("\\[\\^/]\\*\\[\\^/]\\*", ".*");
      // 在save和update资源时，需要检查permission避免正则冲突
      regPermission.add(p);
    }
    return regPermission;
  }
}
