package cn.quantgroup.big.stms.sys.controller;

import cn.quantgroup.big.stms.common.context.EnumClassFactory;
import cn.quantgroup.big.stms.common.controller.BaseController;
import cn.quantgroup.big.stms.common.enums.BaseEnum;
import cn.quantgroup.big.stms.common.enums.BasicDataStatus;
import cn.quantgroup.big.stms.common.exception.BizException;
import cn.quantgroup.big.stms.common.result.Result;
import cn.quantgroup.big.stms.common.result.ResultCode;
import cn.quantgroup.big.stms.sys.dto.DictDto;
import cn.quantgroup.big.stms.sys.model.Dict;
import cn.quantgroup.big.stms.sys.service.DictService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.web.bind.annotation.*;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.validation.Valid;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

@RestController
@RequestMapping("/dict")
public class DictController extends BaseController {
    @Autowired
    private DictService service;

    @RequestMapping(value = "/get/{id}", method = RequestMethod.POST)
    public Result get(@PathVariable Integer id) {
        return Result.success(service.findById(id));
    }

    @RequestMapping(value = "/list/{page}/{size}", method = RequestMethod.POST)
    public Result list(@RequestParam(value = "parentId", required = false) String parentId, @PathVariable("page") int page, @PathVariable("size") int size) {
        PageRequest pageRequest = new PageRequest(page - 1, size, getSort());
        Page<Dict> dicts = service.findAll(new Specification<Dict>() {
            @Override
            public Predicate toPredicate(Root<Dict> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder cb) {
                if (StringUtils.isBlank(parentId)) {
                    return cb.isNull(root.get("parent").get("id"));
                }
                return cb.equal(root.get("parent").get("id"), Integer.parseInt(parentId));
            }
        }, pageRequest);
        //如果没有子节点，返回当前节点
        if (dicts.getTotalElements() == 0 && StringUtils.isNotBlank(parentId)) {
            return Result.success(service.findAll(new Specification<Dict>() {
                @Override
                public Predicate toPredicate(Root<Dict> root, CriteriaQuery<?> criteriaQuery, CriteriaBuilder cb) {
                    return cb.equal(root.get("id"), Integer.parseInt(parentId));
                }
            }, pageRequest));
        } else {
            return Result.success(dicts);
        }
    }

    @RequestMapping(value = "/addnew", method = RequestMethod.POST)
    public Result addNew(@Valid @RequestBody DictDto dictDto) {
        Dict dict = service.findByCodeAndParentId(dictDto.getCode(), dictDto.getParentId());
        if (null != dict) {
            throw new BizException("数据字典已存在", ResultCode.PARAM_ERROR);
        }
        dict = new Dict();
        BeanUtils.copyProperties(dictDto, dict, getNullPropertyNames(dictDto));
        if (dictDto.getParentId() != null) {
            Dict parent = service.findById(dictDto.getParentId());
            if (parent == null || parent.getStatus() != BasicDataStatus.VALID) {
                throw new BizException("数据有误", ResultCode.PARAM_ERROR);
            }
            dict.setParent(parent);
            dict.setType(parent.getType());
        } else {
            if (dict.getType() == null) {
                throw new BizException("上级字典非有效状态，不能添加子节点", ResultCode.PARAM_ERROR);
            }
        }
        service.save(dict);
        return Result.success();
    }

    @RequestMapping(value = "/delete/{id}", method = RequestMethod.POST)
    public Result delete(@PathVariable Integer id) {
        Dict dict = service.findById(id);
        if (null == dict) {
            throw new BizException("数据字典不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.SAVE != dict.getStatus()) {
            throw new BizException("新增状态才能删除", ResultCode.PARAM_ERROR);
        }
        service.delete(dict);
        return Result.success();
    }

    @RequestMapping(value = "/update", method = RequestMethod.POST)
    public Result update(@Valid @RequestBody DictDto dictDto) {
        Dict dict = service.findById(dictDto.getId());
        if (null == dict) {
            throw new BizException("数据字典不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.SAVE != dict.getStatus()) {
            throw new BizException("新增状态才能修改", ResultCode.PARAM_ERROR);
        }
        if (null != dict.getParent() && dictDto.getParentId() != null && !dict.getParent().getId().equals(dictDto.getParentId())) {
            throw new BizException("上级数据字典不能修改", ResultCode.PARAM_ERROR);
        }
        // 校验组织编码是否修改
        if (!dict.getCode().equals(dictDto.getCode())) {
            Dict temp = service.findByCode(dictDto.getCode());
            if (null != temp) {
                throw new BizException("数据字典编码已存在", ResultCode.PARAM_ERROR);
            }
        }

        BeanUtils.copyProperties(dictDto, dict, getNullPropertyNames(dictDto));

        service.save(dict);
        return Result.success();
    }

    @RequestMapping(value = "/submit/{id}", method = RequestMethod.POST)
    public Result submit(@PathVariable Integer id) {
        Dict dict = service.findById(id);
        if (null == dict) {
            throw new BizException("数据字典不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.SAVE != dict.getStatus()) {
            throw new BizException("新增状态才能提交", ResultCode.PARAM_ERROR);
        }
        service.submit(dict);
        return Result.success();
    }

    @RequestMapping(value = "/enabled/{id}", method = RequestMethod.POST)
    public Result enabled(@PathVariable Integer id) {
        Dict dict = service.findById(id);
        if (null == dict) {
            throw new BizException("数据字典不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.INVALID != dict.getStatus()) {
            throw new BizException("无效状态才能启用", ResultCode.PARAM_ERROR);
        }
        service.enabled(dict);
        return Result.success();
    }

    @RequestMapping(value = "/disabled/{id}", method = RequestMethod.POST)
    public Result disabled(@PathVariable Integer id) {
        Dict dict = service.findById(id);
        if (null == dict) {
            throw new BizException("数据字典不存在", ResultCode.PARAM_ERROR);
        }
        if (BasicDataStatus.VALID != dict.getStatus()) {
            throw new BizException("有效状态才能禁用", ResultCode.PARAM_ERROR);
        }
        service.disabled(dict);
        return Result.success();
    }

    @RequestMapping(value = {"/getenums/{clazz:.+}", "/getenums/{clazz:.+}/{value}"}, method = RequestMethod.GET)
    public Result getEnums(@PathVariable String clazz, @PathVariable(required = false) String value) {
        Class<? extends BaseEnum> c = EnumClassFactory.getEnumClass(clazz);

        BaseEnum[] enums = c.getEnumConstants();
        if (StringUtils.isNotEmpty(value)) {
            List<BaseEnum> list = new ArrayList<>();
            List<String> values = Arrays.asList(value.split(","));

            for (BaseEnum anEnum : enums) {
                if (!values.contains(String.valueOf(anEnum.getValue()))) {
                    list.add(anEnum);
                }
            }
            return Result.success(list);
        }

        return Result.success(enums);
    }

    // 2020-1-8 XinM 可能没用了，暂不删
    // 数据字典左侧的菜单，根据id显示下级
    @RequestMapping(value = "/getchild", method = RequestMethod.POST)
    public Result getChild(@RequestParam(value = "parentId", required = false) String parentId) {
        return Result.success(service.getChildren(parentId));
    }

    @RequestMapping(value = "/tree", method = RequestMethod.POST)
    public Result getDictTree() {
        List<Dict> dictList = (List<Dict>) service.findAll(getSort());
        return Result.success(service.getTree(dictList));
    }
}
